/*
 *	Intel IO-APIC support for multi-Pentium hosts.
 *
 *	Copyright (C) 1997, 1998, 1999, 2000 Ingo Molnar, Hajnalka Szabo
 *
 *	Many thanks to Stig Venaas for trying out countless experimental
 *	patches and reporting/debugging problems patiently!
 *
 *	(c) 1999, Multiple IO-APIC support, developed by
 *	Ken-ichi Yaku <yaku@css1.kbnes.nec.co.jp> and
 *      Hidemi Kishimoto <kisimoto@css1.kbnes.nec.co.jp>,
 *	further tested and cleaned up by Zach Brown <zab@redhat.com>
 *	and Ingo Molnar <mingo@redhat.com>
 *
 *	Fixes
 *	Maciej W. Rozycki	:	Bits for genuine 82489DX APICs;
 *					thanks to Eric Gilmore
 *					and Rolf G. Tews
 *					for testing these extensively
 *	Paul Diefenbaugh	:	Added full ACPI support
 */

#include <xen/lib.h>
#include <xen/init.h>
#include <xen/irq.h>
#include <xen/delay.h>
#include <xen/param.h>
#include <xen/sched.h>
#include <xen/acpi.h>
#include <xen/keyhandler.h>
#include <xen/softirq.h>
#include <xen/xvmalloc.h>

#include <asm/apic.h>
#include <asm/genapic.h>
#include <asm/hpet.h>
#include <asm/io-ports.h>
#include <asm/io_apic.h>
#include <asm/irq-vectors.h>
#include <asm/mc146818rtc.h>
#include <asm/smp.h>
#include <asm/desc.h>
#include <asm/msi.h>
#include <asm/setup.h>

#include <public/physdev.h>
#include <xen/trace.h>

/* Where if anywhere is the i8259 connect in external int mode */
static struct { int pin, apic; } ioapic_i8259 = { -1, -1 };

static DEFINE_SPINLOCK(ioapic_lock);

bool __read_mostly skip_ioapic_setup;
bool __initdata ioapic_ack_new = true;
bool __initdata ioapic_ack_forced;

/*
 * # of IRQ routing registers
 */
int __read_mostly nr_ioapic_entries[MAX_IO_APICS];
int __read_mostly nr_ioapics;

/*
 * Rough estimation of how many shared IRQs there are, can
 * be changed anytime.
 */
#define MAX_PLUS_SHARED_IRQS nr_irqs_gsi
#define PIN_MAP_SIZE (MAX_PLUS_SHARED_IRQS + nr_irqs_gsi)


#define ioapic_has_eoi_reg(apic) (mp_ioapics[(apic)].mpc_apicver >= 0x20)

static int apic_pin_2_gsi_irq(int apic, int pin);

static vmask_t *__read_mostly vector_map[MAX_IO_APICS];

/*
 * Store the EOI handle when using interrupt remapping.
 *
 * If using AMD-Vi interrupt remapping the IO-APIC redirection entry remapped
 * format repurposes the vector field to store the offset into the Interrupt
 * Remap table.  This breaks directed EOI, as the CPU vector no longer matches
 * the contents of the RTE vector field.  Add a translation cache so that
 * directed EOI uses the value in the RTE vector field when interrupt remapping
 * is enabled.
 *
 * Intel VT-d Xen code still stores the CPU vector in the RTE vector field when
 * using the remapped format, but use the translation cache uniformly in order
 * to avoid extra logic to differentiate between VT-d and AMD-Vi.
 *
 * The matrix is accessed as [#io-apic][#pin].
 */
static uint8_t **__ro_after_init io_apic_pin_eoi;

static void share_vector_maps(unsigned int src, unsigned int dst)
{
    unsigned int pin;

    if (vector_map[src] == vector_map[dst])
        return;

    bitmap_or(vector_map[src]->_bits, vector_map[src]->_bits,
              vector_map[dst]->_bits, X86_IDT_VECTORS);

    for (pin = 0; pin < nr_ioapic_entries[dst]; ++pin) {
        int irq = apic_pin_2_gsi_irq(dst, pin);
        struct irq_desc *desc;

        if (irq < 0)
            continue;
        desc = irq_to_desc(irq);
        if (desc->arch.used_vectors == vector_map[dst])
            desc->arch.used_vectors = vector_map[src];
    }

    vector_map[dst] = vector_map[src];
}

/*
 * This is performance-critical, we want to do it O(1)
 *
 * the indexing order of this array favors 1:1 mappings
 * between pins and IRQs.
 */

static struct irq_pin_list {
    int apic, pin;
    unsigned int next;
} *__read_mostly irq_2_pin;

static unsigned int irq_2_pin_free_entry;

/*
 * The common case is 1:1 IRQ<->pin mappings. Sometimes there are
 * shared ISA-space IRQs, so we have to support them. We are super
 * fast in the common case, and fast for shared ISA-space IRQs.
 */
static void add_pin_to_irq(unsigned int irq, int apic, int pin)
{
    struct irq_pin_list *entry = irq_2_pin + irq;

    while (entry->next) {
        BUG_ON((entry->apic == apic) && (entry->pin == pin));
        entry = irq_2_pin + entry->next;
    }
    
    BUG_ON((entry->apic == apic) && (entry->pin == pin));

    if (entry->pin != -1) {
        if (irq_2_pin_free_entry >= PIN_MAP_SIZE)
            panic("io_apic.c: whoops\n");
        entry->next = irq_2_pin_free_entry;
        entry = irq_2_pin + entry->next;
        irq_2_pin_free_entry = entry->next;
        entry->next = 0;
    }
    entry->apic = apic;
    entry->pin = pin;
    share_vector_maps(irq_2_pin[irq].apic, apic);
}

static void remove_pin_from_irq(unsigned int irq, int apic, int pin)
{
    struct irq_pin_list *entry, *prev;

    for (entry = &irq_2_pin[irq]; ; entry = &irq_2_pin[entry->next]) {
        if ((entry->apic == apic) && (entry->pin == pin))
            break;
        BUG_ON(!entry->next);
    }

    entry->pin = entry->apic = -1;

    if (entry != &irq_2_pin[irq]) {
        /* Removed entry is not at head of list. */
        prev = &irq_2_pin[irq];
        while (&irq_2_pin[prev->next] != entry)
            prev = &irq_2_pin[prev->next];
        prev->next = entry->next;
    } else if (entry->next) {
        /* Removed entry is at head of multi-item list. */
        prev  = entry;
        entry = &irq_2_pin[entry->next];
        *prev = *entry;
        entry->pin = entry->apic = -1;
    } else
        return;

    entry->next = irq_2_pin_free_entry;
    irq_2_pin_free_entry = entry - irq_2_pin;
}

static struct irq_pin_list *pin_list_next(const struct irq_pin_list *entry)
{
    if ( !entry->next )
        return NULL;

    return irq_2_pin + entry->next;
}

/*
 * Reroute an IRQ to a different pin.
 */
static void __init replace_pin_at_irq(unsigned int irq,
                      int oldapic, int oldpin,
                      int newapic, int newpin)
{
    struct irq_pin_list *entry = irq_2_pin + irq;

    for (; entry; entry = pin_list_next(entry)) {
        if (entry->apic == oldapic && entry->pin == oldpin) {
            entry->apic = newapic;
            entry->pin = newpin;
            share_vector_maps(oldapic, newapic);
        }
    }
}

vmask_t *io_apic_get_used_vector_map(unsigned int irq)
{
    struct irq_pin_list *entry = irq_2_pin + irq;

    if (entry->pin == -1)
        return NULL;

    return vector_map[entry->apic];
}

struct IO_APIC_route_entry **alloc_ioapic_entries(void)
{
    int apic;
    struct IO_APIC_route_entry **ioapic_entries;

    ioapic_entries = xmalloc_array(struct IO_APIC_route_entry *, nr_ioapics);
    if (!ioapic_entries)
        return NULL;

    for (apic = 0; apic < nr_ioapics; apic++) {
        ioapic_entries[apic] =
            xmalloc_array(struct IO_APIC_route_entry,
                          nr_ioapic_entries[apic]);
        if (!ioapic_entries[apic] && nr_ioapic_entries[apic])
            goto nomem;
    }

    return ioapic_entries;

nomem:
    while (--apic >= 0)
        xfree(ioapic_entries[apic]);
    xfree(ioapic_entries);

    return NULL;
}

struct IO_APIC_route_entry __ioapic_read_entry(
    unsigned int apic, unsigned int pin, bool raw)
{
    struct IO_APIC_route_entry entry;

    if ( raw || !iommu_intremap )
    {
        entry.low  = __io_apic_read(apic, 0x10 + 2 * pin);
        entry.high = __io_apic_read(apic, 0x11 + 2 * pin);
    }
    else
    {
        entry.low  = iommu_read_apic_from_ire(apic, 0x10 + 2 * pin);
        entry.high = iommu_read_apic_from_ire(apic, 0x11 + 2 * pin);
    }

    return entry;
}

static struct IO_APIC_route_entry ioapic_read_entry(
    unsigned int apic, unsigned int pin, bool raw)
{
    struct IO_APIC_route_entry entry;
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    entry = __ioapic_read_entry(apic, pin, raw);
    spin_unlock_irqrestore(&ioapic_lock, flags);
    return entry;
}

void __ioapic_write_entry(
    unsigned int apic, unsigned int pin, bool raw,
    struct IO_APIC_route_entry e)
{
    if ( raw || !iommu_intremap )
    {
        __io_apic_write(apic, 0x11 + 2 * pin, e.high);
        __io_apic_write(apic, 0x10 + 2 * pin, e.low);
        /*
         * Might be called before io_apic_pin_eoi is allocated.  Entry will be
         * initialized to the RTE value once the cache is allocated.
         *
         * The vector field is only cached for raw RTE writes when using IR.
         * In that case the vector field might have been repurposed to store
         * something different than the CPU vector, and hence need to be cached
         * for performing EOI.
         */
        if ( io_apic_pin_eoi )
            io_apic_pin_eoi[apic][pin] = e.vector;
    }
    else
        iommu_update_ire_from_apic(apic, pin, e.raw);
}

static void ioapic_write_entry(
    unsigned int apic, unsigned int pin, bool raw,
    struct IO_APIC_route_entry e)
{
    unsigned long flags;
    spin_lock_irqsave(&ioapic_lock, flags);
    __ioapic_write_entry(apic, pin, raw, e);
    spin_unlock_irqrestore(&ioapic_lock, flags);
}

/*
 * EOI an IO-APIC entry.  Vector may be -1, indicating that it should be
 * worked out using the pin.  This function expects that the ioapic_lock is
 * being held, and interrupts are disabled (or there is a good reason not
 * to), and that if both pin and vector are passed, that they refer to the
 * same redirection entry in the IO-APIC.
 *
 * If using Interrupt Remapping the vector is always ignored because the RTE
 * remapping format might have repurposed the vector field and a cached value
 * of the EOI handle to use is obtained based on the provided apic and pin
 * values.
 */
static void __io_apic_eoi(unsigned int apic, unsigned int vector, unsigned int pin)
{
    /* Prefer the use of the EOI register if available */
    if ( ioapic_has_eoi_reg(apic) )
    {
        if ( io_apic_pin_eoi )
            /*
             * If the EOI handle is cached use it. When using AMD-Vi IR the CPU
             * vector no longer matches the vector field in the RTE, because
             * the RTE remapping format repurposes the field.
             *
             * The value in the RTE vector field must always be used to signal
             * which RTE to EOI, hence use the cached value which always
             * mirrors the contents of the raw RTE vector field.
             */
            vector = io_apic_pin_eoi[apic][pin];
        else if ( vector == IRQ_VECTOR_UNASSIGNED )
             /* If vector is unknown, read it from the IO-APIC */
            vector = __ioapic_read_entry(apic, pin, true).vector;

        *(IO_APIC_BASE(apic)+16) = vector;
    }
    else
    {
        /* Else fake an EOI by switching to edge triggered mode
         * and back */
        struct IO_APIC_route_entry entry;
        bool need_to_unmask = false;

        entry = __ioapic_read_entry(apic, pin, true);

        if ( ! entry.mask )
        {
            /* If entry is not currently masked, mask it and make
             * a note to unmask it later */
            entry.mask = 1;
            __ioapic_write_entry(apic, pin, true, entry);
            need_to_unmask = true;
        }

        /* Flip the trigger mode to edge and back */
        entry.trigger = 0;
        __ioapic_write_entry(apic, pin, true, entry);
        entry.trigger = 1;
        __ioapic_write_entry(apic, pin, true, entry);

        if ( need_to_unmask )
        {
            /* Unmask if neccesary */
            entry.mask = 0;
            __ioapic_write_entry(apic, pin, true, entry);
        }
    }
}

/*
 * Saves all the IO-APIC RTE's
 */
int save_IO_APIC_setup(struct IO_APIC_route_entry **ioapic_entries)
{
    int apic, pin;

    if (!ioapic_entries)
        return -ENOMEM;

    for (apic = 0; apic < nr_ioapics; apic++) {
        if (!nr_ioapic_entries[apic])
            continue;

        if (!ioapic_entries[apic])
            return -ENOMEM;

        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++)
	    ioapic_entries[apic][pin] = __ioapic_read_entry(apic, pin, true);
    }

    return 0;
}

/*
 * Mask all IO APIC entries.
 */
void mask_IO_APIC_setup(struct IO_APIC_route_entry **ioapic_entries)
{
    int apic, pin;

    if (!ioapic_entries)
        return;

    for (apic = 0; apic < nr_ioapics; apic++) {
        if (!nr_ioapic_entries[apic])
            continue;

        if (!ioapic_entries[apic])
            break;

        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++) {
            struct IO_APIC_route_entry entry;

            entry = ioapic_entries[apic][pin];
            if (!entry.mask) {
                entry.mask = 1;

                ioapic_write_entry(apic, pin, true, entry);
            }
        }
    }
}

/*
 * Restore IO APIC entries which was saved in ioapic_entries.
 */
int restore_IO_APIC_setup(struct IO_APIC_route_entry **ioapic_entries,
                          bool raw)
{
    int apic, pin;

    if (!ioapic_entries)
        return -ENOMEM;

    for (apic = 0; apic < nr_ioapics; apic++) {
        if (!nr_ioapic_entries[apic])
            continue;

        if (!ioapic_entries[apic])
            return -ENOMEM;

        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++)
	    ioapic_write_entry(apic, pin, raw, ioapic_entries[apic][pin]);
    }

    return 0;
}

void free_ioapic_entries(struct IO_APIC_route_entry **ioapic_entries)
{
    int apic;

    for (apic = 0; apic < nr_ioapics; apic++)
        xfree(ioapic_entries[apic]);

    xfree(ioapic_entries);
}

static void modify_IO_APIC_irq(unsigned int irq, unsigned int enable,
                               unsigned int disable)
{
    struct irq_pin_list *entry = irq_2_pin + irq;

    for (; entry; entry = pin_list_next(entry)) {
        unsigned int pin = entry->pin;
        struct IO_APIC_route_entry rte;

        if (pin == -1)
            break;
        rte = __ioapic_read_entry(entry->apic, pin, false);
        rte.raw &= ~(uint64_t)disable;
        rte.raw |= enable;
        __ioapic_write_entry(entry->apic, pin, false, rte);
    }
}

/* mask = 1 */
static void __mask_IO_APIC_irq (unsigned int irq)
{
    modify_IO_APIC_irq(irq, IO_APIC_REDIR_MASKED, 0);
}

/* mask = 0 */
static void __unmask_IO_APIC_irq (unsigned int irq)
{
    modify_IO_APIC_irq(irq, 0, IO_APIC_REDIR_MASKED);
}

/* trigger = 0 */
static void __edge_IO_APIC_irq (unsigned int irq)
{
    modify_IO_APIC_irq(irq, 0, IO_APIC_REDIR_LEVEL_TRIGGER);
}

/* trigger = 1 */
static void __level_IO_APIC_irq (unsigned int irq)
{
    modify_IO_APIC_irq(irq, IO_APIC_REDIR_LEVEL_TRIGGER, 0);
}

static void cf_check mask_IO_APIC_irq(struct irq_desc *desc)
{
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    __mask_IO_APIC_irq(desc->irq);
    spin_unlock_irqrestore(&ioapic_lock, flags);
}

static void cf_check unmask_IO_APIC_irq(struct irq_desc *desc)
{
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    __unmask_IO_APIC_irq(desc->irq);
    spin_unlock_irqrestore(&ioapic_lock, flags);
}

static void __eoi_IO_APIC_irq(struct irq_desc *desc)
{
    struct irq_pin_list *entry = irq_2_pin + desc->irq;
    unsigned int pin, vector = desc->arch.vector;

    for (; entry; entry = pin_list_next(entry)) {
        pin = entry->pin;
        if (pin == -1)
            break;
        __io_apic_eoi(entry->apic, vector, pin);
    }
}

static void eoi_IO_APIC_irq(struct irq_desc *desc)
{
    unsigned long flags;
    spin_lock_irqsave(&ioapic_lock, flags);
    __eoi_IO_APIC_irq(desc);
    spin_unlock_irqrestore(&ioapic_lock, flags);
}

static void clear_IO_APIC_pin(unsigned int apic, unsigned int pin)
{
    struct IO_APIC_route_entry entry;

    /* Check delivery_mode to be sure we're not clearing an SMI pin */
    entry = __ioapic_read_entry(apic, pin, false);
    if (entry.delivery_mode == dest_SMI)
        return;

    /*
     * Make sure the entry is masked and re-read the contents to check
     * if it is a level triggered pin and if the remoteIRR is set.
     */
    if (!entry.mask) {
        entry.mask = 1;
        __ioapic_write_entry(apic, pin, false, entry);
    }
    entry = __ioapic_read_entry(apic, pin, true);

    if (entry.irr) {
        /* Make sure the trigger mode is set to level. */
        if (!entry.trigger) {
            entry = __ioapic_read_entry(apic, pin, false);
            entry.trigger = 1;
            __ioapic_write_entry(apic, pin, false, entry);
        }
        __io_apic_eoi(apic, entry.vector, pin);
    }

    /*
     * Disable it in the IO-APIC irq-routing table:
     */
    memset(&entry, 0, sizeof(entry));
    entry.mask = 1;
    __ioapic_write_entry(apic, pin, false, entry);

    entry = __ioapic_read_entry(apic, pin, true);
    if (entry.irr)
        printk(KERN_ERR "IO-APIC%02x-%u: Unable to reset IRR\n",
               IO_APIC_ID(apic), pin);
}

static void clear_IO_APIC (void)
{
    int apic, pin;

    for (apic = 0; apic < nr_ioapics; apic++) {
        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++)
            clear_IO_APIC_pin(apic, pin);
    }
}

static void cf_check
set_ioapic_affinity_irq(struct irq_desc *desc, const cpumask_t *mask)
{
    unsigned int dest;
    int pin, irq;
    struct irq_pin_list *entry;

    irq = desc->irq;

    spin_lock(&ioapic_lock);

    dest = set_desc_affinity(desc, mask);
    if (dest != BAD_APICID) {
        if ( !iommu_intremap || !x2apic_enabled )
            dest = SET_APIC_LOGICAL_ID(dest);
        entry = irq_2_pin + irq;
        for (; entry; entry = pin_list_next(entry)) {
            struct IO_APIC_route_entry rte;

            pin = entry->pin;
            if (pin == -1)
                break;

            rte = __ioapic_read_entry(entry->apic, pin, false);
            rte.dest.dest32 = dest;
            rte.vector = desc->arch.vector;
            __ioapic_write_entry(entry->apic, pin, false, rte);
        }
    }

    spin_unlock(&ioapic_lock);
}

/*
 * Find the IRQ entry number of a certain pin.
 */
static int find_irq_entry(int apic, int pin, int type)
{
    int i;

    for (i = 0; i < mp_irq_entries; i++)
        if (mp_irqs[i].mpc_irqtype == type &&
            (mp_irqs[i].mpc_dstapic == mp_ioapics[apic].mpc_apicid ||
             mp_irqs[i].mpc_dstapic == MP_APIC_ALL) &&
            mp_irqs[i].mpc_dstirq == pin)
            return i;

    return -1;
}

/*
 * Find the pin to which IRQ[irq] (ISA) is connected
 */
static int __init find_isa_irq_pin(int irq, int type)
{
    int i;

    for (i = 0; i < mp_irq_entries; i++) {
        int lbus = mp_irqs[i].mpc_srcbus;

        if ((mp_bus_id_to_type[lbus] == MP_BUS_ISA ||
             mp_bus_id_to_type[lbus] == MP_BUS_EISA ||
             mp_bus_id_to_type[lbus] == MP_BUS_MCA ||
             mp_bus_id_to_type[lbus] == MP_BUS_NEC98
            ) &&
            (mp_irqs[i].mpc_irqtype == type) &&
            (mp_irqs[i].mpc_srcbusirq == irq))

            return mp_irqs[i].mpc_dstirq;
    }
    return -1;
}

static int __init find_isa_irq_apic(int irq, int type)
{
    int i;

    for (i = 0; i < mp_irq_entries; i++) {
        int lbus = mp_irqs[i].mpc_srcbus;

        if ((mp_bus_id_to_type[lbus] == MP_BUS_ISA ||
             mp_bus_id_to_type[lbus] == MP_BUS_EISA ||
             mp_bus_id_to_type[lbus] == MP_BUS_MCA ||
             mp_bus_id_to_type[lbus] == MP_BUS_NEC98
            ) &&
            (mp_irqs[i].mpc_irqtype == type) &&
            (mp_irqs[i].mpc_srcbusirq == irq))
            break;
    }
    if (i < mp_irq_entries) {
        int apic;
        for(apic = 0; apic < nr_ioapics; apic++) {
            if (!nr_ioapic_entries[apic])
                continue;
            if (mp_ioapics[apic].mpc_apicid == mp_irqs[i].mpc_dstapic)
                return apic;
        }
    }

    return -1;
}

/*
 * Find a specific PCI IRQ entry.
 * Not an __init, possibly needed by modules
 */
static int pin_2_irq(int idx, int apic, int pin);

/*
 * This function currently is only a helper for the i386 smp boot process where 
 * we need to reprogram the ioredtbls to cater for the cpus which have come online
 * so mask in all cases should simply be TARGET_CPUS
 */
void /*__init*/ setup_ioapic_dest(void)
{
    int pin, ioapic, irq, irq_entry;

    if (skip_ioapic_setup)
        return;

    for (ioapic = 0; ioapic < nr_ioapics; ioapic++) {
        for (pin = 0; pin < nr_ioapic_entries[ioapic]; pin++) {
            struct irq_desc *desc;
            unsigned long flags;

            irq_entry = find_irq_entry(ioapic, pin, mp_INT);
            if (irq_entry == -1)
                continue;
            irq = pin_2_irq(irq_entry, ioapic, pin);
            desc = irq_to_desc(irq);

            spin_lock_irqsave(&desc->lock, flags);
            BUG_ON(!cpumask_intersects(desc->arch.cpu_mask, &cpu_online_map));
            set_ioapic_affinity_irq(desc, desc->arch.cpu_mask);
            spin_unlock_irqrestore(&desc->lock, flags);
        }
    }
}

/*
 * EISA Edge/Level control register, ELCR
 */
static int EISA_ELCR(unsigned int irq)
{
    if (platform_legacy_irq(irq)) {
        unsigned int port = 0x4d0 + (irq >> 3);
        return (inb(port) >> (irq & 7)) & 1;
    }
    apic_printk(APIC_VERBOSE, KERN_INFO
                "Broken MPtable reports ISA irq %d\n", irq);
    return 0;
}

/* EISA interrupts are always polarity zero and can be edge or level
 * trigger depending on the ELCR value.  If an interrupt is listed as
 * EISA conforming in the MP table, that means its trigger type must
 * be read in from the ELCR */

#define default_EISA_trigger(idx)    (EISA_ELCR(mp_irqs[idx].mpc_srcbusirq))
#define default_EISA_polarity(idx)	(0)

/* ISA interrupts are always polarity zero edge triggered,
 * when listed as conforming in the MP table. */

#define default_ISA_trigger(idx)	(0)
#define default_ISA_polarity(idx)	(0)

/* PCI interrupts are always polarity one level triggered,
 * when listed as conforming in the MP table. */

#define default_PCI_trigger(idx)	(1)
#define default_PCI_polarity(idx)	(1)

/* MCA interrupts are always polarity zero level triggered,
 * when listed as conforming in the MP table. */

#define default_MCA_trigger(idx)	(1)
#define default_MCA_polarity(idx)	(0)

/* NEC98 interrupts are always polarity zero edge triggered,
 * when listed as conforming in the MP table. */

#define default_NEC98_trigger(idx)     (0)
#define default_NEC98_polarity(idx)    (0)

static int __init MPBIOS_polarity(int idx)
{
    int bus = mp_irqs[idx].mpc_srcbus;
    int polarity;

    /*
     * Determine IRQ line polarity (high active or low active):
     */
    switch (mp_irqs[idx].mpc_irqflag & 3)
    {
    case 0: /* conforms, ie. bus-type dependent polarity */
    {
        switch (mp_bus_id_to_type[bus])
        {
        case MP_BUS_ISA: /* ISA pin */
        {
            polarity = default_ISA_polarity(idx);
            break;
        }
        case MP_BUS_EISA: /* EISA pin */
        {
            polarity = default_EISA_polarity(idx);
            break;
        }
        case MP_BUS_PCI: /* PCI pin */
        {
            polarity = default_PCI_polarity(idx);
            break;
        }
        case MP_BUS_MCA: /* MCA pin */
        {
            polarity = default_MCA_polarity(idx);
            break;
        }
        case MP_BUS_NEC98: /* NEC 98 pin */
        {
            polarity = default_NEC98_polarity(idx);
            break;
        }
        default:
        {
            printk(KERN_WARNING "broken BIOS!!\n");
            polarity = 1;
            break;
        }
        }
        break;
    }
    case 1: /* high active */
    {
        polarity = 0;
        break;
    }
    case 2: /* reserved */
    {
        printk(KERN_WARNING "broken BIOS!!\n");
        polarity = 1;
        break;
    }
    case 3: /* low active */
    {
        polarity = 1;
        break;
    }
    default: /* invalid */
    {
        printk(KERN_WARNING "broken BIOS!!\n");
        polarity = 1;
        break;
    }
    }
    return polarity;
}

static int MPBIOS_trigger(int idx)
{
    int bus = mp_irqs[idx].mpc_srcbus;
    int trigger;

    /*
     * Determine IRQ trigger mode (edge or level sensitive):
     */
    switch ((mp_irqs[idx].mpc_irqflag>>2) & 3)
    {
    case 0: /* conforms, ie. bus-type dependent */
    {
        switch (mp_bus_id_to_type[bus])
        {
        case MP_BUS_ISA: /* ISA pin */
        {
            trigger = default_ISA_trigger(idx);
            break;
        }
        case MP_BUS_EISA: /* EISA pin */
        {
            trigger = default_EISA_trigger(idx);
            break;
        }
        case MP_BUS_PCI: /* PCI pin */
        {
            trigger = default_PCI_trigger(idx);
            break;
        }
        case MP_BUS_MCA: /* MCA pin */
        {
            trigger = default_MCA_trigger(idx);
            break;
        }
        case MP_BUS_NEC98: /* NEC 98 pin */
        {
            trigger = default_NEC98_trigger(idx);
            break;
        }
        default:
        {
            printk(KERN_WARNING "broken BIOS!!\n");
            trigger = 1;
            break;
        }
        }
        break;
    }
    case 1: /* edge */
    {
        trigger = 0;
        break;
    }
    case 2: /* reserved */
    {
        printk(KERN_WARNING "broken BIOS!!\n");
        trigger = 1;
        break;
    }
    case 3: /* level */
    {
        trigger = 1;
        break;
    }
    default: /* invalid */
    {
        printk(KERN_WARNING "broken BIOS!!\n");
        trigger = 0;
        break;
    }
    }
    return trigger;
}

static inline int irq_polarity(int idx)
{
    return MPBIOS_polarity(idx);
}

static inline int irq_trigger(int idx)
{
    return MPBIOS_trigger(idx);
}

static int pin_2_irq(int idx, int apic, int pin)
{
    int irq, i;
    int bus = mp_irqs[idx].mpc_srcbus;

    /*
     * Debugging check, we are in big trouble if this message pops up!
     */
    if (mp_irqs[idx].mpc_dstirq != pin)
        printk(KERN_ERR "broken BIOS or MPTABLE parser, ayiee!!\n");

    switch (mp_bus_id_to_type[bus])
    {
    case MP_BUS_ISA: /* ISA pin */
    case MP_BUS_EISA:
    case MP_BUS_MCA:
    case MP_BUS_NEC98:
    {
        irq = mp_irqs[idx].mpc_srcbusirq;
        break;
    }
    case MP_BUS_PCI: /* PCI pin */
    {
        /*
         * PCI IRQs are mapped in order
         */
        i = irq = 0;
        while (i < apic)
            irq += nr_ioapic_entries[i++];
        irq += pin;
        break;
    }
    default:
    {
        printk(KERN_ERR "unknown bus type %d.\n",bus);
        irq = 0;
        break;
    }
    }

    return irq;
}

int gsi_2_irq(unsigned int gsi)
{
    int ioapic, irq;
    unsigned int pin;

    if ( gsi > highest_gsi() )
        return -ERANGE;

    ioapic = mp_find_ioapic(gsi);
    if ( ioapic < 0 )
        return -EINVAL;

    pin = gsi - io_apic_gsi_base(ioapic);

    irq = apic_pin_2_gsi_irq(ioapic, pin);

    return irq ?: -EINVAL;
}

static inline int IO_APIC_irq_trigger(int irq)
{
    int apic, idx, pin;

    for (apic = 0; apic < nr_ioapics; apic++) {
        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++) {
            idx = find_irq_entry(apic,pin,mp_INT);
            if ((idx != -1) && (irq == pin_2_irq(idx,apic,pin)))
                return irq_trigger(idx);
        }
    }
    /*
     * nonexistent IRQs are edge default
     */
    return 0;
}

static struct hw_interrupt_type ioapic_level_type;
static hw_irq_controller ioapic_edge_type;

#define IOAPIC_AUTO	-1
#define IOAPIC_EDGE	0
#define IOAPIC_LEVEL	1

#define SET_DEST(ent, mode, val) do { \
    if (x2apic_enabled && iommu_intremap) \
        (ent).dest.dest32 = (val); \
    else \
        (ent).dest.mode.mode##_dest = (val); \
} while (0)

static inline void ioapic_register_intr(int irq, unsigned long trigger)
{
    if ((trigger == IOAPIC_AUTO && IO_APIC_irq_trigger(irq)) ||
        trigger == IOAPIC_LEVEL)
        irq_desc[irq].handler = &ioapic_level_type;
    else
        irq_desc[irq].handler = &ioapic_edge_type;
}

static void __init setup_IO_APIC_irqs(void)
{
    struct IO_APIC_route_entry entry;
    int apic, pin, idx, irq, first_notcon = 1, vector;
    unsigned long flags;

    apic_printk(APIC_VERBOSE, KERN_DEBUG "init IO_APIC IRQs\n");

    for (apic = 0; apic < nr_ioapics; apic++) {
        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++) {
            /*
             * add it to the IO-APIC irq-routing table:
             */
            memset(&entry,0,sizeof(entry));

            entry.delivery_mode = dest_Fixed;
            entry.dest_mode = 0; /* physical delivery */
            entry.mask = 0;                /* enable IRQ */

            idx = find_irq_entry(apic,pin,mp_INT);
            if (idx == -1) {
                if (first_notcon) {
                    apic_printk(APIC_VERBOSE, KERN_DEBUG
                                " IO-APIC (apicid-pin) %d-%d",
                                mp_ioapics[apic].mpc_apicid,
                                pin);
                    first_notcon = 0;
                } else
                    apic_printk(APIC_VERBOSE, ", %d-%d",
                                mp_ioapics[apic].mpc_apicid, pin);
                continue;
            }

            entry.trigger = irq_trigger(idx);
            entry.polarity = irq_polarity(idx);

            if (irq_trigger(idx)) {
                entry.trigger = 1;
                entry.mask = 1;
            }

            irq = pin_2_irq(idx, apic, pin);

            add_pin_to_irq(irq, apic, pin);

            if (!IO_APIC_IRQ(irq))
                continue;

            vector = assign_irq_vector(irq, NULL);
            BUG_ON(vector < 0);
            entry.vector = vector;
            ioapic_register_intr(irq, IOAPIC_AUTO);

            if (platform_legacy_irq(irq))
                disable_8259A_irq(irq_to_desc(irq));

            SET_DEST(entry, logical, cpu_mask_to_apicid(TARGET_CPUS));
            spin_lock_irqsave(&ioapic_lock, flags);
            __ioapic_write_entry(apic, pin, false, entry);
            spin_unlock_irqrestore(&ioapic_lock, flags);
        }
    }

    if (!first_notcon)
        apic_printk(APIC_VERBOSE, " not connected.\n");
}

/*
 * Set up the 8259A-master output pin:
 */
static void __init setup_ExtINT_IRQ0_pin(unsigned int apic, unsigned int pin, int vector)
{
    struct IO_APIC_route_entry entry;

    memset(&entry,0,sizeof(entry));

    disable_8259A_irq(irq_to_desc(0));

    /* mask LVT0 */
    apic_write(APIC_LVT0, APIC_LVT_MASKED | APIC_DM_EXTINT);

    /*
     * We use logical delivery to get the timer IRQ
     * to the first CPU.
     */
    entry.dest_mode = 0; /* physical delivery */
    entry.mask = 0;					/* unmask IRQ now */
    SET_DEST(entry, logical, cpu_mask_to_apicid(TARGET_CPUS));
    entry.delivery_mode = dest_Fixed;
    entry.polarity = 0;
    entry.trigger = 0;
    entry.vector = vector;

    /*
     * The timer IRQ doesn't have to know that behind the
     * scene we have a 8259A-master in AEOI mode ...
     */
    irq_desc[0].handler = &ioapic_edge_type;

    /*
     * Add it to the IO-APIC irq-routing table:
     */
    ioapic_write_entry(apic, pin, false, entry);

    enable_8259A_irq(irq_to_desc(0));
}

static inline void UNEXPECTED_IO_APIC(void)
{
}

static void /*__init*/ __print_IO_APIC(bool boot)
{
    int apic, i;
    union IO_APIC_reg_00 reg_00;
    union IO_APIC_reg_01 reg_01;
    union IO_APIC_reg_02 reg_02;
    union IO_APIC_reg_03 reg_03;
    unsigned long flags;

    printk(KERN_DEBUG "number of MP IRQ sources: %d.\n", mp_irq_entries);
    for (i = 0; i < nr_ioapics; i++)
        printk(KERN_DEBUG "number of IO-APIC #%d registers: %d.\n",
               mp_ioapics[i].mpc_apicid, nr_ioapic_entries[i]);

    /*
     * We are a bit conservative about what we expect.  We have to
     * know about every hardware change ASAP.
     */
    printk(KERN_INFO "testing the IO APIC.......................\n");

    for (apic = 0; apic < nr_ioapics; apic++) {
        if ( !boot )
            process_pending_softirqs();

        if (!nr_ioapic_entries[apic])
            continue;

	spin_lock_irqsave(&ioapic_lock, flags);
	reg_00.raw = io_apic_read(apic, 0);
	reg_01.raw = io_apic_read(apic, 1);
	if (reg_01.bits.version >= 0x10)
            reg_02.raw = io_apic_read(apic, 2);
	if (reg_01.bits.version >= 0x20)
            reg_03.raw = io_apic_read(apic, 3);
	spin_unlock_irqrestore(&ioapic_lock, flags);

	printk(KERN_DEBUG "IO APIC #%d......\n", mp_ioapics[apic].mpc_apicid);
	printk(KERN_DEBUG ".... register #00: %08X\n", reg_00.raw);
	printk(KERN_DEBUG ".......    : physical APIC id: %02X\n", reg_00.bits.ID);
	printk(KERN_DEBUG ".......    : Delivery Type: %X\n", reg_00.bits.delivery_type);
	printk(KERN_DEBUG ".......    : LTS          : %X\n", reg_00.bits.LTS);
	if (reg_00.bits.__reserved_1 || reg_00.bits.__reserved_2)
            UNEXPECTED_IO_APIC();

	printk(KERN_DEBUG ".... register #01: %08X\n", reg_01.raw);
	printk(KERN_DEBUG ".......     : max redirection entries: %04X\n", reg_01.bits.entries);
	if (	(reg_01.bits.entries != 0x0f) && /* older (Neptune) boards */
		(reg_01.bits.entries != 0x17) && /* typical ISA+PCI boards */
		(reg_01.bits.entries != 0x1b) && /* Compaq Proliant boards */
		(reg_01.bits.entries != 0x1f) && /* dual Xeon boards */
		(reg_01.bits.entries != 0x22) && /* bigger Xeon boards */
		(reg_01.bits.entries != 0x2E) &&
		(reg_01.bits.entries != 0x3F)
            )
            UNEXPECTED_IO_APIC();

	printk(KERN_DEBUG ".......     : PRQ implemented: %X\n", reg_01.bits.PRQ);
	printk(KERN_DEBUG ".......     : IO APIC version: %04X\n", reg_01.bits.version);
	if (	(reg_01.bits.version != 0x01) && /* 82489DX IO-APICs */
		(reg_01.bits.version != 0x10) && /* oldest IO-APICs */
		(reg_01.bits.version != 0x11) && /* Pentium/Pro IO-APICs */
		(reg_01.bits.version != 0x13) && /* Xeon IO-APICs */
		(reg_01.bits.version != 0x20)    /* Intel P64H (82806 AA) */
            )
            UNEXPECTED_IO_APIC();
	if (reg_01.bits.__reserved_1 || reg_01.bits.__reserved_2)
            UNEXPECTED_IO_APIC();

	/*
	 * Some Intel chipsets with IO APIC VERSION of 0x1? don't have reg_02,
	 * but the value of reg_02 is read as the previous read register
	 * value, so ignore it if reg_02 == reg_01.
	 */
	if (reg_01.bits.version >= 0x10 && reg_02.raw != reg_01.raw) {
            printk(KERN_DEBUG ".... register #02: %08X\n", reg_02.raw);
            printk(KERN_DEBUG ".......     : arbitration: %02X\n", reg_02.bits.arbitration);
            if (reg_02.bits.__reserved_1 || reg_02.bits.__reserved_2)
                UNEXPECTED_IO_APIC();
	}

	/*
	 * Some Intel chipsets with IO APIC VERSION of 0x2? don't have reg_02
	 * or reg_03, but the value of reg_0[23] is read as the previous read
	 * register value, so ignore it if reg_03 == reg_0[12].
	 */
	if (reg_01.bits.version >= 0x20 && reg_03.raw != reg_02.raw &&
	    reg_03.raw != reg_01.raw) {
            printk(KERN_DEBUG ".... register #03: %08X\n", reg_03.raw);
            printk(KERN_DEBUG ".......     : Boot DT    : %X\n", reg_03.bits.boot_DT);
            if (reg_03.bits.__reserved_1)
                UNEXPECTED_IO_APIC();
	}

	printk(KERN_DEBUG ".... IRQ redirection table:\n");

	printk(KERN_DEBUG " NR %s Msk Trg IRR Pol Stat DstM DelM Vec\n",
               (x2apic_enabled && iommu_intremap) ? " DestID" : "Dst");

	for (i = 0; i <= reg_01.bits.entries; i++) {
            struct IO_APIC_route_entry entry;

            entry = ioapic_read_entry(apic, i, false);

            if ( x2apic_enabled && iommu_intremap )
                printk(KERN_DEBUG " %02x %08x", i, entry.dest.dest32);
            else
                printk(KERN_DEBUG " %02x  %02x ", i,
                       entry.dest.logical.logical_dest);

            printk(" %d   %d   %d   %d   %d    %d    %d    %02X\n",
                   entry.mask,
                   entry.trigger,
                   entry.irr,
                   entry.polarity,
                   entry.delivery_status,
                   entry.dest_mode,
                   entry.delivery_mode,
                   entry.vector
		);
	}
    }
    printk(KERN_INFO "Using vector-based indexing\n");
    printk(KERN_DEBUG "IRQ to pin mappings:\n");
    for (i = 0; i < nr_irqs_gsi; i++) {
        struct irq_pin_list *entry = irq_2_pin + i;

        if ( !boot && !(i & 0x1f) )
            process_pending_softirqs();

        if (entry->pin < 0)
            continue;
        printk(KERN_DEBUG "IRQ%d ", irq_to_desc(i)->arch.vector);
        for (; entry; entry = pin_list_next(entry))
            printk("-> %d:%d", entry->apic, entry->pin);
        printk("\n");
    }

    printk(KERN_INFO ".................................... done.\n");

    return;
}

static void __init print_IO_APIC(void)
{
    if (apic_verbosity != APIC_QUIET)
        __print_IO_APIC(1);
}

static void cf_check _print_IO_APIC_keyhandler(unsigned char key)
{
    __print_IO_APIC(0);
}

void __init enable_IO_APIC(void)
{
    int i8259_apic, i8259_pin;
    int i, apic;

    /* Initialise dynamic irq_2_pin free list. */
    irq_2_pin = xzalloc_array(struct irq_pin_list, PIN_MAP_SIZE);
        
    for (i = 0; i < PIN_MAP_SIZE; i++)
        irq_2_pin[i].pin = -1;
    for (i = irq_2_pin_free_entry = nr_irqs_gsi; i < PIN_MAP_SIZE; i++)
        irq_2_pin[i].next = i + 1;

    if (directed_eoi_enabled) {
        for (apic = 0; apic < nr_ioapics; apic++) {
            if (!nr_ioapic_entries[apic])
                continue;
            vector_map[apic] = xzalloc(vmask_t);
            BUG_ON(!vector_map[apic]);
        }
    } else {
        vector_map[0] = xzalloc(vmask_t);
        BUG_ON(!vector_map[0]);
        for (apic = 1; apic < nr_ioapics; apic++)
            vector_map[apic] = vector_map[0];
    }

    if ( iommu_intremap != iommu_intremap_off )
    {
        io_apic_pin_eoi = xvmalloc_array(typeof(*io_apic_pin_eoi), nr_ioapics);
        BUG_ON(!io_apic_pin_eoi);
    }

    for(apic = 0; apic < nr_ioapics; apic++) {
        int pin;

        if ( io_apic_pin_eoi )
        {
            io_apic_pin_eoi[apic] = xvmalloc_array(typeof(**io_apic_pin_eoi),
                                                   nr_ioapic_entries[apic]);
            BUG_ON(!io_apic_pin_eoi[apic]);
        }

        /* See if any of the pins is in ExtINT mode and cache EOI handle */
        for (pin = 0; pin < nr_ioapic_entries[apic]; pin++) {
            struct IO_APIC_route_entry entry = ioapic_read_entry(apic, pin, false);

            if ( io_apic_pin_eoi )
                io_apic_pin_eoi[apic][pin] =
                    ioapic_read_entry(apic, pin, true).vector;

            /* If the interrupt line is enabled and in ExtInt mode
             * I have found the pin where the i8259 is connected.
             */
            if ( ioapic_i8259.pin == -1 && entry.mask == 0 &&
                 entry.delivery_mode == dest_ExtINT )
            {
                ioapic_i8259.apic = apic;
                ioapic_i8259.pin  = pin;
            }
        }
    }
    /* Look to see what if the MP table has reported the ExtINT */
    /* If we could not find the appropriate pin by looking at the ioapic
     * the i8259 probably is not connected the ioapic but give the
     * mptable a chance anyway.
     */
    i8259_pin  = find_isa_irq_pin(0, mp_ExtINT);
    i8259_apic = find_isa_irq_apic(0, mp_ExtINT);
    /* Trust the MP table if nothing is setup in the hardware */
    if ((ioapic_i8259.pin == -1) && (i8259_pin >= 0)) {
        printk(KERN_WARNING "ExtINT not setup in hardware but reported by MP table\n");
        ioapic_i8259.pin  = i8259_pin;
        ioapic_i8259.apic = i8259_apic;
    }
    /* Complain if the MP table and the hardware disagree */
    if (((ioapic_i8259.apic != i8259_apic) || (ioapic_i8259.pin != i8259_pin)) &&
        (i8259_pin >= 0) && (ioapic_i8259.pin >= 0))
    {
        printk(KERN_WARNING "ExtINT in hardware and MP table differ\n");
    }

    /*
     * Do not trust the IO-APIC being empty at bootup
     */
    clear_IO_APIC();
}

/*
 * Not an __init, needed by the reboot code
 */
void disable_IO_APIC(void)
{
    /*
     * Clear the IO-APIC before rebooting:
     */
    clear_IO_APIC();

    /*
     * If the i8259 is routed through an IOAPIC
     * Put that IOAPIC in virtual wire mode
     * so legacy interrupts can be delivered.
     */
    if (ioapic_i8259.pin != -1) {
        struct IO_APIC_route_entry entry;

        memset(&entry, 0, sizeof(entry));
        entry.mask            = 0; /* Enabled */
        entry.trigger         = 0; /* Edge */
        entry.irr             = 0;
        entry.polarity        = 0; /* High */
        entry.delivery_status = 0;
        entry.dest_mode       = 0; /* Physical */
        entry.delivery_mode   = dest_ExtINT; /* ExtInt */
        entry.vector          = 0;
        SET_DEST(entry, physical, get_apic_id());

        /*
         * Add it to the IO-APIC irq-routing table:
         */
        ioapic_write_entry(ioapic_i8259.apic, ioapic_i8259.pin, false, entry);
    }
    disconnect_bsp_APIC(ioapic_i8259.pin != -1);
}

/*
 * There is a nasty bug in some older SMP boards, their mptable lies
 * about the timer IRQ. We do the following to work around the situation:
 *
 *	- timer IRQ defaults to IO-APIC IRQ
 *	- if this function detects that timer IRQs are defunct, then we fall
 *	  back to ISA timer IRQs
 */
static int __init timer_irq_works(void)
{
    unsigned long t1, flags;

    t1 = ACCESS_ONCE(pit0_ticks);

    local_save_flags(flags);
    local_irq_enable();
    /* Let ten ticks pass... */
    mdelay((10 * 1000) / HZ);
    local_irq_restore(flags);

    /*
     * Expect a few ticks at least, to be sure some possible
     * glue logic does not lock up after one or two first
     * ticks in a non-ExtINT mode.  Also the local APIC
     * might have cached one ExtINT interrupt.  Finally, at
     * least one tick may be lost due to delays.
     */
    if ( (ACCESS_ONCE(pit0_ticks) - t1) > 4 )
        return 1;

    return 0;
}

/*
 * In the SMP+IOAPIC case it might happen that there are an unspecified
 * number of pending IRQ events unhandled. These cases are very rare,
 * so we 'resend' these IRQs via IPIs, to the same CPU. It's much
 * better to do it this way as thus we do not have to be aware of
 * 'pending' interrupts in the IRQ path, except at this point.
 */
/*
 * Edge triggered needs to resend any interrupt
 * that was delayed but this is now handled in the device
 * independent code.
 */

/*
 * Starting up a edge-triggered IO-APIC interrupt is
 * nasty - we need to make sure that we get the edge.
 * If it is already asserted for some reason, we need
 * return 1 to indicate that is was pending.
 *
 * This is not complete - we should be able to fake
 * an edge even if it isn't on the 8259A...
 */
static unsigned int cf_check startup_edge_ioapic_irq(struct irq_desc *desc)
{
    int was_pending = 0;
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    if (platform_legacy_irq(desc->irq)) {
        disable_8259A_irq(desc);
        if (i8259A_irq_pending(desc->irq))
            was_pending = 1;
    }
    __unmask_IO_APIC_irq(desc->irq);
    spin_unlock_irqrestore(&ioapic_lock, flags);

    return was_pending;
}

/*
 * Once we have recorded IRQ_PENDING already, we can mask the
 * interrupt for real. This prevents IRQ storms from unhandled
 * devices.
 */
static void cf_check ack_edge_ioapic_irq(struct irq_desc *desc)
{
    irq_complete_move(desc);
    move_native_irq(desc);

    if ((desc->status & (IRQ_PENDING | IRQ_DISABLED))
        == (IRQ_PENDING | IRQ_DISABLED))
        mask_IO_APIC_irq(desc);
    ack_APIC_irq();
}

/*
 * Level triggered interrupts can just be masked,
 * and shutting down and starting up the interrupt
 * is the same as enabling and disabling them -- except
 * with a startup need to return a "was pending" value.
 *
 * Level triggered interrupts are special because we
 * do not touch any IO-APIC register while handling
 * them. We ack the APIC in the end-IRQ handler, not
 * in the start-IRQ-handler. Protection against reentrance
 * from the same interrupt is still provided, both by the
 * generic IRQ layer and by the fact that an unacked local
 * APIC does not accept IRQs.
 */
static unsigned int cf_check startup_level_ioapic_irq(struct irq_desc *desc)
{
    unmask_IO_APIC_irq(desc);

    return 0; /* don't check for pending */
}

static int __init cf_check setup_ioapic_ack(const char *s)
{
    if ( !strcmp(s, "old") )
    {
        ioapic_ack_new = false;
        ioapic_ack_forced = true;
    }
    else if ( !strcmp(s, "new") )
    {
        ioapic_ack_new = true;
        ioapic_ack_forced = true;
    }
    else
        return -EINVAL;

    return 0;
}
custom_param("ioapic_ack", setup_ioapic_ack);

static bool io_apic_level_ack_pending(unsigned int irq)
{
    struct irq_pin_list *entry;
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    for (entry = &irq_2_pin[irq]; entry; entry = pin_list_next(entry)) {
        unsigned int reg;
        int pin;

        pin = entry->pin;
        if (pin == -1)
            continue;
        reg = io_apic_read(entry->apic, 0x10 + pin*2);
        /* Is the remote IRR bit set? */
        if (reg & IO_APIC_REDIR_REMOTE_IRR) {
            spin_unlock_irqrestore(&ioapic_lock, flags);
            return 1;
        }
    }
    spin_unlock_irqrestore(&ioapic_lock, flags);

    return 0;
}

static void cf_check mask_and_ack_level_ioapic_irq(struct irq_desc *desc)
{
    bool is_level;

    irq_complete_move(desc);

    if ( !directed_eoi_enabled )
        mask_IO_APIC_irq(desc);

/*
 * It appears there is an erratum which affects at least version 0x11
 * of I/O APIC (that's the 82093AA and cores integrated into various
 * chipsets).  Under certain conditions a level-triggered interrupt is
 * erroneously delivered as edge-triggered one but the respective IRR
 * bit gets set nevertheless.  As a result the I/O unit expects an EOI
 * message but it will never arrive and further interrupts are blocked
 * from the source.  The exact reason is so far unknown, but the
 * phenomenon was observed when two consecutive interrupt requests
 * from a given source get delivered to the same CPU and the source is
 * temporarily disabled in between.
 *
 * A workaround is to simulate an EOI message manually.  We achieve it
 * by setting the trigger mode to edge and then to level when the edge
 * trigger mode gets detected in the TMR of a local APIC for a
 * level-triggered interrupt.  We mask the source for the time of the
 * operation to prevent an edge-triggered interrupt escaping meanwhile.
 * The idea is from Manfred Spraul.  --macro
 */

    is_level = apic_tmr_read(desc->arch.vector);

    ack_APIC_irq();
    
    if ( directed_eoi_enabled )
        return;

    if ((desc->status & IRQ_MOVE_PENDING) &&
       !io_apic_level_ack_pending(desc->irq))
        move_masked_irq(desc);

    if ( !is_level )
    {
        spin_lock(&ioapic_lock);
        __edge_IO_APIC_irq(desc->irq);
        __level_IO_APIC_irq(desc->irq);
        spin_unlock(&ioapic_lock);
    }
}

static void cf_check end_level_ioapic_irq_old(struct irq_desc *desc, u8 vector)
{
    if ( directed_eoi_enabled )
    {
        if ( !(desc->status & (IRQ_DISABLED|IRQ_MOVE_PENDING)) )
        {
            eoi_IO_APIC_irq(desc);
            return;
        }

        mask_IO_APIC_irq(desc);
        eoi_IO_APIC_irq(desc);
        if ( (desc->status & IRQ_MOVE_PENDING) &&
             !io_apic_level_ack_pending(desc->irq) )
            move_masked_irq(desc);
    }

    if ( !(desc->status & IRQ_DISABLED) )
        unmask_IO_APIC_irq(desc);
}

static void cf_check end_level_ioapic_irq_new(struct irq_desc *desc, u8 vector)
{
/*
 * It appears there is an erratum which affects at least version 0x11
 * of I/O APIC (that's the 82093AA and cores integrated into various
 * chipsets).  Under certain conditions a level-triggered interrupt is
 * erroneously delivered as edge-triggered one but the respective IRR
 * bit gets set nevertheless.  As a result the I/O unit expects an EOI
 * message but it will never arrive and further interrupts are blocked
 * from the source.  The exact reason is so far unknown, but the
 * phenomenon was observed when two consecutive interrupt requests
 * from a given source get delivered to the same CPU and the source is
 * temporarily disabled in between.
 *
 * A workaround is to simulate an EOI message manually.  We achieve it
 * by setting the trigger mode to edge and then to level when the edge
 * trigger mode gets detected in the TMR of a local APIC for a
 * level-triggered interrupt.  We mask the source for the time of the
 * operation to prevent an edge-triggered interrupt escaping meanwhile.
 * The idea is from Manfred Spraul.  --macro
 */
    unsigned int i = desc->arch.vector;
    bool is_level;

    /* Manually EOI the old vector if we are moving to the new */
    if ( vector && i != vector )
        eoi_IO_APIC_irq(desc);

    is_level = apic_tmr_read(i);

    end_nonmaskable_irq(desc, vector);

    if ( (desc->status & IRQ_MOVE_PENDING) &&
         !io_apic_level_ack_pending(desc->irq) )
        move_native_irq(desc);

    if ( !is_level )
    {
        spin_lock(&ioapic_lock);
        __mask_IO_APIC_irq(desc->irq);
        __edge_IO_APIC_irq(desc->irq);
        __level_IO_APIC_irq(desc->irq);
        if ( !(desc->status & IRQ_DISABLED) )
            __unmask_IO_APIC_irq(desc->irq);
        spin_unlock(&ioapic_lock);
    }
}

/*
 * Level and edge triggered IO-APIC interrupts need different handling,
 * so we use two separate IRQ descriptors. Edge triggered IRQs can be
 * handled with the level-triggered descriptor, but that one has slightly
 * more overhead. Level-triggered interrupts cannot be handled with the
 * edge-triggered handler, without risking IRQ storms and other ugly
 * races.
 */
static hw_irq_controller ioapic_edge_type = {
    .typename 	= "IO-APIC-edge",
    .startup 	= startup_edge_ioapic_irq,
    .shutdown 	= irq_shutdown_none,
    .enable 	= unmask_IO_APIC_irq,
    .disable 	= irq_disable_none,
    .ack 		= ack_edge_ioapic_irq,
    .set_affinity 	= set_ioapic_affinity_irq,
};

static struct hw_interrupt_type __read_mostly ioapic_level_type = {
    .typename 	= "IO-APIC-level",
    .startup 	= startup_level_ioapic_irq,
    .shutdown 	= mask_IO_APIC_irq,
    .enable 	= unmask_IO_APIC_irq,
    .disable 	= mask_IO_APIC_irq,
    .ack 		= mask_and_ack_level_ioapic_irq,
    .end 		= end_level_ioapic_irq_old,
    .set_affinity 	= set_ioapic_affinity_irq,
};

static inline void init_IO_APIC_traps(void)
{
    int irq;
    /* Xen: This is way simpler than the Linux implementation. */
    for (irq = 0; platform_legacy_irq(irq); irq++)
        if (IO_APIC_IRQ(irq) && !irq_to_vector(irq))
            make_8259A_irq(irq);
}

static void cf_check enable_lapic_irq(struct irq_desc *desc)
{
    unsigned long v;

    v = apic_read(APIC_LVT0);
    apic_write(APIC_LVT0, v & ~APIC_LVT_MASKED);
}

static void cf_check disable_lapic_irq(struct irq_desc *desc)
{
    unsigned long v;

    v = apic_read(APIC_LVT0);
    apic_write(APIC_LVT0, v | APIC_LVT_MASKED);
}

static void cf_check ack_lapic_irq(struct irq_desc *desc)
{
    ack_APIC_irq();
}

static hw_irq_controller lapic_irq_type = {
    .typename 	= "local-APIC-edge",
    .startup 	= NULL, /* startup_irq() not used for IRQ0 */
    .shutdown 	= NULL, /* shutdown_irq() not used for IRQ0 */
    .enable 	= enable_lapic_irq,
    .disable 	= disable_lapic_irq,
    .ack 		= ack_lapic_irq,
};

/*
 * This looks a bit hackish but it's about the only one way of sending
 * a few INTA cycles to 8259As and any associated glue logic.  ICR does
 * not support the ExtINT mode, unfortunately.  We need to send these
 * cycles as some i82489DX-based boards have glue logic that keeps the
 * 8259A interrupt line asserted until INTA.  --macro
 */
static void __init unlock_ExtINT_logic(void)
{
    int apic, pin, i;
    struct IO_APIC_route_entry entry0, entry1;
    unsigned char save_control, save_freq_select;

    pin = find_isa_irq_pin(8, mp_INT);
    apic = find_isa_irq_apic(8, mp_INT);
    if ( pin == -1 || apic == -1 )
        return;

    entry0 = ioapic_read_entry(apic, pin, false);
    clear_IO_APIC_pin(apic, pin);

    memset(&entry1, 0, sizeof(entry1));

    entry1.dest_mode = 0;			/* physical delivery */
    entry1.mask = 0;			/* unmask IRQ now */
    SET_DEST(entry1, physical, get_apic_id());
    entry1.delivery_mode = dest_ExtINT;
    entry1.polarity = entry0.polarity;
    entry1.trigger = 0;
    entry1.vector = 0;

    ioapic_write_entry(apic, pin, false, entry1);

    save_control = CMOS_READ(RTC_CONTROL);
    save_freq_select = CMOS_READ(RTC_FREQ_SELECT);
    CMOS_WRITE((save_freq_select & ~RTC_RATE_SELECT) | 0x6,
               RTC_FREQ_SELECT);
    CMOS_WRITE(save_control | RTC_PIE, RTC_CONTROL);

    i = 100;
    while (i-- > 0) {
        mdelay(10);
        if ((CMOS_READ(RTC_INTR_FLAGS) & RTC_PF) == RTC_PF)
            i -= 10;
    }

    CMOS_WRITE(save_control, RTC_CONTROL);
    CMOS_WRITE(save_freq_select, RTC_FREQ_SELECT);
    clear_IO_APIC_pin(apic, pin);

    ioapic_write_entry(apic, pin, false, entry0);
}

/*
 * This code may look a bit paranoid, but it's supposed to cooperate with
 * a wide range of boards and BIOS bugs.  Fortunately only the timer IRQ
 * is so screwy.  Thanks to Brian Perkins for testing/hacking this beast
 * fanatically on his truly buggy board.
 */
static void __init check_timer(void)
{
    int apic1, pin1, apic2, pin2;
    int vector, ret;
    unsigned long flags;

    local_irq_save(flags);

    /*
     * get/set the timer IRQ vector:
     */
    disable_8259A_irq(irq_to_desc(0));
    vector = IRQ0_VECTOR;
    clear_irq_vector(0);

    if ((ret = bind_irq_vector(0, vector, &cpumask_all)))
        printk(KERN_ERR"..IRQ0 is not set correctly with ioapic!!!, err:%d\n", ret);
    
    irq_desc[0].status &= ~IRQ_DISABLED;

    /*
     * Subtle, code in do_timer_interrupt() expects an AEOI
     * mode for the 8259A whenever interrupts are routed
     * through I/O APICs.  Also IRQ0 has to be enabled in
     * the 8259A which implies the virtual wire has to be
     * disabled in the local APIC.
     */
    apic_write(APIC_LVT0, APIC_LVT_MASKED | APIC_DM_EXTINT);
    init_8259A(1);
    /* XEN: Ripped out the legacy missed-tick logic, so below is not needed. */
    /*timer_ack = 1;*/
    /*enable_8259A_irq(irq_to_desc(0));*/

    pin1  = find_isa_irq_pin(0, mp_INT);
    apic1 = find_isa_irq_apic(0, mp_INT);
    pin2  = ioapic_i8259.pin;
    apic2 = ioapic_i8259.apic;

    printk(KERN_INFO "..TIMER: vector=0x%02X apic1=%d pin1=%d apic2=%d pin2=%d\n",
           vector, apic1, pin1, apic2, pin2);

    if (pin1 != -1) {
        /*
         * Ok, does IRQ0 through the IOAPIC work?
         */
        unmask_IO_APIC_irq(irq_to_desc(0));
        if (timer_irq_works()) {
            local_irq_restore(flags);
            return;
        }

        /*
         * Intel chipsets from Skylake/ApolloLake onwards can statically clock
         * gate the 8254 PIT.  This option is enabled by default in slightly
         * later systems, as turning the PIT off is a prerequisite to entering
         * the C11 power saving state.
         *
         * Xen currently depends on the legacy timer interrupt being active
         * while IRQ routing is configured.
         *
         * If the user hasn't made an explicit choice, attempt to reconfigure
         * the HPET into legacy mode to re-establish the timer interrupt.
         */
        if ( opt_hpet_legacy_replacement < 0 &&
             hpet_enable_legacy_replacement_mode() )
        {
            printk(XENLOG_ERR "..no 8254 timer found - trying HPET Legacy Replacement Mode\n");

            if ( timer_irq_works() )
            {
                printk(XENLOG_DEBUG
                       "IRQ test with HPET Legacy Replacement Mode worked - disabling it again\n");
                hpet_disable_legacy_replacement_mode();
                local_irq_restore(flags);
                return;
            }

            /* Legacy Replacement mode hasn't helped.  Undo it. */
            printk(XENLOG_ERR "..no HPET timer found - reverting Legacy Replacement Mode\n");
            hpet_disable_legacy_replacement_mode();
        }

        clear_IO_APIC_pin(apic1, pin1);
        printk(KERN_ERR "..MP-BIOS bug: 8254 timer not connected to "
               "IO-APIC\n");
    }

    printk(KERN_INFO "...trying to set up timer (IRQ0) through the 8259A ... ");
    if (pin2 != -1) {
        printk("\n..... (found pin %d) ...", pin2);
        /*
         * legacy devices should be connected to IO APIC #0
         */
        setup_ExtINT_IRQ0_pin(apic2, pin2, vector);
        if (timer_irq_works()) {
            local_irq_restore(flags);
            printk("works.\n");
            if (pin1 != -1)
                replace_pin_at_irq(0, apic1, pin1, apic2, pin2);
            else
                add_pin_to_irq(0, apic2, pin2);
            return;
        }
        /*
         * Cleanup, just in case ...
         */
        clear_IO_APIC_pin(apic2, pin2);
    }
    printk(" failed.\n");

    printk(KERN_INFO "...trying to set up timer as Virtual Wire IRQ...");

    disable_8259A_irq(irq_to_desc(0));
    irq_desc[0].handler = &lapic_irq_type;
    apic_write(APIC_LVT0, APIC_DM_FIXED | vector);	/* Fixed mode */
    enable_8259A_irq(irq_to_desc(0));

    if (timer_irq_works()) {
        local_irq_restore(flags);
        printk(" works.\n");
        return;
    }
    apic_write(APIC_LVT0, APIC_LVT_MASKED | APIC_DM_FIXED | vector);
    printk(" failed.\n");

    printk(KERN_INFO "...trying to set up timer as ExtINT IRQ...");

    /*timer_ack = 0;*/
    init_8259A(0);
    make_8259A_irq(0);
    apic_write(APIC_LVT0, APIC_DM_EXTINT);

    unlock_ExtINT_logic();

    local_irq_restore(flags);

    if (timer_irq_works()) {
        printk(" works.\n");
        return;
    }
    printk(" failed :(.\n");
    panic("IO-APIC + timer doesn't work!  Boot with apic_verbosity=debug and send a report\n");
}

/*
 *
 * IRQ's that are handled by the PIC in the MPS IOAPIC case.
 * - IRQ2 is the cascade IRQ, and cannot be a io-apic IRQ.
 *   Linux doesn't really care, as it's not actually used
 *   for any interrupt handling anyway.
 */
#define PIC_IRQS	(1 << PIC_CASCADE_IR)

static struct IO_APIC_route_entry *ioapic_pm_state;

static void __init ioapic_pm_state_alloc(void)
{
    int i, nr_entry = 0;

    for (i = 0; i < nr_ioapics; i++)
        nr_entry += nr_ioapic_entries[i];

    ioapic_pm_state = _xmalloc(sizeof(struct IO_APIC_route_entry)*nr_entry,
                               sizeof(struct IO_APIC_route_entry));
    BUG_ON(ioapic_pm_state == NULL);
}

void __init setup_IO_APIC(void)
{
    if (acpi_ioapic)
        io_apic_irqs = ~0;	/* all IRQs go through IOAPIC */
    else
        io_apic_irqs = ~PIC_IRQS;

    printk("ENABLING IO-APIC IRQs\n");
    printk(" -> Using %s ACK method\n", ioapic_ack_new ? "new" : "old");

    if (ioapic_ack_new) {
        ioapic_level_type.ack = irq_complete_move;
        ioapic_level_type.end = end_level_ioapic_irq_new;
    }

    setup_IO_APIC_irqs();
    init_IO_APIC_traps();
    check_timer();
    print_IO_APIC();
    ioapic_pm_state_alloc();

    register_keyhandler('z', _print_IO_APIC_keyhandler, "dump IOAPIC info", 1);
}

void ioapic_suspend(void)
{
    struct IO_APIC_route_entry *entry = ioapic_pm_state;
    unsigned long flags;
    int apic, i;

    spin_lock_irqsave(&ioapic_lock, flags);
    for (apic = 0; apic < nr_ioapics; apic++) {
        for (i = 0; i < nr_ioapic_entries[apic]; i ++, entry ++ ) {
            *(((int *)entry) + 1) = __io_apic_read(apic, 0x11 + 2 * i);
            *(((int *)entry) + 0) = __io_apic_read(apic, 0x10 + 2 * i);
        }
    }
    spin_unlock_irqrestore(&ioapic_lock, flags);
}

void ioapic_resume(void)
{
    struct IO_APIC_route_entry *entry = ioapic_pm_state;
    unsigned long flags;
    union IO_APIC_reg_00 reg_00;
    int i, apic;

    spin_lock_irqsave(&ioapic_lock, flags);
    for (apic = 0; apic < nr_ioapics; apic++){
        if (!nr_ioapic_entries[apic])
            continue;
        reg_00.raw = __io_apic_read(apic, 0);
        if (reg_00.bits.ID != mp_ioapics[apic].mpc_apicid) {
            reg_00.bits.ID = mp_ioapics[apic].mpc_apicid;
            __io_apic_write(apic, 0, reg_00.raw);
        }
        for (i = 0; i < nr_ioapic_entries[apic]; i++, entry++)
            __ioapic_write_entry(apic, i, true, *entry);
    }
    spin_unlock_irqrestore(&ioapic_lock, flags);
}

int __init io_apic_get_version (int ioapic)
{
    union IO_APIC_reg_01	reg_01;
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    reg_01.raw = io_apic_read(ioapic, 1);
    spin_unlock_irqrestore(&ioapic_lock, flags);

    return reg_01.bits.version;
}


int __init io_apic_get_redir_entries (int ioapic)
{
    union IO_APIC_reg_01	reg_01;
    unsigned long flags;

    spin_lock_irqsave(&ioapic_lock, flags);
    reg_01.raw = io_apic_read(ioapic, 1);
    spin_unlock_irqrestore(&ioapic_lock, flags);

    return reg_01.bits.entries;
}


int io_apic_set_pci_routing (int ioapic, int pin, int irq, int edge_level, int active_high_low)
{
    struct irq_desc *desc = irq_to_desc(irq);
    struct IO_APIC_route_entry entry;
    unsigned long flags;
    int vector;

    if (!IO_APIC_IRQ(irq)) {
        printk(KERN_ERR "IOAPIC[%d]: Invalid reference to IRQ %d\n",
               ioapic, irq);
        return -EINVAL;
    }

    /*
     * Generate a PCI IRQ routing entry and program the IOAPIC accordingly.
     * Note that we mask (disable) IRQs now -- these get enabled when the
     * corresponding device driver registers for this IRQ.
     */

    memset(&entry,0,sizeof(entry));

    entry.delivery_mode = dest_Fixed;
    entry.dest_mode = 0; /* physical delivery */
    entry.trigger = edge_level;
    entry.polarity = active_high_low;
    entry.mask  = 1;

    /*
     * IRQs < 16 are already in the irq_2_pin[] map
     */
    if (!platform_legacy_irq(irq))
        add_pin_to_irq(irq, ioapic, pin);

    vector = assign_irq_vector(irq, NULL);
    if (vector < 0)
        return vector;
    entry.vector = vector;

    if (cpumask_intersects(desc->arch.cpu_mask, TARGET_CPUS)) {
        cpumask_t *mask = this_cpu(scratch_cpumask);

        cpumask_and(mask, desc->arch.cpu_mask, TARGET_CPUS);
        SET_DEST(entry, logical, cpu_mask_to_apicid(mask));
    } else {
        printk(XENLOG_ERR "IRQ%d: no target CPU (%*pb vs %*pb)\n",
               irq, CPUMASK_PR(desc->arch.cpu_mask), CPUMASK_PR(TARGET_CPUS));
        desc->status |= IRQ_DISABLED;
    }

    apic_printk(APIC_DEBUG, KERN_DEBUG "IOAPIC[%d]: Set PCI routing entry "
		"(%d-%d -> %#x -> IRQ %d Mode:%i Active:%i)\n", ioapic,
		mp_ioapics[ioapic].mpc_apicid, pin, entry.vector, irq,
		edge_level, active_high_low);

    ioapic_register_intr(irq, edge_level);

    if (!ioapic && platform_legacy_irq(irq))
        disable_8259A_irq(desc);

    spin_lock_irqsave(&ioapic_lock, flags);
    __ioapic_write_entry(ioapic, pin, false, entry);
    spin_unlock(&ioapic_lock);

    spin_lock(&desc->lock);
    if (!(desc->status & (IRQ_DISABLED | IRQ_GUEST)))
        desc->handler->startup(desc);
    spin_unlock_irqrestore(&desc->lock, flags);

    return 0;
}

static int ioapic_physbase_to_id(unsigned long physbase)
{
    int apic;
    for ( apic = 0; apic < nr_ioapics; apic++ )
    {
        if ( !nr_ioapic_entries[apic] )
            continue;
        if ( mp_ioapics[apic].mpc_apicaddr == physbase )
            return apic;
    }
    return -EINVAL;
}

static int apic_pin_2_gsi_irq(int apic, int pin)
{
    int idx;

    if (apic < 0)
       return -EINVAL;

    idx = find_irq_entry(apic, pin, mp_INT);

    return idx >= 0 ? pin_2_irq(idx, apic, pin)
                    : io_apic_gsi_base(apic) + pin;
}

int ioapic_guest_read(unsigned long physbase, unsigned int reg, u32 *pval)
{
    int apic;
    unsigned long flags;

    if ( (apic = ioapic_physbase_to_id(physbase)) < 0 )
        return apic;

    spin_lock_irqsave(&ioapic_lock, flags);
    *pval = io_apic_read(apic, reg);
    spin_unlock_irqrestore(&ioapic_lock, flags);

    return 0;
}

#define WARN_BOGUS_WRITE(f, a...)                             \
    dprintk(XENLOG_INFO, "IO-APIC: apic=%d, pin=%d, irq=%d\n" \
            XENLOG_INFO "IO-APIC: new_entry=%08x\n"           \
            XENLOG_INFO "IO-APIC: " f "\n",                   \
            apic, pin, irq, rte.low, ##a )

int ioapic_guest_write(unsigned long physbase, unsigned int reg, u32 val)
{
    int apic, pin, irq, ret, pirq;
    struct IO_APIC_route_entry rte = { };
    unsigned long flags;
    struct irq_desc *desc;

    if ( (apic = ioapic_physbase_to_id(physbase)) < 0 )
        return apic;

    /* Only write to the first half of a route entry. */
    if ( (reg < 0x10) || (reg & 1) )
        return 0;
    
    pin = (reg - 0x10) >> 1;

    /* Write first half from guest; second half is target info. */
    rte.low = val;

    /*
     * What about weird destination types?
     *  SMI:    Ignore? Ought to be set up by the BIOS.
     *  NMI:    Ignore? Watchdog functionality is Xen's concern.
     *  INIT:   Definitely ignore: probably a guest OS bug.
     *  ExtINT: Ignore? Linux only asserts this at start of day.
     * For now, print a message and return an error. We can fix up on demand.
     */
    if ( rte.delivery_mode > dest_LowestPrio )
    {
        printk("ERROR: Attempt to write weird IOAPIC destination mode!\n");
        printk("       APIC=%d/%d, lo-reg=%x\n", apic, pin, val);
        return -EINVAL;
    }

    /*
     * The guest does not know physical APIC arrangement (flat vs. cluster).
     * Apply genapic conventions for this platform.
     */
    rte.delivery_mode = dest_Fixed;
    rte.dest_mode     = 0; /* physical delivery */

    irq = apic_pin_2_gsi_irq(apic, pin);
    if ( irq < 0 )
        return irq;

    desc = irq_to_desc(irq);

    /*
     * Since PHYSDEVOP_alloc_irq_vector is dummy, rte.vector is the pirq
     * which corresponds to this ioapic pin, retrieve it for building
     * pirq and irq mapping. Where the GSI is greater than 256, we assume
     * that dom0 pirq == irq.
     */
    if ( !rte.mask )
    {
        pirq = (irq >= 256) ? irq : rte.vector;
        if ( pirq >= hardware_domain->nr_pirqs )
            return -EINVAL;
    }
    else
        pirq = -1;
    
    if ( desc->action )
    {
        spin_lock_irqsave(&ioapic_lock, flags);
        ret = io_apic_read(apic, 0x10 + 2 * pin);
        spin_unlock_irqrestore(&ioapic_lock, flags);
        rte.vector = desc->arch.vector;
        if ( rte.low != ret )
            WARN_BOGUS_WRITE("old_entry=%08x pirq=%d\n" XENLOG_INFO
                             "IO-APIC: Attempt to modify IO-APIC pin for in-use IRQ!",
                             ret, pirq);
        return 0;
    }

    if ( desc->arch.vector <= 0 || desc->arch.vector > LAST_DYNAMIC_VECTOR ||
         desc->handler->enable == enable_8259A_irq )
    {
        int vector = desc->arch.vector;

        if ( vector < FIRST_HIPRIORITY_VECTOR )
            add_pin_to_irq(irq, apic, pin);
        else
            desc->arch.vector = IRQ_VECTOR_UNASSIGNED;
        ret = assign_irq_vector(irq, NULL);
        if ( ret < 0 )
        {
            if ( vector < FIRST_HIPRIORITY_VECTOR )
                remove_pin_from_irq(irq, apic, pin);
            else
                desc->arch.vector = vector;
            return ret;
        }

        printk(XENLOG_INFO "allocated vector %02x for irq %d\n", ret, irq);
    }
    if ( pirq >= 0 )
    {
        write_lock(&hardware_domain->event_lock);
        ret = map_domain_pirq(hardware_domain, pirq, irq,
                              MAP_PIRQ_TYPE_GSI, NULL);
        write_unlock(&hardware_domain->event_lock);
        if ( ret < 0 )
            return ret;
    }

    spin_lock_irqsave(&ioapic_lock, flags);
    /* Set the correct irq-handling type. */
    desc->handler = rte.trigger ? 
        &ioapic_level_type: &ioapic_edge_type;

    /* Mask iff level triggered. */
    rte.mask = rte.trigger;
    /* Set the vector field to the real vector! */
    rte.vector = desc->arch.vector;

    if ( cpumask_intersects(desc->arch.cpu_mask, TARGET_CPUS) )
    {
        cpumask_t *mask = this_cpu(scratch_cpumask);

        cpumask_and(mask, desc->arch.cpu_mask, TARGET_CPUS);
        SET_DEST(rte, logical, cpu_mask_to_apicid(mask));
    }
    else
    {
        gprintk(XENLOG_ERR, "IRQ%d: no target CPU (%*pb vs %*pb)\n",
               irq, CPUMASK_PR(desc->arch.cpu_mask), CPUMASK_PR(TARGET_CPUS));
        desc->status |= IRQ_DISABLED;
        rte.mask = 1;
    }

    __ioapic_write_entry(apic, pin, false, rte);
    
    spin_unlock_irqrestore(&ioapic_lock, flags);

    return 0;
}

static const char * delivery_mode_2_str(
    const enum ioapic_irq_destination_types mode)
{
    switch ( mode )
    {
    case dest_Fixed: return "Fixed";
    case dest_LowestPrio: return "LoPri";
    case dest_SMI: return "SMI";
    case dest_NMI: return "NMI";
    case dest_INIT: return "INIT";
    case dest_ExtINT: return "ExINT";
    case dest__reserved_1:
    case dest__reserved_2: return "Resvd";
    default: return "INVAL";
    }
}

void dump_ioapic_irq_info(void)
{
    struct irq_pin_list *entry;
    struct IO_APIC_route_entry rte;
    unsigned int irq, pin, printed = 0;

    if ( !irq_2_pin )
        return;

    for ( irq = 0; irq < nr_irqs_gsi; irq++ )
    {
        if ( !(irq & 0x1f) )
            process_pending_softirqs();

        entry = &irq_2_pin[irq];
        if ( entry->pin == -1 )
            continue;

        if ( !printed++ )
            printk("IO-APIC interrupt information:\n");

        printk("    IRQ%3d Vec%3d:\n", irq, irq_to_vector(irq));

        for ( ; entry; entry = pin_list_next(entry))
        {
            pin = entry->pin;

            printk("      Apic 0x%02x, Pin %2d: ", entry->apic, pin);

            rte = ioapic_read_entry(entry->apic, pin, false);

            printk("vec=%02x delivery=%-5s dest=%c status=%d "
                   "polarity=%d irr=%d trig=%c mask=%d dest_id:%0*x\n",
                   rte.vector, delivery_mode_2_str(rte.delivery_mode),
                   rte.dest_mode ? 'L' : 'P',
                   rte.delivery_status, rte.polarity, rte.irr,
                   rte.trigger ? 'L' : 'E', rte.mask,
                   (x2apic_enabled && iommu_intremap) ? 8 : 2,
                   (x2apic_enabled && iommu_intremap) ?
                       rte.dest.dest32 : rte.dest.logical.logical_dest);
        }
    }
}

static unsigned int __initdata max_gsi_irqs;
integer_param("max_gsi_irqs", max_gsi_irqs);

static __init bool bad_ioapic_register(unsigned int idx)
{
    union IO_APIC_reg_00 reg_00 = { .raw = io_apic_read(idx, 0) };
    union IO_APIC_reg_01 reg_01 = { .raw = io_apic_read(idx, 1) };
    union IO_APIC_reg_02 reg_02 = { .raw = io_apic_read(idx, 2) };

    if ( reg_00.raw == -1 && reg_01.raw == -1 && reg_02.raw == -1 )
    {
        printk(KERN_WARNING "I/O APIC %#x registers return all ones, skipping!\n",
               mp_ioapics[idx].mpc_apicaddr);
        return true;
    }

    return false;
}

static void __init ioapic_init_mappings(void)
{
    unsigned int i, idx = FIX_IO_APIC_BASE_0;

    nr_irqs_gsi = 0;

    for ( i = 0; i < nr_ioapics; i++, idx++ )
    {
        union IO_APIC_reg_01 reg_01;
        paddr_t ioapic_phys = mp_ioapics[i].mpc_apicaddr;

        if ( !ioapic_phys || !IS_ALIGNED(ioapic_phys, KB(1)) )
        {
            printk(KERN_ERR
                   "WARNING: bogus IO-APIC address %08lx found in MPTABLE, disabling IO-APIC support\n",
                   ioapic_phys);
            smp_found_config = false;
            skip_ioapic_setup = true;
            break;
        }

        set_fixmap_nocache(idx, ioapic_phys);
        apic_printk(APIC_VERBOSE, "mapped IOAPIC to %p (%08lx)\n",
                    fix_to_virt(idx), ioapic_phys);

        if ( bad_ioapic_register(i) )
        {
            clear_fixmap(idx);
            continue;
        }

        /* The number of IO-APIC IRQ registers (== #pins): */
        reg_01.raw = io_apic_read(i, 1);
        nr_ioapic_entries[i] = reg_01.bits.entries + 1;
        nr_irqs_gsi += nr_ioapic_entries[i];

        if ( rangeset_add_singleton(mmio_ro_ranges,
                                    ioapic_phys >> PAGE_SHIFT) )
            printk(KERN_ERR "Failed to mark IO-APIC page %lx read-only\n",
                   ioapic_phys);
    }
}

void __init ioapic_init(void)
{
    if ( smp_found_config )
        ioapic_init_mappings();

    nr_irqs_gsi = max(nr_irqs_gsi, highest_gsi() + 1);

    if ( max_gsi_irqs == 0 )
        max_gsi_irqs = nr_irqs ? nr_irqs / 8 : PAGE_SIZE;
    else if ( nr_irqs != 0 && max_gsi_irqs > nr_irqs )
    {
        printk(XENLOG_WARNING "\"max_gsi_irqs=\" cannot be specified larger"
                              " than \"nr_irqs=\"\n");
        max_gsi_irqs = nr_irqs;
    }
    if ( max_gsi_irqs < NR_ISA_IRQS )
        max_gsi_irqs = NR_ISA_IRQS;

    /* for PHYSDEVOP_pirq_eoi_gmfn guest assumptions */
    if ( max_gsi_irqs > PAGE_SIZE * 8 )
        max_gsi_irqs = PAGE_SIZE * 8;

    if ( !smp_found_config || skip_ioapic_setup || nr_irqs_gsi < NR_ISA_IRQS )
        nr_irqs_gsi = NR_ISA_IRQS;
    else if ( nr_irqs_gsi > max_gsi_irqs )
    {
        printk(XENLOG_WARNING "Limiting to %u GSI IRQs (found %u)\n",
               max_gsi_irqs, nr_irqs_gsi);
        nr_irqs_gsi = max_gsi_irqs;
    }

    if ( nr_irqs == 0 )
        nr_irqs = cpu_has_apic ?
                  max(0U + num_present_cpus() * NR_DYNAMIC_VECTORS,
                      8 * nr_irqs_gsi) :
                  nr_irqs_gsi;
    else if ( nr_irqs < NR_ISA_IRQS )
        nr_irqs = NR_ISA_IRQS;
    printk(XENLOG_INFO "IRQ limits: %u GSI, %u MSI/MSI-X\n",
           nr_irqs_gsi, nr_irqs - nr_irqs_gsi);
}

unsigned int __hwdom_init arch_hwdom_irqs(const struct domain *d)
{
    unsigned int n = num_present_cpus();
    /* Bounding by the domain pirq EOI bitmap capacity. */
    const unsigned int max_irqs = min_t(unsigned int, nr_irqs,
                                        PAGE_SIZE * BITS_PER_BYTE);

    if ( is_system_domain(d) )
        return max_irqs;

    /* PVH (generally: HVM) can't use PHYSDEVOP_pirq_eoi_gmfn_v{1,2}. */
    if ( is_hvm_domain(d) )
        n = nr_irqs;
    else
    {
        if ( !d->domain_id )
            n = min(n, d->max_vcpus);
        n = min(nr_irqs_gsi + n * NR_DYNAMIC_VECTORS, max_irqs);
    }

    printk("%pd has maximum %u PIRQs\n", d, n);

    return n;
}
