//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2023 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//

import ArgumentParser
import Foundation

struct LocalPrPrecheck: ParsableCommand {
  static let configuration = CommandConfiguration(
    abstract: """
      Ensure changes are fully tested, formatted, and validated before pull request submission.
      """
  )

  @OptionGroup
  var arguments: SourceCodeGeneratorArguments

  func run() throws {
    let executor = LocalPrPrecheckExecutor(
      toolchain: try arguments.toolchain,
      verbose: arguments.verbose
    )
    try executor.run()
  }
}

struct LocalPrPrecheckExecutor {
  private let verbose: Bool
  private let generateSourceCodeExecutor: GenerateSourceCodeExecutor
  private let buildExecutor: BuildExecutor
  private let testExecutor: TestExecutor

  /// Creates an executor
  /// - Parameters:
  ///   - toolchain: The path to the toolchain that shall be used to build SwiftSyntax.
  ///   - verbose: Enable verbose logging.
  init(toolchain: URL, verbose: Bool = false) {
    self.verbose = verbose
    self.generateSourceCodeExecutor = GenerateSourceCodeExecutor(toolchain: toolchain, verbose: verbose)
    self.buildExecutor = BuildExecutor(
      swiftPMBuilder: SwiftPMBuilder(toolchain: toolchain, useLocalDeps: false, verbose: verbose)
    )
    self.testExecutor = TestExecutor(
      swiftPMBuilder: SwiftPMBuilder(toolchain: toolchain, useLocalDeps: false, verbose: verbose)
    )
  }

  private func checkFormatting() throws {
    try ProcessRunner(
      executableURL: Paths.swiftExec,
      arguments: ["format", "lint", "--strict", "--parallel", "--recursive", Paths.sourcesDir.path]
    )
    .run(captureStdout: false, captureStderr: false, verbose: verbose)
  }

  func run() throws {
    try checkFormatting()
    try generateSourceCodeExecutor.run(sourceDir: Paths.sourcesDir)
    try buildExecutor.run()
    try testExecutor.run()
  }
}
