//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2023 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//

import SwiftSyntax

/// Generates grammar doc comments for syntax nodes.
struct GrammarGenerator {

  /// Returns grammar for a ``TokenChoice``.
  ///
  /// - parameters:
  ///   - tokenChoice: ``TokenChoice`` to describe
  private func grammar(for tokenChoice: TokenChoice) -> String {
    switch tokenChoice {
    case .keyword(let keyword):
      return "`\(keyword.spec.name)`"
    case .token(let token):
      let tokenSpec = token.spec
      if let tokenText = tokenSpec.text {
        return "`\(tokenText)`"
      } else {
        return "`<\(tokenSpec.identifier)>`"
      }
    }
  }

  private func grammar(for child: Child) -> String {
    let optionality = child.isOptional ? "?" : ""
    switch child.kind {
    case .node(let kind):
      return "\(kind.doccLink)\(optionality)"
    case .nodeChoices(let choices, _):
      let choicesDescriptions = choices.map { grammar(for: $0) }
      return "(\(choicesDescriptions.joined(separator: " | ")))\(optionality)"
    case .collection(kind: let kind, _, _, _, _):
      return "\(kind.doccLink)\(optionality)"
    case .token(let choices, _, _):
      if choices.count == 1 {
        return "\(grammar(for: choices.first!))\(optionality)"
      } else {
        let choicesDescriptions = choices.map { grammar(for: $0) }
        return "(\(choicesDescriptions.joined(separator: " | ")))\(optionality)"
      }
    }
  }

  /// Generates a markdown list containing the children’s names and their
  /// grammar.
  ///
  /// - Parameter children: The children to show in the list
  static func childrenList(for children: [Child]) -> String {
    let generator = GrammarGenerator()
    return
      children
      .filter { !$0.isUnexpectedNodes }
      .map { " - `\($0.identifier)`: \(generator.grammar(for: $0))" }
      .joined(separator: "\n")
  }

  /// Generates a markdown string describing possible choices for the given child
  /// token.
  static func childTokenChoices(for choices: [TokenChoice]) -> String {
    let grammar = GrammarGenerator()

    if choices.count == 1 {
      return """
        ### Tokens

        For syntax trees generated by the parser, this is guaranteed to be \(grammar.grammar(for: choices.first!)).
        """
    } else {
      return """
        ### Tokens

        For syntax trees generated by the parser, this is guaranteed to be one of the following kinds:
        \(choices.map { " - \(grammar.grammar(for: $0))" }.joined(separator: "\n"))
        """
    }
  }
}
