//===----------------------------------------------------------------------===//
//
// This source file is part of the SwiftPrometheus open source project
//
// Copyright (c) 2018-2025 SwiftPrometheus project authors
// Licensed under Apache License v2.0
//
// See LICENSE.txt for license information
// See CONTRIBUTORS.txt for the list of SwiftPrometheus project authors
//
// SPDX-License-Identifier: Apache-2.0
//
//===----------------------------------------------------------------------===//

/// A descriptor that defines the components of a fully qualified Prometheus metric name.
///
/// The final, underscore-separated metric name is generated by the ``name`` computed property. The ``helpText``
/// is used to generate the corresponding `# HELP` line in the Prometheus exposition format.
/// - Warning: This initializer will trigger a `preconditionFailure` if ``metricName`` is an empty string.
public struct MetricNameDescriptor {
    /// An optional top-level namespace for the metric.
    public let namespace: String?

    /// An optional subsystem to group related metrics.
    public let subsystem: String?

    /// The required, descriptive base name of the metric.
    public let metricName: String

    /// An optional suffix describing the metric's unit (e.g., `total`).
    public let unitName: String?

    /// Optional help text for the metric. If a non-empty string is provided, it will be emitted as a `# HELP` line in the exposition format.
    /// If the parameter is omitted or an empty string is passed, the `# HELP` line will not be generated for this metric.
    public let helpText: String?

    /// Creates a new ``MetricNameDescriptor`` that defines the components of a fully qualified Prometheus metric name.
    ///
    /// - Parameter namespace: An optional top-level namespace for the metric.
    /// - Parameter subsystem: An optional subsystem to group related metrics within a namespace.
    /// - Parameter metricName: The required, descriptive base name of the metric.
    /// - Parameter unitName: An optional suffix describing the metric's unit (e.g., `total`).
    /// - Parameter helpText: Optional help text for the metric. If a non-empty string is provided, it will be emitted as a `# HELP` line in the exposition format.
    ///                       If the parameter is omitted or an empty string is passed, the `# HELP` line will not be generated for this metric.
    public init(
        namespace: String? = nil,
        subsystem: String? = nil,
        metricName: String,
        unitName: String? = nil,
        helpText: String? = nil
    ) {
        precondition(!metricName.isEmpty, "metricName must not be empty")
        self.namespace = namespace
        self.subsystem = subsystem
        self.metricName = metricName
        self.unitName = unitName
        self.helpText = helpText
    }

    /// The fully qualified metric name, joining non-empty components with underscores (e.g. `namespace_subsytem_metricName_unitName`).
    public var name: String {
        [namespace, subsystem, metricName, unitName]
            .compactMap { $0?.isEmpty == false ? $0 : nil }
            .joined(separator: "_")
    }
}
