use marksman_escape::Escape;
use regex::Regex;
use std::collections::BTreeMap;
use std::fs::File;
use std::io::{BufWriter, Write};
use std::path::Path;

use crate::snake_to_camel;

const PG_TYPE_H: &str = include_str!("pg_type.h");
const PG_RANGE_H: &str = include_str!("pg_range.h");

struct Type {
    name: &'static str,
    variant: String,
    ident: String,
    kind: &'static str,
    element: u32,
    doc: String,
}

pub fn build(path: &Path) {
    let mut file = BufWriter::new(File::create(path.join("types/type_gen.rs")).unwrap());

    let ranges = parse_ranges();
    let types = parse_types(&ranges);

    make_header(&mut file);
    make_enum(&mut file, &types);
    make_impl(&mut file, &types);
    make_consts(&mut file, &types);
}

fn parse_ranges() -> BTreeMap<u32, u32> {
    let mut ranges = BTreeMap::new();

    for line in PG_RANGE_H.lines() {
        if !line.starts_with("DATA") {
            continue;
        }

        let split = line.split_whitespace().collect::<Vec<_>>();

        let oid = split[2].parse().unwrap();
        let element = split[3].parse().unwrap();

        ranges.insert(oid, element);
    }

    ranges
}

fn parse_types(ranges: &BTreeMap<u32, u32>) -> BTreeMap<u32, Type> {
    let doc_re = Regex::new(r#"DESCR\("([^"]+)"\)"#).unwrap();
    let range_vector_re = Regex::new("(range|vector)$").unwrap();
    let array_re = Regex::new("^_(.*)").unwrap();

    let mut types = BTreeMap::new();

    let mut lines = PG_TYPE_H.lines().peekable();
    while let Some(line) = lines.next() {
        if !line.starts_with("DATA") {
            continue;
        }

        let split = line.split_whitespace().collect::<Vec<_>>();

        let oid = split[3].parse().unwrap();

        let name = split[5];

        let ident = range_vector_re.replace(name, "_$1");
        let ident = array_re.replace(&ident, "$1_array");
        let variant = snake_to_camel(&ident);
        let ident = ident.to_ascii_uppercase();

        let kind = split[11];

        // we need to be able to pull composite fields and enum variants at runtime
        if kind == "C" || kind == "E" {
            continue;
        }

        let element = if let Some(&element) = ranges.get(&oid) {
            element
        } else {
            split[16].parse().unwrap()
        };

        let doc = array_re.replace(name, "$1[]");
        let mut doc = doc.to_ascii_uppercase();

        let descr = lines
            .peek()
            .and_then(|line| doc_re.captures(line))
            .and_then(|captures| captures.at(1));
        if let Some(descr) = descr {
            doc.push_str(" - ");
            doc.push_str(descr);
        }
        let doc = Escape::new(doc.as_bytes().iter().cloned()).collect();
        let doc = String::from_utf8(doc).unwrap();

        let type_ = Type {
            name,
            variant,
            ident,
            kind,
            element,
            doc,
        };

        types.insert(oid, type_);
    }

    types
}

fn make_header(w: &mut BufWriter<File>) {
    write!(
        w,
        "// Autogenerated file - DO NOT EDIT
use std::sync::Arc;

use crate::types::{{Type, Oid, Kind}};

#[derive(PartialEq, Eq, Debug)]
pub struct Other {{
    pub name: String,
    pub oid: Oid,
    pub kind: Kind,
    pub schema: String,
}}
"
    )
    .unwrap();
}

fn make_enum(w: &mut BufWriter<File>, types: &BTreeMap<u32, Type>) {
    write!(
        w,
        "
#[derive(PartialEq, Eq, Clone, Debug)]
pub enum Inner {{"
    )
    .unwrap();

    for type_ in types.values() {
        write!(
            w,
            "
    {},",
            type_.variant
        )
        .unwrap();
    }

    write!(
        w,
        r"
    Other(Arc<Other>),
}}

"
    )
    .unwrap();
}

fn make_impl(w: &mut BufWriter<File>, types: &BTreeMap<u32, Type>) {
    write!(
        w,
        "impl Inner {{
    pub fn from_oid(oid: Oid) -> Option<Inner> {{
        match oid {{
",
    )
    .unwrap();

    for (oid, type_) in types {
        write!(
            w,
            "            {} => Some(Inner::{}),
",
            oid, type_.variant
        )
        .unwrap();
    }

    write!(
        w,
        "            _ => None,
        }}
    }}

    pub fn oid(&self) -> Oid {{
        match *self {{
",
    )
    .unwrap();

    for (oid, type_) in types {
        write!(
            w,
            "            Inner::{} => {},
",
            type_.variant, oid
        )
        .unwrap();
    }

    write!(
        w,
        "            Inner::Other(ref u) => u.oid,
        }}
    }}

    pub fn kind(&self) -> &Kind {{
        match *self {{
",
    )
    .unwrap();

    for type_ in types.values() {
        let kind = match type_.kind {
            "P" => "Pseudo".to_owned(),
            "A" => format!("Array(Type(Inner::{}))", types[&type_.element].variant),
            "R" => format!("Range(Type(Inner::{}))", types[&type_.element].variant),
            _ => "Simple".to_owned(),
        };

        write!(
            w,
            "            Inner::{} => {{
                &Kind::{}
            }}
",
            type_.variant, kind
        )
        .unwrap();
    }

    write!(
        w,
        r#"            Inner::Other(ref u) => &u.kind,
        }}
    }}

    pub fn name(&self) -> &str {{
        match *self {{
"#,
    )
    .unwrap();

    for type_ in types.values() {
        write!(
            w,
            r#"            Inner::{} => "{}",
"#,
            type_.variant, type_.name
        )
        .unwrap();
    }

    write!(
        w,
        "            Inner::Other(ref u) => &u.name,
        }}
    }}
}}
"
    )
    .unwrap();
}

fn make_consts(w: &mut BufWriter<File>, types: &BTreeMap<u32, Type>) {
    write!(w, "impl Type {{").unwrap();
    for type_ in types.values() {
        write!(
            w,
            "
    /// {docs}
    pub const {ident}: Type = Type(Inner::{variant});
",
            docs = type_.doc,
            ident = type_.ident,
            variant = type_.variant
        )
        .unwrap();
    }

    write!(w, "}}").unwrap();
}
