from __future__ import annotations

from dataclasses import dataclass
from typing import (
    TYPE_CHECKING,
    Any,
    Callable,
    Generic,
    NamedTuple,
    Protocol,
    TypeVar,
    Union,
)

from django.http import HttpRequest
from reactpy.types import ComponentType, Connection, Key
from typing_extensions import ParamSpec

if TYPE_CHECKING:
    from collections.abc import MutableMapping, Sequence

    from django.contrib.auth.models import AbstractUser
    from django.forms import Form, ModelForm

    from reactpy_django.websocket.consumer import ReactpyAsyncWebsocketConsumer


FuncParams = ParamSpec("FuncParams")
Inferred = TypeVar("Inferred")
ConnectionType = Connection[Union["ReactpyAsyncWebsocketConsumer", HttpRequest]]


@dataclass
class Query(Generic[Inferred]):
    """Queries generated by the `use_query` hook."""

    data: Inferred
    loading: bool
    error: Exception | None
    refetch: Callable[[], None]


@dataclass
class Mutation(Generic[FuncParams]):
    """Mutations generated by the `use_mutation` hook."""

    execute: Callable[FuncParams, None]
    loading: bool
    error: Exception | None
    reset: Callable[[], None]

    def __call__(self, *args: FuncParams.args, **kwargs: FuncParams.kwargs) -> None:
        """Execute the mutation."""
        self.execute(*args, **kwargs)


@dataclass
class FormEventData:
    """State of a form provided to Form custom events."""

    form: Form | ModelForm
    submitted_data: dict[str, Any]
    set_submitted_data: Callable[[dict[str, Any] | None], None]


class AsyncFormEvent(Protocol):
    async def __call__(self, event: FormEventData) -> None: ...


class SyncFormEvent(Protocol):
    def __call__(self, event: FormEventData) -> None: ...


class AsyncPostprocessor(Protocol):
    async def __call__(self, data: Any) -> Any: ...


class SyncPostprocessor(Protocol):
    def __call__(self, data: Any) -> Any: ...


@dataclass
class ComponentParams:
    """Container used for serializing component parameters.
    This dataclass is pickled & stored in the database, then unpickled when needed."""

    args: Sequence
    kwargs: MutableMapping[str, Any]


class UserData(NamedTuple):
    query: Query[dict | None]
    mutation: Mutation[dict]


class AsyncMessageReceiver(Protocol):
    async def __call__(self, message: dict) -> None: ...


class AsyncMessageSender(Protocol):
    async def __call__(self, message: dict) -> None: ...


class ViewToComponentConstructor(Protocol):
    def __call__(
        self, request: HttpRequest | None = None, *args: Any, key: Key | None = None, **kwargs: Any
    ) -> ComponentType: ...


class ViewToIframeConstructor(Protocol):
    def __call__(self, *args: Any, key: Key | None = None, **kwargs: Any) -> ComponentType: ...


class UseAuthLogin(Protocol):
    async def __call__(self, user: AbstractUser, rerender: bool = True) -> None: ...


class UseAuthLogout(Protocol):
    async def __call__(self, rerender: bool = True) -> None: ...


class UseAuthTuple(NamedTuple):
    login: UseAuthLogin
    """Login a user.

        Args:
            user: The user to login.
            rerender: If True, the root component will be re-rendered after the user is logged in."""

    logout: UseAuthLogout
    """Logout the current user.

        Args:
            rerender: If True, the root component will be re-rendered after the user is logged out."""
