//#import <Foundation/Foundation.h>

#include <sched.h>
#include <sys/param.h>
#include <sys/stat.h>

#include "common.h"
#include "kern_utils.h"
#include "helpers/kexecute.h"
#include "helpers/kmem.h"
#include "helpers/offsetof.h"
#include "helpers/osobject.h"
#include "sandbox.h"

mach_port_t tfp0;
uint64_t kernel_base;
uint64_t kernel_slide;

uint64_t kernprocaddr;
uint64_t offset_zonemap;

uint64_t offset_add_ret_gadget;
uint64_t offset_osboolean_true;
uint64_t offset_osboolean_false;
uint64_t offset_osunserializexml;
uint64_t offset_smalloc;

// Please call `proc_release` after you are finished with your proc!
uint64_t proc_find(int pid) {
    uint64_t proc = kernprocaddr;
    
    while (proc) {
        uint32_t found_pid = rk32(proc + 0x10);
        
        if (found_pid == pid) {
            return proc;
        }
        
        proc = rk64(proc + 0x8);
    }
    
    return 0;
}

CACHED_FIND(uint64_t, our_task_addr) {
    uint64_t proc = rk64(kernprocaddr + 0x8);
    
    while (proc) {
        uint32_t proc_pid = rk32(proc + 0x10);
        
        if (proc_pid == getpid()) {
            break;
        }
        
        proc = rk64(proc + 0x8);
    }
    
    if (proc == 0) {
        DEBUGLOG("failed to find our_task_addr!");
        exit(EXIT_FAILURE);
    }

    return rk64(proc + offsetof_task);
}

uint64_t find_port(mach_port_name_t port) {
    uint64_t task_addr = our_task_addr();
  
    uint64_t itk_space = rk64(task_addr + offsetof_itk_space);
  
    uint64_t is_table = rk64(itk_space + offsetof_ipc_space_is_table);
  
    uint32_t port_index = port >> 8;
    const int sizeof_ipc_entry_t = 0x18;
  
    return rk64(is_table + (port_index * sizeof_ipc_entry_t));
}

void fixup_setuid(int pid, uint64_t proc) {
    char pathbuf[PROC_PIDPATHINFO_MAXSIZE];
    bzero(pathbuf, sizeof(pathbuf));
    
    int ret = proc_pidpath(pid, pathbuf, sizeof(pathbuf));
    if (ret < 0) {
        DEBUGLOG("Unable to get path for PID %d", pid);
        return;
    }
    
    struct stat file_st;
    if (lstat(pathbuf, &file_st) == -1) {
        DEBUGLOG("Unable to get stat for file %s", pathbuf);
        return;
    }
    
    if (!(file_st.st_mode & S_ISUID) && !(file_st.st_mode & S_ISGID)) {
        DEBUGLOG("File is not setuid or setgid: %s", pathbuf);
        return;
    }
    
    if (proc == 0) {
        DEBUGLOG("Invalid proc for pid %d", pid);
        return;
    }
    
    DEBUGLOG("Found proc %llx for pid %d", proc, pid);
    
    uid_t fileUid = file_st.st_uid;
    uid_t fileGid = file_st.st_gid;
    
    DEBUGLOG("Applying UID %d to process %d", fileUid, pid);
    uint64_t ucred = rk64(proc + offsetof_p_ucred);
    
    if (file_st.st_mode & S_ISUID) {
        wk32(proc + offsetof_p_svuid, fileUid);
        wk32(ucred + offsetof_ucred_cr_svuid, fileUid);
        wk32(ucred + offsetof_ucred_cr_uid, fileUid);
    }

    if (file_st.st_mode & S_ISGID) {
        wk32(proc + offsetof_p_svgid, fileGid);
        wk32(ucred + offsetof_ucred_cr_svgid, fileGid);
        wk32(ucred + offsetof_ucred_cr_groups, fileGid);
    }
}

void set_tfplatform(uint64_t proc) {
    // task.t_flags & TF_PLATFORM
    uint64_t task = rk64(proc + offsetof_task);
    uint32_t t_flags = rk32(task + offsetof_t_flags);
    t_flags |= TF_PLATFORM;
    wk32(task+offsetof_t_flags, t_flags);
}

const char* abs_path_exceptions[] = {
    "/Library",
    "/private/var/mobile/Library",
    NULL
};

uint64_t exception_osarray_cache = 0;
uint64_t get_exception_osarray(void) {
    if (exception_osarray_cache == 0) {
        exception_osarray_cache = OSUnserializeXML(
            "<array>"
            "<string>/Library/</string>"
            "<string>/private/var/mobile/Library/</string>"
            "</array>"
        );
    }

    return exception_osarray_cache;
}

static const char *exc_key = "com.apple.security.exception.files.absolute-path.read-only";

void set_sandbox_extensions(uint64_t proc) {
    uint64_t proc_ucred = rk64(proc + 0x100);
    uint64_t sandbox = rk64(rk64(proc_ucred + 0x78) + 0x8 + 0x8);
    
    if (sandbox == 0) {
        DEBUGLOG("no sandbox, skipping (proc: %llx)", proc);
        return;
    }

    if (has_file_extension(sandbox, abs_path_exceptions[0])) {
        DEBUGLOG("already has '%s', skipping", abs_path_exceptions[0]);
        return;
    }

    uint64_t ext = 0;
    const char** path = abs_path_exceptions;
    while (*path != NULL) {
        ext = extension_create_file(*path, ext);
        if (ext == 0) {
            DEBUGLOG("extension_create_file(%s) failed, panic!", *path);
        }
        ++path;
    }
    
    if (ext != 0) {
        extension_add(ext, sandbox, exc_key);
    }
}

void set_amfi_entitlements(uint64_t proc) {
    uint64_t proc_ucred = rk64(proc + 0x100);
    uint64_t amfi_entitlements = rk64(rk64(proc_ucred + 0x78) + 0x8);

    int rv = 0;
    
    uint64_t present = OSDictionary_GetItem(amfi_entitlements, exc_key);

    if (present == 0) {
        DEBUGLOG("present=0; setting to %llx", get_exception_osarray());
        rv = OSDictionary_SetItem(amfi_entitlements, exc_key, get_exception_osarray());
    } else if (present != get_exception_osarray()) {
        unsigned int itemCount = OSArray_ItemCount(present);
        DEBUGLOG("got item count: %d", itemCount);

        Boolean foundEntitlements = false;

        uint64_t itemBuffer = OSArray_ItemBuffer(present);

        for (int i = 0; i < itemCount; i++) {
            uint64_t item = rk64(itemBuffer + (i * sizeof(void *)));
            char *entitlementString = OSString_CopyString(item);
            DEBUGLOG("found ent string: %s", entitlementString);
            if (strcmp(entitlementString, "/Library/") == 0) {
                foundEntitlements = true;
                free(entitlementString);
                break;
            }
            free(entitlementString);
        }

        if (!foundEntitlements){
            rv = OSArray_Merge(present, get_exception_osarray());
        } else {
            rv = 1;
        }
    } else {
        rv = 1;
    }

    if (rv != 1) {
        DEBUGLOG("Setting exc FAILED! amfi_entitlements: 0x%llx present: 0x%llx", amfi_entitlements, present);
    }
}

void fixup_tfplatform(uint64_t proc) {
    uint64_t proc_ucred = rk64(proc + 0x100);
    uint64_t amfi_entitlements = rk64(rk64(proc_ucred + 0x78) + 0x8);

    uint64_t key = OSDictionary_GetItem(amfi_entitlements, "platform-application");
    if (key == offset_osboolean_true) {
        DEBUGLOG("platform-application is set");
        set_tfplatform(proc);

        uint32_t csflags = rk32(proc + offsetof_p_csflags);
        csflags |= CS_PLATFORM_BINARY;
        wk32(proc + offsetof_p_csflags, csflags);
    } else {
        DEBUGLOG("platform-application is not set");
    }
}

void fixup_sandbox(uint64_t proc) {
    set_sandbox_extensions(proc);
}

void fixup_cs_valid(uint64_t proc) {
    uint32_t csflags = rk32(proc + offsetof_p_csflags);

    csflags |= CS_VALID;
    
    wk32(proc + offsetof_p_csflags, csflags);
}

void fixup(int pid) {
    uint64_t proc = proc_find(pid);
    if (proc == 0) {
        DEBUGLOG("failed to find proc for pid %d!", pid);
        return;
    }

    DEBUGLOG("fixup_setuid");
    fixup_setuid(pid, proc);
    DEBUGLOG("fixup_sandbox");
    fixup_sandbox(proc);
    DEBUGLOG("fixup_tfplatform");
    fixup_tfplatform(proc);
    DEBUGLOG("set_amfi_entitlements");
    set_amfi_entitlements(proc);
}
