// kind of hand-written fuzzing data
// any errors should not break Encoder/Decoder instance states
import assert from "assert";
import { encode, decode, Encoder, Decoder } from "../src";
import { DataViewIndexOutOfBoundsError } from "../src/Decoder";

function testEncoder(encoder: Encoder): void {
  const object = {
    foo: 1,
    bar: 2,
    baz: ["one", "two", "three"],
  };
  assert.deepStrictEqual(decode(encoder.encode(object)), object);
}

function testDecoder(decoder: Decoder): void {
  const object = {
    foo: 1,
    bar: 2,
    baz: ["one", "two", "three"],
  };
  assert.deepStrictEqual(decoder.decode(encode(object)), object);
}

describe("edge cases", () => {
  context("try to encode cyclic refs", () => {
    it("throws errors on arrays", () => {
      const encoder = new Encoder();
      const cyclicRefs: Array<any> = [];
      cyclicRefs.push(cyclicRefs);
      assert.throws(() => {
        encoder.encode(cyclicRefs);
      }, /too deep/i);
      testEncoder(encoder);
    });

    it("throws errors on objects", () => {
      const encoder = new Encoder();
      const cyclicRefs: Record<string, any> = {};
      cyclicRefs["foo"] = cyclicRefs;
      assert.throws(() => {
        encoder.encode(cyclicRefs);
      }, /too deep/i);
      testEncoder(encoder);
    });
  });

  context("try to encode unrecognized objects", () => {
    it("throws errors", () => {
      const encoder = new Encoder();
      assert.throws(() => {
        encode(() => {});
      }, /unrecognized object/i);
      testEncoder(encoder);
    });
  });

  context("try to decode a map with non-string keys (asynchronous)", () => {
    it("throws errors", async () => {
      const decoder = new Decoder();
      const createStream = async function* () {
        yield [0x81]; // fixmap size=1
        yield encode(null);
        yield encode(null);
      };

      await assert.rejects(async () => {
        await decoder.decodeAsync(createStream());
      }, /The type of key must be string/i);
      testDecoder(decoder);
    });
  });

  context("try to decode invalid MessagePack binary", () => {
    it("throws errors", () => {
      const decoder = new Decoder();
      const TYPE_NEVER_USED = 0xc1;

      assert.throws(() => {
        decoder.decode([TYPE_NEVER_USED]);
      }, /unrecognized type byte/i);
      testDecoder(decoder);
    });
  });

  context("try to decode insufficient data", () => {
    it("throws errors (synchronous)", () => {
      const decoder = new Decoder();
      assert.throws(() => {
        decoder.decode([
          0x92, // fixarray size=2
          0xc0, // nil
        ]);
        // [IE11] A raw error thrown by DataView
      }, DataViewIndexOutOfBoundsError);
      testDecoder(decoder);
    });

    it("throws errors (asynchronous)", async () => {
      const decoder = new Decoder();
      const createStream = async function* () {
        yield [0x92]; // fixarray size=2
        yield encode(null);
      };

      await assert.rejects(async () => {
        await decoder.decodeAsync(createStream());
      }, RangeError);
      testDecoder(decoder);
    });
  });

  context("try to decode data with extra bytes", () => {
    it("throws errors (synchronous)", () => {
      const decoder = new Decoder();
      assert.throws(() => {
        decoder.decode([
          0x90, // fixarray size=0
          ...encode(null),
        ]);
      }, RangeError);
      testDecoder(decoder);
    });

    it("throws errors (asynchronous)", async () => {
      const decoder = new Decoder();
      const createStream = async function* () {
        yield [0x90]; // fixarray size=0
        yield encode(null);
      };

      await assert.rejects(async () => {
        await decoder.decodeAsync(createStream());
      }, RangeError);
      testDecoder(decoder);
    });

    it("throws errors (asynchronous)", async () => {
      const decoder = new Decoder();
      const createStream = async function* () {
        yield [0x90, ...encode(null)]; // fixarray size=0 + nil
      };

      await assert.rejects(async () => {
        await decoder.decodeAsync(createStream());
      }, RangeError);
      testDecoder(decoder);
    });
  });
});
