import Combine
import Foundation
import MapboxDirections
import MapboxMaps
@testable import MapboxNavigationCore
@testable import MapboxNavigationUIKit
import Turf
import UIKit

private enum Constants {
    static let dotSize = CGSize(width: 15, height: 15)
}

extension UIColor {
    public class var route: UIColor { return #colorLiteral(red: 0.00, green: 0.70, blue: 0.99, alpha: 1.0) }
    public class var routeCoordinate: UIColor { return #colorLiteral(red: 0, green: 1, blue: 0.99, alpha: 0.3012764085) }
    private class var rawLocation: UIColor { return #colorLiteral(red: 0.9254902005, green: 0.2352941185, blue: 0.1019607857, alpha: 1) }
    private class var snappedLocation: UIColor { return #colorLiteral(red: 0.5843137503, green: 0.8235294223, blue: 0.4196078479, alpha: 1) }
}

@MainActor
public protocol Plotter {
    var color: UIColor { get }
    var drawIndexesAsText: Bool { get }
    func draw(on plotter: NavigationPlotter)
}

@MainActor
public struct CoordinatePlotter: Plotter {
    public let coordinates: [CLLocationCoordinate2D]
    public let color: UIColor
    public let drawIndexesAsText: Bool

    public init(coordinates: [CLLocationCoordinate2D], color: UIColor, drawIndexesAsText: Bool) {
        self.coordinates = coordinates
        self.color = color
        self.drawIndexesAsText = drawIndexesAsText
    }
}

@MainActor
public struct LocationPlotter: Plotter {
    public let locations: [CLLocation]
    public let color: UIColor
    public let drawIndexesAsText: Bool

    public init(locations: [CLLocation], color: UIColor, drawIndexesAsText: Bool) {
        self.locations = locations
        self.color = color
        self.drawIndexesAsText = drawIndexesAsText
    }
}

@MainActor
public struct LinePlotter: Plotter {
    public let coordinates: [CLLocationCoordinate2D]
    public let color: UIColor
    public let lineWidth: CGFloat
    public let drawIndexesAsText: Bool

    public init(coordinates: [CLLocationCoordinate2D], color: UIColor, lineWidth: CGFloat, drawIndexesAsText: Bool) {
        self.coordinates = coordinates
        self.color = color
        self.lineWidth = lineWidth
        self.drawIndexesAsText = drawIndexesAsText
    }
}

@MainActor
public struct RoutePlotter: Plotter {
    public let route: Route
    public let color: UIColor
    public let lineWidth: CGFloat
    public let drawIndexesAsText: Bool
    public let drawDotIndicator: Bool
    public let drawTextIndicator: Bool

    public init(
        route: Route,
        color: UIColor = UIColor.route,
        lineWidth: CGFloat = 4,
        drawIndexesAsText: Bool = false,
        drawDotIndicator: Bool = true,
        drawTextIndicator: Bool = true
    ) {
        self.route = route
        self.color = color
        self.lineWidth = lineWidth
        self.drawIndexesAsText = drawIndexesAsText
        self.drawDotIndicator = drawDotIndicator
        self.drawTextIndicator = drawTextIndicator
    }
}

@MainActor
public struct MatchPlotter: Plotter {
    public let match: Match
    public let color: UIColor
    public let lineWidth: CGFloat
    public let drawIndexesAsText: Bool
    public let drawDotIndicator: Bool
    public let drawTextIndicator: Bool

    public init(
        match: Match,
        color: UIColor = UIColor.route,
        lineWidth: CGFloat = 4,
        drawIndexesAsText: Bool = false,
        drawDotIndicator: Bool = true,
        drawTextIndicator: Bool = true
    ) {
        self.match = match
        self.color = color
        self.lineWidth = lineWidth
        self.drawIndexesAsText = drawIndexesAsText
        self.drawDotIndicator = drawDotIndicator
        self.drawTextIndicator = drawTextIndicator
    }
}

extension RoutePlotter {
    @MainActor
    public func draw(on plotter: NavigationPlotter) {
        plotter.drawLines(
            between: route.shape!.coordinates,
            color: color,
            lineWidth: lineWidth,
            drawDotIndicator: drawDotIndicator,
            drawTextIndicator: drawTextIndicator
        )
    }
}

extension MatchPlotter {
    @MainActor
    public func draw(on plotter: NavigationPlotter) {
        plotter.drawLines(
            between: match.shape!.coordinates,
            color: color,
            lineWidth: lineWidth,
            drawDotIndicator: drawDotIndicator,
            drawTextIndicator: drawTextIndicator
        )
    }
}

extension CoordinatePlotter {
    @MainActor
    public func draw(on plotter: NavigationPlotter) {
        guard let mapView = plotter.navigationMapView?.mapView else { return }

        for (i, coordinate) in coordinates.enumerated() {
            let position = mapView.mapboxMap.point(for: coordinate)
            let centeredPosition = CGPoint(
                x: position.x - Constants.dotSize.width / 2,
                y: position.y - Constants.dotSize.height / 2
            )
            plotter.drawDot(at: centeredPosition, color: color)

            if drawIndexesAsText {
                plotter.drawText(at: centeredPosition, text: "\(i)")
            }
        }
    }
}

extension LocationPlotter {
    @MainActor
    public func draw(on plotter: NavigationPlotter) {
        guard let mapView = plotter.navigationMapView?.mapView else { return }

        for (i, location) in locations.enumerated() {
            let position = mapView.mapboxMap.point(for: location.coordinate)
            let centeredPosition = CGPoint(
                x: position.x - Constants.dotSize.width / 2,
                y: position.y - Constants.dotSize.height / 2
            )
            plotter.drawDot(at: centeredPosition, color: color)
            plotter.drawCourseIndicator(at: centeredPosition, course: location.course)

            if drawIndexesAsText {
                plotter.drawText(at: centeredPosition, text: "\(i)")
            }
        }
    }
}

extension LinePlotter {
    @MainActor
    public func draw(on plotter: NavigationPlotter) {
        plotter.drawLines(
            between: coordinates,
            color: color,
            lineWidth: lineWidth,
            drawDotIndicator: false,
            drawTextIndicator: false
        )
    }
}

@MainActor
public class NavigationPlotter: UIView {
    var navigationMapView: NavigationMapView?
    var coordinateBounds: CoordinateBounds?
    public var routePlotters: [RoutePlotter]? { didSet { setNeedsDisplay() } }
    public var matchPlotters: [MatchPlotter]? { didSet { setNeedsDisplay() } }
    public var coordinatePlotters: [CoordinatePlotter]? { didSet { setNeedsDisplay() } }
    public var locationPlotters: [LocationPlotter]? { didSet { setNeedsDisplay() } }
    public var linePlotters: [LinePlotter]? { didSet { setNeedsDisplay() } }

    public var locationPublisher: PassthroughSubject<CLLocation, Never> = .init()
    public var routeProgressPublisher: CurrentValueSubject<RouteProgress?, Never> = .init(nil)

    func updateCoordinateBounds() {
        let navigationMapView = NavigationMapView(
            location: locationPublisher.eraseToAnyPublisher(),
            routeProgress: routeProgressPublisher.eraseToAnyPublisher()
        )
        navigationMapView.mapView.frame = bounds
        navigationMapView.frame = bounds
        self.navigationMapView = navigationMapView
        let padding = UIEdgeInsets(top: 50, left: 50, bottom: 50, right: 50)

        coordinateBounds = allBoundingCoordinates.bounds

        if let coordinateBounds {
            let cameraOptions = navigationMapView.mapView.mapboxMap.camera(
                for: coordinateBounds, padding: padding, bearing: nil, pitch: nil, maxZoom: nil, offset: nil
            )
            navigationMapView.mapView.mapboxMap.setCamera(to: cameraOptions)
        }
    }

    override public init(frame: CGRect) {
        super.init(frame: frame)
    }

    @available(*, unavailable)
    public required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }

    var allBoundingCoordinates: [CLLocationCoordinate2D] {
        var coordinates = [CLLocationCoordinate2D]()

        routePlotters?.forEach { plotter in
            coordinates += plotter.route.shape!.coordinates
        }

        matchPlotters?.forEach { plotter in
            coordinates += plotter.match.shape!.coordinates
        }

        coordinatePlotters?.forEach { plotter in
            coordinates += plotter.coordinates
        }

        locationPlotters?.forEach { plotter in
            coordinates += plotter.locations.map { $0.coordinate }
        }

        linePlotters?.forEach { plotter in
            coordinates += plotter.coordinates
        }

        return coordinates
    }

    override public func draw(_ rect: CGRect) {
        super.draw(rect)
        updateCoordinateBounds()

        routePlotters?.forEach { $0.draw(on: self) }
        matchPlotters?.forEach { $0.draw(on: self) }
        linePlotters?.forEach { $0.draw(on: self) }
        coordinatePlotters?.forEach { $0.draw(on: self) }
        locationPlotters?.forEach { $0.draw(on: self) }
    }

    func drawLines(
        between coordinates: [CLLocationCoordinate2D]?,
        color: UIColor = UIColor.route,
        lineWidth: CGFloat = 4,
        drawDotIndicator: Bool = true,
        drawTextIndicator: Bool = true
    ) {
        guard let coordinates, let navigationMapView else { return }
        let path = UIBezierPath()
        for coordinate in coordinates {
            let position = navigationMapView.mapView.mapboxMap.point(for: coordinate)
            if coordinate == coordinates.first {
                path.move(to: navigationMapView.mapView.mapboxMap.point(for: coordinates.first!))
            } else {
                path.addLine(to: position)
            }
        }

        color.setStroke()
        path.lineWidth = lineWidth
        path.stroke()

        for (i, coordinate) in coordinates.enumerated() {
            let position = navigationMapView.mapView.mapboxMap.point(for: coordinate)
            let centeredPosition = CGPoint(
                x: position.x - Constants.dotSize.width / 2,
                y: position.y - Constants.dotSize.height / 2
            )
            if drawDotIndicator {
                drawDot(at: centeredPosition, color: .routeCoordinate)
            }
            if drawTextIndicator {
                drawText(at: position, text: "\(i)")
            }
        }
    }
}

extension UIView {
    fileprivate func drawDot(at point: CGPoint, color: UIColor) {
        let path = UIBezierPath(ovalIn: CGRect(origin: point, size: Constants.dotSize))
        color.setFill()
        path.fill()
        UIColor.white.setStroke()
        path.lineWidth = 2
        path.stroke()
    }

    fileprivate func drawCourseIndicator(at point: CGPoint, course: CLLocationDirection) {
        let path = UIBezierPath()

        let angle = CGFloat(CLLocationDirection(270 - course).toRadians())

        let centerPoint = CGPoint(
            x: point.x + Constants.dotSize.midWidth,
            y: point.y + Constants.dotSize.midHeight
        )
        let startPoint = CGPoint(
            x: centerPoint.x + sin(angle - .pi / 2) * Constants.dotSize.midWidth,
            y: centerPoint.y + cos(angle - .pi / 2) * Constants.dotSize.midHeight
        )
        path.move(to: startPoint)
        path.addLine(to: CGPoint(
            x: startPoint.x + sin(angle - .pi / 2) * Constants.dotSize.midWidth,
            y: startPoint.y + cos(angle - .pi / 2) * Constants.dotSize.midHeight
        ))

        UIColor.white.setStroke()
        path.lineWidth = 2
        path.stroke()
    }

    fileprivate func drawText(at point: CGPoint, text: String) {
        let context = UIGraphicsGetCurrentContext()!
        let textRect = CGRect(origin: point, size: CGSize(width: 50, height: 20))
        let textStyle = NSMutableParagraphStyle()
        textStyle.alignment = .left

#if swift(>=4.2)
        let attributes: [NSAttributedString.Key: Any]
#else
        let attributes: [NSAttributedStringKey: Any]
#endif

        attributes = [
            .font: UIFont.systemFont(ofSize: 9, weight: .medium),
            .foregroundColor: UIColor.white,
            .paragraphStyle: textStyle,
            .strokeColor: UIColor.black,
            .strokeWidth: -1,
        ]

        let boundingRect = text.boundingRect(
            with: CGSize(width: textRect.width, height: CGFloat.infinity),
            options: .usesLineFragmentOrigin,
            attributes: attributes,
            context: nil
        )

        context.saveGState()
        let rect = CGRect(
            x: point.x - boundingRect.midX + Constants.dotSize.width / 2,
            y: point.y - boundingRect.midY + Constants.dotSize.height / 2,
            width: boundingRect.width,
            height: boundingRect.height
        )
        text.draw(in: rect, withAttributes: attributes)
        context.restoreGState()
    }

    private func drawCourseText(at point: CGPoint, course: CLLocationDirection) {
        let context = UIGraphicsGetCurrentContext()!
        let textRect = CGRect(origin: point, size: CGSize(width: 50, height: 20))
        let text = "\(Int(round(course)))"
        let textStyle = NSMutableParagraphStyle()
        textStyle.alignment = .left

#if swift(>=4.2)
        let attributes: [NSAttributedString.Key: Any]
#else
        let attributes: [NSAttributedStringKey: Any]
#endif

        attributes = [
            .font: UIFont.systemFont(ofSize: 7, weight: .medium),
            .foregroundColor: UIColor.white,
            .paragraphStyle: textStyle,
            .strokeColor: UIColor.black,
            .strokeWidth: -1,
        ]

        let height: CGFloat = text.boundingRect(
            with: CGSize(width: textRect.width, height: CGFloat.infinity),
            options: .usesLineFragmentOrigin,
            attributes: attributes,
            context: nil
        )
        .height
        context.saveGState()
        context.clip(to: textRect)
        let rect = CGRect(
            x: textRect.minX,
            y: textRect.minY + (textRect.height - height) / 2,
            width: textRect.width,
            height: height
        )
        text.draw(in: rect, withAttributes: attributes)
        context.restoreGState()
    }
}

extension [CLLocationCoordinate2D] {
    fileprivate var bounds: CoordinateBounds {
        var maximumLatitude: CLLocationDegrees = -80
        var minimumLatitude: CLLocationDegrees = 80
        var maximumLongitude: CLLocationDegrees = -180
        var minimumLongitude: CLLocationDegrees = 180

        for coordinate in self {
            maximumLatitude = Swift.max(maximumLatitude, coordinate.latitude)
            minimumLatitude = Swift.min(minimumLatitude, coordinate.latitude)
            maximumLongitude = Swift.max(maximumLongitude, coordinate.longitude)
            minimumLongitude = Swift.min(minimumLongitude, coordinate.longitude)
        }

        let southwest = CLLocationCoordinate2D(latitude: minimumLatitude, longitude: minimumLongitude)
        let northeast = CLLocationCoordinate2D(latitude: maximumLatitude, longitude: maximumLongitude)

        return CoordinateBounds(southwest: southwest, northeast: northeast)
    }
}

extension CGSize {
    var midWidth: CGFloat { return width / 2 }
    var midHeight: CGFloat { return height / 2 }
}
