import CoreLocation
import Foundation
import MapboxDirections
import MapboxMaps

private let trafficTileSetIdentifiers = Set([
    "mapbox.mapbox-traffic-v1",
    "mapbox.mapbox-traffic-v2-beta",
])

private let incidentsTileSetIdentifiers = Set([
    "mapbox.mapbox-incidents-v1",
    "mapbox.mapbox-incidents-v2-beta",
])

/// An extension on `MapView` that allows for toggling traffic on a map style that contains a [Mapbox Traffic
/// source](https://docs.mapbox.com/vector-tiles/mapbox-traffic-v1/).
extension MapView {
    /// Returns a set of tile set identifiers for specific `sourceIdentifier`.
    ///
    /// - parameter sourceIdentifier: Identifier of the source, which will be searched for in current style of the
    /// ///`MapView`.
    /// - returns: Set of tile set identifiers.
    func tileSetIdentifiers(_ sourceIdentifier: String) -> Set<String> {
        if let properties = try? mapboxMap.sourceProperties(for: sourceIdentifier),
           let url = properties["url"] as? String,
           let configurationURL = URL(string: url),
           configurationURL.scheme == "mapbox",
           let tileSetIdentifiers = configurationURL.host?.components(separatedBy: ",")
        {
            return Set(tileSetIdentifiers)
        }

        return Set()
    }

    /// Returns a list of identifiers of the tile sets that make up specific source type.
    ///
    /// This array contains multiple entries for a composited source. This property is empty for non-Mapbox-hosted tile
    /// sets and sources with type other than `vector`.
    ///
    /// - parameter sourceIdentifier: Identifier of the source.
    /// - parameter sourceType: Type of the source (e.g. `vector`).
    /// - returns: List of tile set identifiers.
    func tileSetIdentifiers(_ sourceIdentifier: String, sourceType: String) -> [String] {
        if sourceType == "vector" {
            return Array(tileSetIdentifiers(sourceIdentifier))
        }

        return []
    }

    /// Returns a set of source identifiers for tilesets that are or include the given source.
    ///
    /// - parameter tileSetIdentifier: Identifier of the tile set in the form `user.tileset`.
    /// - returns: Set of source identifiers.
    func sourceIdentifiers(_ tileSetIdentifiers: Set<String>) -> Set<String> {
        return Set(mapboxMap.allSourceIdentifiers.filter {
            $0.type.rawValue == "vector"
        }.filter {
            !self.tileSetIdentifiers($0.id).isDisjoint(with: tileSetIdentifiers)
        }.map(\.id))
    }

    /// Returns a Boolean value indicating whether data from the given tile set layers is currently all visible in the
    /// map view’s style.
    ///
    /// - parameter tileSetIdentifiers: Identifiers of the tile sets in the form `user.tileset`.
    /// - parameter layerIdentifier: Identifier of the layer in the tile set; in other words, a source layer identifier.
    /// Not to be confused with a style layer.
    func showsTileSet(with tileSetIdentifiers: Set<String>, layerIdentifier: String) -> Bool {
        let sourceIdentifiers = sourceIdentifiers(tileSetIdentifiers)
        var foundTileSets = false

        for mapViewLayerIdentifier in mapboxMap.allLayerIdentifiers.map(\.id) {
            guard let sourceIdentifier = mapboxMap.layerProperty(
                for: mapViewLayerIdentifier,
                property: "source"
            ).value as? String,
                let sourceLayerIdentifier = mapboxMap.layerProperty(
                    for: mapViewLayerIdentifier,
                    property: "source-layer"
                ).value as? String
            else { return false }

            if sourceIdentifiers.contains(sourceIdentifier), sourceLayerIdentifier == layerIdentifier {
                foundTileSets = true
                let visibility = mapboxMap.layerProperty(for: mapViewLayerIdentifier, property: "visibility")
                    .value as? String
                if visibility != "visible" {
                    return false
                }
            }
        }

        return foundTileSets
    }

    /// Shows or hides data from the given tile set layers.
    ///
    /// - parameter isVisible: Parameter, which controls whether layer should be visible or not.
    /// - parameter tileSetIdentifiers: Identifiers of the tile sets in the form `user.tileset`.
    /// - parameter layerIdentifier: Identifier of the layer in the tile set; in other words, a source layer identifier.
    /// Not to be confused with a style layer.
    func setShowsTileSet(_ isVisible: Bool, with tileSetIdentifiers: Set<String>, layerIdentifier: String) {
        let sourceIdentifiers = sourceIdentifiers(tileSetIdentifiers)

        for mapViewLayerIdentifier in mapboxMap.allLayerIdentifiers.map(\.id) {
            guard let sourceIdentifier = mapboxMap.layerProperty(
                for: mapViewLayerIdentifier,
                property: "source"
            ).value as? String,
                let sourceLayerIdentifier = mapboxMap.layerProperty(
                    for: mapViewLayerIdentifier,
                    property: "source-layer"
                ).value as? String
            else { return }

            if sourceIdentifiers.contains(sourceIdentifier), sourceLayerIdentifier == layerIdentifier {
                let properties = [
                    "visibility": isVisible ? "visible" : "none",
                ]
                try? mapboxMap.setLayerProperties(for: mapViewLayerIdentifier, properties: properties)
            }
        }
    }

    /// A Boolean value indicating whether traffic congestion lines are visible in the map view’s style.
    var showsTraffic: Bool {
        get {
            return showsTileSet(with: trafficTileSetIdentifiers, layerIdentifier: "traffic")
        }
        set {
            setShowsTileSet(newValue, with: trafficTileSetIdentifiers, layerIdentifier: "traffic")
        }
    }

    /// A Boolean value indicating whether incidents, such as road closures and detours, are visible in the map view’s
    /// style.
    var showsIncidents: Bool {
        get {
            return showsTileSet(with: incidentsTileSetIdentifiers, layerIdentifier: "closures")
        }
        set {
            setShowsTileSet(newValue, with: incidentsTileSetIdentifiers, layerIdentifier: "closures")
        }
    }

    /// Method, which returns list of source identifiers, which contain streets tile set.
    func streetsSources() -> [SourceInfo] {
        return mapboxMap.allSourceIdentifiers.filter {
            let identifiers = tileSetIdentifiers($0.id, sourceType: $0.type.rawValue)
            return VectorSource.isMapboxStreets(identifiers)
        }
    }

    /// Attempts to localize road labels into the local language and other labels into the given locale.
    func localizeLabels(into locale: Locale) {
        guard let mapboxStreetsSource = streetsSources().first else { return }

        let streetsSourceTilesetIdentifiers = tileSetIdentifiers(mapboxStreetsSource.id)
        let roadLabelSourceLayerIdentifier = streetsSourceTilesetIdentifiers
            .compactMap { VectorSource.roadLabelLayerIdentifiersByTileSetIdentifier[$0]
            }.first

        let localizableLayerIdentifiers = mapboxMap.allLayerIdentifiers.lazy
            .filter {
                $0.type == .symbol
            }
            // We only know how to localize layers backed by the Mapbox Streets source.
            .filter {
                self.mapboxMap.layerProperty(for: $0.id, property: "source").value as? String == mapboxStreetsSource.id
            }
            // Road labels should match road signage, so they should not be localized.
            // TODO: Actively delocalize road labels into the “name” property: https://github.com/mapbox/mapbox-maps-ios/issues/653
            .filter {
                self.mapboxMap.layerProperty(
                    for: $0.id,
                    property: "source-layer"
                ).value as? String != roadLabelSourceLayerIdentifier
            }
            .map(\.id)
        try? mapboxMap.localizeLabels(into: locale, forLayerIds: Array(localizableLayerIdentifiers))
    }
}

extension MapView {
    /// Returns a tileset descriptor for current map style.
    ///
    /// - parameter zoomRange: Closed range zoom level for the tile package.
    /// - returns: A tileset descriptor.
    func tilesetDescriptor(zoomRange: ClosedRange<UInt8>) -> TilesetDescriptor? {
        guard let styleURI = mapboxMap.styleURI,
              URL(string: styleURI.rawValue)?.scheme == "mapbox"
        else { return nil }

        let offlineManager = OfflineManager()
        let tilesetDescriptorOptions = TilesetDescriptorOptions(
            styleURI: styleURI,
            zoomRange: zoomRange,
            tilesets: nil
        )
        return offlineManager.createTilesetDescriptor(for: tilesetDescriptorOptions)
    }
}
