import MapboxMaps
import MapboxNavigationCore

/// The ``CarPlayMapViewControllerDelegate`` protocol provides methods for reacting to events during free-drive
/// navigation or route previewing in ``CarPlayMapViewController``.
public protocol CarPlayMapViewControllerDelegate: AnyObject, UnimplementedLogging {
    /// Tells the receiver that the final destination `PointAnnotation` was added to the ``CarPlayMapViewController``.
    /// - Parameters:
    ///   - carPlayMapViewController: The ``CarPlayMapViewController`` object.
    ///   - finalDestinationAnnotation: The point annotation that was added to the map view.
    ///   - pointAnnotationManager: The object that manages the point annotation in the map view.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        didAdd finalDestinationAnnotation: PointAnnotation,
        pointAnnotationManager: PointAnnotationManager
    )

    /// Asks the receiver to return a `LineLayer` for the route line, given a layer identifier and a source identifier.
    /// This method is invoked when the map view loads and any time routes are added.
    /// - Parameters:
    ///   - carPlayMapViewController: The ``CarPlayMapViewController`` object.
    ///   - identifier: The `LineLayer` identifier.
    ///   - sourceIdentifier: Identifier of the source, which contains the route data that this method would style.
    /// - Returns: A `LineLayer` that is applied to the route line.
    /// - SeeAlso: ``CarPlayManagerDelegate/carPlayManager(_:routeLineLayerWithIdentifier:sourceIdentifier:for:)``.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        routeLineLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> LineLayer?

    /// Asks the receiver to return a `LineLayer` for the casing layer that surrounds route line, given a layer
    /// identifier and a source identifier.
    /// This method is invoked when the map view loads and any time routes are added.
    /// This method is invoked when the map view loads and any time routes are added.
    /// - Parameters:
    ///   - carPlayMapViewController: The ``CarPlayMapViewController`` object.
    ///   - identifier: The `LineLayer` identifier.
    ///   - sourceIdentifier: Identifier of the source, which contains the route data that this method would style.
    /// - Returns: A `LineLayer` that is applied as a casing around the route line.
    /// - SeeAlso:
    /// ``CarPlayManagerDelegate/carPlayManager(_:routeCasingLineLayerWithIdentifier:sourceIdentifier:for:)``.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        routeCasingLineLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> LineLayer?

    /// Asks the receiver to return a `LineLayer` for highlighting restricted areas portions of the route, given a layer
    /// identifier and a source identifier.
    ///  This method is invoked when the map view loads and any time routes are added.
    /// - Parameters:
    ///   - carPlayMapViewController: The ``CarPlayMapViewController`` object.
    ///   - identifier: The `LineLayer` identifier.
    ///   - sourceIdentifier: Identifier of the source, which contains the route data that this method would style.
    /// - Returns: `LineLayer` that is applied as restricted areas on the route line.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        routeRestrictedAreasLineLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> LineLayer?

    /// Asks the receiver to adjust the default layer which will be added to the map view and return a `Layer`.
    ///  This method is invoked when the map view loads and any time a layer will be added.
    /// - Parameters:
    ///   - carPlayMapViewController: The ``CarPlayMapViewController`` object.
    ///   - layer: A default `Layer` generated by the carPlayMapViewController.
    /// - Returns: An adjusted `Layer` that will be added to the map view by the SDK.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        willAdd layer: Layer
    ) -> Layer?

    /// Returns a `FeatureCollection` that represents intermediate waypoints along the route (that is, excluding the
    /// origin).
    /// If this method is unimplemented, the navigation view controller's map view draws the waypoints using default
    /// `FeatureCollection`.
    /// - Parameters:
    ///   - carPlayMapViewController: The ``CarPlayMapViewController`` object.
    ///   - waypoints: The intermediate waypoints to be displayed on the map.
    ///   - legIndex: The index of the current leg during navigation.
    /// - Returns: A `FeatureCollection` that represents intermediate waypoints along the route (that is, excluding the
    /// origin).
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        shapeFor waypoints: [Waypoint],
        legIndex: Int
    ) -> FeatureCollection?

    // Asks the receiver to return a `CircleLayer` for waypoints, given an identifier and source.
    /// This method is invoked any time waypoints are added or shown.
    /// - Parameters:
    ///   - carPlayMapViewController: The `CarPlayMapViewController` object.
    ///   - identifier: The `CircleLayer` identifier.
    ///   - sourceIdentifier: Identifier of the source, which contains the waypoint data that this method would style.
    /// - Returns: A `CircleLayer` that the map applies to all intermediate waypoints.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        waypointCircleLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> CircleLayer?

    /// Asks the receiver to return a `SymbolLayer` for waypoint symbols, given an identifier and source.
    /// This method is invoked any time waypoints are added or shown.
    /// - Parameters:
    ///   - carPlayMapViewController: The `CarPlayMapViewController` object.
    ///   - identifier: The `SymbolLayer` identifier.
    ///   - sourceIdentifier: Identifier of the source, which contains the waypoint data that this method would style.
    /// - Returns: A `SymbolLayer` that the map applies to all intermediate waypoint symbols.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        waypointSymbolLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> SymbolLayer?
}

extension CarPlayMapViewControllerDelegate {
    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        didAdd finalDestinationAnnotation: PointAnnotation,
        pointAnnotationManager: PointAnnotationManager
    ) {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayNavigationViewController,
        routeLineLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> LineLayer? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayNavigationViewController,
        routeCasingLineLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> LineLayer? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayNavigationViewController,
        routeRestrictedAreasLineLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> LineLayer? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        willAdd layer: Layer
    ) -> Layer? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        shapeFor waypoints: [Waypoint],
        legIndex: Int
    ) -> FeatureCollection? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        waypointCircleLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> CircleLayer? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }

    /// `UnimplementedLogging` prints a warning to standard output the first time this method is called.
    public func carPlayMapViewController(
        _ carPlayMapViewController: CarPlayMapViewController,
        waypointSymbolLayerWithIdentifier identifier: String,
        sourceIdentifier: String
    ) -> SymbolLayer? {
        logUnimplemented(protocolType: CarPlayMapViewControllerDelegate.self, level: .info)
        return nil
    }
}
