import Foundation
import MapboxDirections
import MapboxNavigationNative

extension Incident {
    init(_ incidentInfo: IncidentInfo) {
        let incidentType: Incident.Kind
        switch incidentInfo.type {
        case .accident:
            incidentType = .accident
        case .congestion:
            incidentType = .congestion
        case .construction:
            incidentType = .construction
        case .disabledVehicle:
            incidentType = .disabledVehicle
        case .laneRestriction:
            incidentType = .laneRestriction
        case .massTransit:
            incidentType = .massTransit
        case .miscellaneous:
            incidentType = .miscellaneous
        case .otherNews:
            incidentType = .otherNews
        case .plannedEvent:
            incidentType = .plannedEvent
        case .roadClosure:
            incidentType = .roadClosure
        case .roadHazard:
            incidentType = .roadHazard
        case .weather:
            incidentType = .weather
        @unknown default:
            assertionFailure("Unknown IncidentInfo type.")
            incidentType = .undefined
        }

        self.init(
            identifier: incidentInfo.id,
            type: incidentType,
            description: incidentInfo.description ?? "",
            creationDate: incidentInfo.creationTime ?? Date.distantPast,
            startDate: incidentInfo.startTime ?? Date.distantPast,
            endDate: incidentInfo.endTime ?? Date.distantPast,
            impact: .init(incidentInfo.impact),
            subtype: incidentInfo.subType,
            subtypeDescription: incidentInfo.subTypeDescription,
            alertCodes: Set(incidentInfo.alertcCodes.map(\.intValue)),
            lanesBlocked: BlockedLanes(descriptions: incidentInfo.lanesBlocked),
            shapeIndexRange: -1 ..< -1,
            countryCodeAlpha3: incidentInfo.iso_3166_1_alpha3,
            countryCode: incidentInfo.iso_3166_1_alpha2,
            roadIsClosed: incidentInfo.roadClosed,
            longDescription: incidentInfo.longDescription,
            numberOfBlockedLanes: incidentInfo.numLanesBlocked?.intValue,
            congestionLevel: incidentInfo.congestion?.value?.intValue,
            affectedRoadNames: incidentInfo.affectedRoadNames
        )
    }
}

extension Incident.Impact {
    init(_ incidentImpact: IncidentImpact) {
        switch incidentImpact {
        case .unknown:
            self = .unknown
        case .critical:
            self = .critical
        case .major:
            self = .major
        case .minor:
            self = .minor
        case .low:
            self = .low
        @unknown default:
            assertionFailure("Unknown IncidentImpact value.")
            self = .unknown
        }
    }
}
