import CoreLocation
import Foundation
import MapboxNavigationNative_Private

/// Provides information about the status of the enhanced location updates generated by the map matching engine of the
/// Navigation SDK.
public struct MapMatchingResult: Equatable, @unchecked Sendable {
    /// The best possible location update, snapped to the route or map matched to the road if possible
    public var enhancedLocation: CLLocation

    /// A list of predicted location points leading up to the target update.
    ///
    /// The last point on the list (if it is not empty) is always equal to `enhancedLocation`.
    public var keyPoints: [CLLocation]

    /// Whether the SDK thinks that the user is off road.
    ///
    /// Based on the `offRoadProbability`.
    public var isOffRoad: Bool

    /// Probability that the user is off road.
    public var offRoadProbability: Double

    /// Returns true if map matcher changed its opinion about most probable path on last update.
    ///
    /// In practice it means we don't need to animate puck movement from previous to current location and just do an
    /// immediate transition instead.
    public var isTeleport: Bool

    /// When map matcher snaps to a road, this is the confidence in the chosen edge from all nearest edges.
    public var roadEdgeMatchProbability: Double

    /// Creates a new `MapMatchingResult` with given parameters
    ///
    /// It is not intended for user to create his own `MapMatchingResult` except for testing purposes.
    @_documentation(visibility: internal)
    public init(
        enhancedLocation: CLLocation,
        keyPoints: [CLLocation],
        isOffRoad: Bool,
        offRoadProbability: Double,
        isTeleport: Bool,
        roadEdgeMatchProbability: Double
    ) {
        self.enhancedLocation = enhancedLocation
        self.keyPoints = keyPoints
        self.isOffRoad = isOffRoad
        self.offRoadProbability = offRoadProbability
        self.isTeleport = isTeleport
        self.roadEdgeMatchProbability = roadEdgeMatchProbability
    }

    init(status: NavigationStatus) {
        self.enhancedLocation = CLLocation(status.location)
        self.keyPoints = status.keyPoints.map { CLLocation($0) }
        self.isOffRoad = status.offRoadProba > 0.5
        self.offRoadProbability = Double(status.offRoadProba)
        self.isTeleport = status.mapMatcherOutput.isTeleport
        self.roadEdgeMatchProbability = Double(status.mapMatcherOutput.matches.first?.proba ?? 0.0)
    }
}

extension CLLocation {
    convenience init(_ location: FixLocation) {
        let timestamp = Date(timeIntervalSince1970: TimeInterval(location.monotonicTimestampNanoseconds) / 1e9)
        self.init(
            coordinate: location.coordinate,
            altitude: location.altitude?.doubleValue ?? 0,
            horizontalAccuracy: location.accuracyHorizontal?.doubleValue ?? -1,
            verticalAccuracy: location.verticalAccuracy?.doubleValue ?? -1,
            course: location.bearing?.doubleValue ?? -1,
            courseAccuracy: location.bearingAccuracy?.doubleValue ?? -1,
            // TODO: investigate why we need 0 when in v2 we used to have -1
            speed: location.speed?.doubleValue ?? 0,
            speedAccuracy: location.speedAccuracy?.doubleValue ?? -1,
            timestamp: timestamp
        )
    }
}
