import Foundation
import MapboxNavigationNative
import CoreLocation

/// Provides information about the status of the enhanced location updates generated by the map matching engine of the Navigation SDK.
public struct MapMatchingResult {
    
    /// The best possible location update, snapped to the route or map matched to the road if possible
    public var enhancedLocation: CLLocation
    
    /// A list of predicted location points leading up to the target update.
    ///
    /// The last point on the list (if it is not empty) is always equal to `enhancedLocation`.
    public var keyPoints: [CLLocation]
    
    /// Whether the SDK thinks that the user is off road.
    ///
    /// Based on the `offRoadProbability`.
    public var isOffRoad: Bool
    
    /// Probability that the user is off road.
    public var offRoadProbability: Double
    
    /// Returns true if map matcher changed its opinion about most probable path on last update.
    ///
    /// In practice it means we don't need to animate puck movement from previous to current location and just do an immediate transition instead.
    public var isTeleport: Bool
    
    /// When map matcher snaps to a road, this is the confidence in the chosen edge from all nearest edges.
    public var roadEdgeMatchProbability: Double
    
    /// :nodoc:
    /// Creates new `MapMatchingResult` with given parameters
    ///
    /// It is not intended for user to create his own `MapMatchingResult` except for testing purposes.
    public init(enhancedLocation: CLLocation,
                keyPoints: [CLLocation],
                isOffRoad: Bool,
                offRoadProbability: Double,
                isTeleport: Bool,
                roadEdgeMatchProbability: Double) {
        self.enhancedLocation = enhancedLocation
        self.keyPoints = keyPoints
        self.isOffRoad = isOffRoad
        self.offRoadProbability = offRoadProbability
        self.isTeleport = isTeleport
        self.roadEdgeMatchProbability = roadEdgeMatchProbability
    }
    
    init(status: NavigationStatus) {
        enhancedLocation = CLLocation(status.location)
        keyPoints = status.keyPoints.map { CLLocation($0) }
        isOffRoad = status.offRoadProba > 0.5
        offRoadProbability = Double(status.offRoadProba)
        isTeleport = status.mapMatcherOutput.isIsTeleport
        roadEdgeMatchProbability = Double(status.mapMatcherOutput.matches.first?.proba ?? 0.0)
    }
}
