//
//  Copyright © 2025 Kyle Hughes. All rights reserved.
//  SPDX-License-Identifier: MIT
//

import Foundation
import Testing
@testable import Weakify

@Suite("Weakify w/ Sendable Value-Returning Functions & Optional Closures")
struct WeakifySendableOptionalValueReturningTests {
    // MARK: Tests

    @Test("Falls back to default after target is deallocated")
    @available(macOS 15.0, iOS 18.0, tvOS 18.0, watchOS 11.0, *)
    func returnsDefaultAfterTargetReleased() {
        let box = WeakBox<SendableCounter>()
        var add: (@Sendable (Int) -> Int?)!

        do {
            let counter = SendableCounter()
            box.reference = counter
            
            add = weakify(SendableCounter.add(_:), on: counter, default: nil)
        }

        #expect(box.reference == nil)
        #expect(add(42) == nil)
    }
    
    @Test("Returns the wrapped value while target lives")
    @available(macOS 15.0, iOS 18.0, tvOS 18.0, watchOS 11.0, *)
    func returnsValueWhileTargetAlive() {
        let counter = SendableCounter()
        
        let add: @Sendable (Int) -> Int? = weakify(SendableCounter.add(_:), on: counter, default: nil)
        
        #expect(add(3) == 3)
        #expect(counter.value == 3)
    }
    
    @Test("Falls back to non-nil default after target is deallocated")
    @available(macOS 15.0, iOS 18.0, tvOS 18.0, watchOS 11.0, *)
    func returnsNonNilDefaultAfterTargetReleased() {
        let box = WeakBox<SendableCounter>()
        var add: (@Sendable (Int) -> Int?)!

        do {
            let counter = SendableCounter()
            box.reference = counter
            
            add = weakify(SendableCounter.add(_:), on: counter, default: -1)
        }

        #expect(box.reference == nil)
        #expect(add(42) == -1)
    }
    
    @Test("Returns the wrapped value while target lives with non-nil default")
    @available(macOS 15.0, iOS 18.0, tvOS 18.0, watchOS 11.0, *)
    func returnsValueWhileTargetAliveWithNonNilDefault() {
        let counter = SendableCounter()
        
        let add: @Sendable (Int) -> Int? = weakify(SendableCounter.add(_:), on: counter, default: -1)
        
        #expect(add(3) == 3)
        #expect(counter.value == 3)
    }
}
