//
//  Copyright © 2025 Kyle Hughes. All rights reserved.
//  SPDX-License-Identifier: MIT
//

// MARK: - Non-Sendable Interface -

/// Creates a closure that weakly captures the given `target` and invokes the supplied unapplied method
/// reference.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// This helper evaluates `default()` whenever `target` has been deallocated.
///
/// The returned closure can be stored safely without extending the lifetime of `target`.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured weakly.
/// - Parameter default: Expression evaluated when `target` is `nil`.
/// - Returns: A closure with the same arity and return type as the method referenced by `unappliedMethodReference`, but
///   which is safe to store without extending the lifetime of `target`.
@inlinable
public func weakify<Target, Output, each Argument>(
    _ unappliedMethodReference: @escaping (Target) -> (repeat each Argument) -> Output,
    on target: Target,
    default: @autoclosure @escaping () -> Output
) -> (repeat each Argument) -> Output where Target: AnyObject {
    { [weak target] (argument: repeat each Argument) in
        guard let target else {
            return `default`()
        }
        
        return unappliedMethodReference(target)(repeat each argument)
    }
}

/// Creates a closure that weakly captures the given `target` and invokes the supplied unapplied method
/// reference, returning an optional value for compatibility with closures that can accept optional return types.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// This helper evaluates `default()` whenever `target` has been deallocated.
///
/// The returned closure can be stored safely without extending the lifetime of `target`.
///
/// Use this overload when you need to assign the weakified closure to a context that can accept an optional return
/// type, even though the underlying method reference does not return an optional. This is particularly useful if
/// you want the default value to be nil, which is more ergonomic than coming up with a fake return value you don't
/// care about.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured weakly.
/// - Parameter default: Expression evaluated when `target` is `nil`.
/// - Returns: A closure with the same arity and optional return type as compatible with calling contexts that can
///   accept optionals, but which is safe to store without extending the lifetime of `target`.
@inlinable
public func weakify<Target, Output, each Argument>(
    _ unappliedMethodReference: @escaping (Target) -> (repeat each Argument) -> Output,
    on target: Target,
    default: @autoclosure @escaping () -> Output?
) -> (repeat each Argument) -> Output? where Target: AnyObject {
    { [weak target] (argument: repeat each Argument) in
        guard let target else {
            return `default`()
        }
        
        return unappliedMethodReference(target)(repeat each argument)
    }
}

/// Creates a closure that weakly captures the given `target` and invokes the supplied unapplied method
/// reference.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// This helper has no side effects whenever `target` has been deallocated.
///
/// The returned closure can be stored safely without extending the lifetime of `target`.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured weakly.
/// - Returns: A closure with the same arity and return type as the method referenced by `unappliedMethodReference`,
///   but which is safe to store without extending the lifetime of `target`.
@inlinable
public func weakify<Target, each Argument>(
    _ unappliedMethodReference: @escaping (Target) -> (repeat each Argument) -> Void,
    on target: Target
) -> (repeat each Argument) -> Void where Target: AnyObject {
    weakify(unappliedMethodReference, on: target, default: ())
}

// MARK: - Sendable Interface -

/// Creates a `Sendable` closure that weakly captures the given `target` and invokes the supplied `Sendable`
/// unapplied method reference.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// This helper evaluates `default()` whenever `target` has been deallocated.
///
/// The returned `Sendable` closure can be stored safely without extending the lifetime of `target`.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured weakly.
/// - Parameter default: Expression evaluated when `target` is `nil`.
/// - Returns: A `Sendable` closure with the same arity and return type as the method referenced by
///   `unappliedMethodReference`, but which is safe to store without extending the lifetime of `target`.
@inlinable
public func weakify<Target, Output, each Argument>(
    _ unappliedMethodReference: @Sendable @escaping (Target) -> (repeat each Argument) -> Output,
    on target: Target,
    default: @Sendable @autoclosure @escaping () -> Output
) -> @Sendable (repeat each Argument) -> Output where Target: AnyObject & Sendable {
    { [weak target] (argument: repeat each Argument) in
        guard let target else {
            return `default`()
        }
        
        return unappliedMethodReference(target)(repeat each argument)
    }
}

/// Creates a `Sendable` closure that weakly captures the given `target` and invokes the supplied `Sendable`
/// unapplied method reference, returning an optional value for compatibility with closures that can accept optional return types.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// This helper evaluates `default()` whenever `target` has been deallocated.
///
/// The returned `Sendable` closure can be stored safely without extending the lifetime of `target`.
///
/// Use this overload when you need to assign the weakified closure to a context that can accept an optional return
/// type, even though the underlying method reference does not return an optional. This is particularly useful if
/// you want the default value to be nil, which is more ergonomic than coming up with a fake return value you don't
/// care about.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured weakly.
/// - Parameter default: Expression evaluated when `target` is `nil`.
/// - Returns: A `Sendable` closure with the same arity and optional return type as compatible with calling contexts
///   that can accept optionals, but which is safe to store without extending the lifetime of `target`.
@inlinable
public func weakify<Target, Output, each Argument>(
    _ unappliedMethodReference: @Sendable @escaping (Target) -> (repeat each Argument) -> Output,
    on target: Target,
    default: @Sendable @autoclosure @escaping () -> Output?
) -> @Sendable (repeat each Argument) -> Output? where Target: AnyObject & Sendable {
    { [weak target] (argument: repeat each Argument) in
        guard let target else {
            return `default`()
        }
        
        return unappliedMethodReference(target)(repeat each argument)
    }
}

/// Creates a `Sendable` closure that weakly captures the given `target` and invokes the supplied `Sendable`
/// unapplied method reference.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// This helper has no side effects whenever `target` has been deallocated.
///
/// The returned `Sendable` closure can be stored safely without extending the lifetime of `target`.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured weakly.
/// - Returns: A `Sendable` closure with the same arity and return type as the method referenced by
///   `unappliedMethodReference`, but which is safe to store without extending the lifetime of `target`.
@inlinable
public func weakify<Target, each Argument>(
    _ unappliedMethodReference: @Sendable @escaping (Target) -> (repeat each Argument) -> Void,
    on target: Target
) -> @Sendable (repeat each Argument) -> Void where Target: AnyObject & Sendable {
    weakify(unappliedMethodReference, on: target, default: ())
}
