//
//  Copyright © 2025 Kyle Hughes. All rights reserved.
//  SPDX-License-Identifier: MIT
//

// MARK: - Non-Sendable Interface -

/// Creates a closure that unownedly captures the given `target` and invokes the supplied unapplied method
/// reference.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// The returned closure can be stored safely without extending the lifetime of `target`.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The object to be captured unownedly.
/// - Returns: A closure with the same arity and return type as the method referenced by `unappliedMethodReference`,
///   but which is safe to store without extending the lifetime of `target`.
@inlinable
public func disown<Target, Output, each Argument>(
    _ unappliedMethodReference: @escaping (Target) -> (repeat each Argument) -> Output,
    on target: Target
) -> (repeat each Argument) -> Output where Target: AnyObject {
    { [unowned target] (argument: repeat each Argument) in
        unappliedMethodReference(target)(repeat each argument)
    }
}

// MARK: - Sendable Interface -

/// Creates a `Sendable` closure that unownedly captures the given `target` and invokes the supplied `Sendable`
/// unapplied method reference.
///
/// An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
/// instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
/// `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the
/// regular `(Args…) -> Result` closure you would normally call.
///
/// The returned `Sendable` closure can be stored safely without extending the lifetime of `target`.
///
/// - Parameter unappliedMethodReference: Unapplied method reference on `Target`.
/// - Parameter target: The `Sendable` object to be captured unownedly.
/// - Returns: A `Sendable` closure with the same arity and return type as the method referenced by
///   `unappliedMethodReference`, but which is safe to store without extending the lifetime of `target`.
@inlinable
public func disown<Target, Output, each Argument>(
    _ unappliedMethodReference: @Sendable @escaping (Target) -> (repeat each Argument) -> Output,
    on target: Target
) -> @Sendable (repeat each Argument) -> Output where Target: AnyObject & Sendable {
    { [unowned target] (argument: repeat each Argument) in
        unappliedMethodReference(target)(repeat each argument)
    }
}
