# Weakify

[![Platform Versions](https://img.shields.io/endpoint?url=https%3A%2F%2Fswiftpackageindex.com%2Fapi%2Fpackages%2Fkylehughes%2FWeakify%2Fbadge%3Ftype%3Dplatforms)](https://swiftpackageindex.com/kylehughes/Weakify)
[![Swift Versions](https://img.shields.io/endpoint?url=https%3A%2F%2Fswiftpackageindex.com%2Fapi%2Fpackages%2Fkylehughes%2FWeakify%2Fbadge%3Ftype%3Dswift-versions)](https://swiftpackageindex.com/kylehughes/Weakify)
[![Test](https://github.com/kylehughes/Weakify/actions/workflows/test.yml/badge.svg)](https://github.com/kylehughes/Weakify/actions/workflows/test.yml)

*A simple, ergonomic solution for safely capturing method references in Swift.*

## About

Weakify provides convenient and safe mechanisms for capturing method references weakly or unownedly, ensuring closures referencing methods on an object do not inadvertently extend the object's lifetime. It relies on unapplied method references, and uses parameter packs to support heterogeneous argument lists.

This package encourages:

* Single-line syntax for capturing method references.
* Avoidance of retain cycles through `weak` or `unowned` captures.

Weakify is extremely lightweight and has a robust test suite.

## Capabilities

* [x] Weakly capture method references with automatic fallback values.
* [x] Unownedly capture method references when you know the target will outlive the closure.
* [x] Ergonomic handling of heterogenous arguments.
* [x] Swift 6 language mode support.

## Supported Platforms

* iOS 13.0+
* macOS 10.15+
* tvOS 13.0+
* visionOS 1.0+
* watchOS 6.0+

## Requirements

* Swift 6.1+
* Xcode 16.3+

> [!NOTE] 
> This package ideally would only require Swift 5.9, but compiler versions prior to 6.1 (packaged with Xcode 16.3)
> have a bug that makes the intended usage incompatible with `@MainActor`-isolated closures. We require Swift language
> tools version 6.1 as a proxy for this tribal knowledge.

## Documentation

[Documentation is available on GitHub Pages.](https://kylehughes.github.io/Weakify/)

## Installation

### Swift Package Manager

```swift
dependencies: [
    .package(url: "https://github.com/kylehughes/Weakify.git", .upToNextMajor(from: "1.0.0")),
]
```

## Quick Start

Weakly capture a method reference:

```swift
import Weakify

class MyViewController: UIViewController {
    private lazy var button: UIButton = {
        let button = UIButton()
        button.addAction(
            UIAction(handler: weakify(MyViewController.buttonTapped, on: self)),
            for: .primaryActionTriggered
        )
        return button
    }()

    private func buttonTapped(_ action: UIAction) {
        print("Button tapped")
    }
}
```

Unownedly capture a method reference:

```swift
import Weakify

class MyViewController: UIViewController {
    func observe(notificationCenter: NotificationCenter) {
        notificationCenter.addObserver(
            forName: NSNotification.Name("MyNotification"),
            object: nil,
            queue: .main,
            using: disown(MyViewController.handleNotification, on: self)
        )
    }

    private func handleNotification(_ notification: Notification) {
        print("Notification received")
    }
}
```

## Usage

> [!NOTE]
> An unapplied method reference (UMR) is the function value produced by writing an instance method on the type
> instead of an instance, for example `UIView.layoutIfNeeded` or `Array.append`. Its curried shape is
> `(Self) -> (Args…) -> Result`, so supplying a specific instance—`unappliedMethodReference(target)`—yields the regular
> `(Args…) -> Result` closure you would normally call.

### Weak Capture with Fallback

Use `weakify` to safely capture `self` without retaining it. A default fallback value can be provided for when `self` has been deallocated:

```swift
let weakHandler = weakify(MyViewController.formatMessage, on: self, default: "N/A")
```

### Weak Capture without Fallback

Use `weakify` to safely capture `self` without retaining it. If the accepting closure does not need a return value, you can omit the fallback. No side effects will occur if the target is deallocated.

```swift
let weakHandler = weakify(MyViewController.fireAndForget, on: self)
```

### Unowned Capture

Use `disown` to capture `self` when the target object will definitely outlive the closure:

```swift
let unownedHandler = disown(MyViewController.updateStatus, on: self)
```

### Heterogeneous Arguments

Weakify supports methods with heterogeneous argument lists:

```swift
func printMessage(_ prefix: String, count: Int) {
    print("\(prefix): \(count)")
}

let printer = weakify(MyViewController.printMessage, on: self)
printer("Age", 5)
```

## Important Behavior

* `weakify` closures evaluate the provided default when the target is deallocated.
* `disown` closures will crash if called after the target is deallocated; ensure the target outlives the closure.

## Contributions

Weakify is not accepting source contributions at this time. Bug reports will be considered.

## Author

[Kyle Hughes](https://kylehugh.es)

[![Bluesky][bluesky_image]][bluesky_url]  
[![LinkedIn][linkedin_image]][linkedin_url]  
[![Mastodon][mastodon_image]][mastodon_url]

[bluesky_image]: https://img.shields.io/badge/Bluesky-0285FF?logo=bluesky&logoColor=fff
[bluesky_url]: https://bsky.app/profile/kylehugh.es
[linkedin_image]: https://img.shields.io/badge/LinkedIn-0A66C2?logo=linkedin&logoColor=fff
[linkedin_url]: https://www.linkedin.com/in/kyle-hughes
[mastodon_image]: https://img.shields.io/mastodon/follow/109356914477272810?domain=https%3A%2F%2Fmister.computer&style=social
[mastodon_url]: https://mister.computer/@kyle

## License

Weakify is available under the MIT license.

See `LICENSE` for details.
