// Copyright (c) 2014-2018, The Monero Project
//
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
// 1. Redistributions of source code must retain the above copyright notice, this list of
//    conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright notice, this list
//    of conditions and the following disclaimer in the documentation and/or other
//    materials provided with the distribution.
//
// 3. Neither the name of the copyright holder nor the names of its contributors may be
//    used to endorse or promote products derived from this software without specific
//    prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
// THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
// THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import QtQuick 2.9
import QtQuick.Controls 2.0
import QtQuick.Dialogs 1.2
import QtQuick.Layouts 1.1
import QtQuick.Controls.Styles 1.4
import QtQuick.Window 2.0

import "../components" as BittubeComponents

Item {
    id: root
    visible: false
    property alias labelText: label.text
    property alias inputText: input.text

    // same signals as Dialog has
    signal accepted()
    signal rejected()

    function open(prepopulate) {
        leftPanel.enabled = false
        middlePanel.enabled = false
        titleBar.state = "essentials"
        root.visible = true;
        input.focus = true;
        input.text = prepopulate ? prepopulate : "";
    }

    function close() {
        leftPanel.enabled = true
        middlePanel.enabled = true
        titleBar.state = "default"
        root.visible = false;
    }

    ColumnLayout {
        z: parent.z + 1
        id: mainLayout
        spacing: 10
        anchors { fill: parent; margins: 35 }

        ColumnLayout {
            id: column

            Layout.fillWidth: true
            Layout.alignment: Qt.AlignHCenter
            Layout.maximumWidth: 400

            Label {
                id: label
                Layout.fillWidth: true

                font.pixelSize: 16
                font.family: BittubeComponents.Style.fontLight.name

                color: BittubeComponents.Style.defaultFontColor
            }

            BittubeComponents.Input {
                id : input
                focus: true
                Layout.topMargin: 6
                Layout.fillWidth: true
                horizontalAlignment: TextInput.AlignLeft
                verticalAlignment: TextInput.AlignVCenter
                font.family: BittubeComponents.Style.fontLight.name
                font.pixelSize: 24
                KeyNavigation.tab: okButton
                bottomPadding: 10
                leftPadding: 10
                topPadding: 10
                color: BittubeComponents.Style.defaultFontColor
                selectionColor: BittubeComponents.Style.textSelectionColor
                selectedTextColor: BittubeComponents.Style.textSelectedColor

                background: Rectangle {
                    radius: 2
                    border.color: BittubeComponents.Style.inputBorderColorActive
                    border.width: 1
                    color: BittubeComponents.Style.blackTheme ? "black" : "#A9FFFFFF"
                }

                Keys.enabled: root.visible
                Keys.onEnterPressed: Keys.onReturnPressed(event)
                Keys.onReturnPressed: {
                    root.close()
                    root.accepted()
                }
                Keys.onEscapePressed: {
                    root.close()
                    root.rejected()
                }
            }

            // Ok/Cancel buttons
            RowLayout {
                id: buttons
                spacing: 16
                Layout.topMargin: 16
                Layout.alignment: Qt.AlignRight

                BittubeComponents.StandardButton {
                    id: cancelButton
                    small: true
                    width: 120
                    fontSize: 14
                    text: qsTr("Cancel") + translationManager.emptyString
                    KeyNavigation.tab: input
                    onClicked: {
                        root.close()
                        root.rejected()
                    }
                }
                BittubeComponents.StandardButton {
                    id: okButton
                    small: true
                    width: 120
                    fontSize: 14
                    text: qsTr("Ok") + translationManager.emptyString
                    KeyNavigation.tab: cancelButton
                    onClicked: {
                        root.close()
                        root.accepted()
                    }
                }
            }
        }
    }
}
