import Foundation

let arguments = ProcessInfo().arguments
if arguments.count < 3 {
    print("missing arguments")
    exit(1)
}

let (inputPath, outputPath) = (arguments[1], arguments[2])

var generatedCode: [String] = [
    "import Foundation",
    "import SwiftUI",
    "",
    "// This file was autogenerated — do not modify.",
    ""
]

let stringsDictionary: [String: String] = {
    do {
        let stringsData = try Data(contentsOf: URL(fileURLWithPath: inputPath))
        let plistObject = try PropertyListSerialization.propertyList(from: stringsData, format: nil)
        guard let stringsDictionary = plistObject as? [String: String] else {
            print("Failed to parse strings file — is it valid?")
            exit(1)
        }
        return stringsDictionary
    } catch {
        print("Failed to parse strings file — is it valid?")
        exit(1)
    }
}()

let tableName = URL(fileURLWithPath: inputPath).deletingPathExtension().lastPathComponent
let pluralizedKeySuffix: String = "_Plural"

// First, let's make some SwiftUI keys.
generatedCode.append("extension LocalizedStringKey {")

func symbolizeKey(_ key: String) -> String {
    guard !key.isEmpty else { return key }
    var keyNameAsSymbol: String = key

    // First, strip out leading digits (they're not allowed)
    while !keyNameAsSymbol.isEmpty, keyNameAsSymbol[keyNameAsSymbol.startIndex].isNumber {
        keyNameAsSymbol = String(keyNameAsSymbol.dropFirst())
    }

    guard !keyNameAsSymbol.isEmpty else { return keyNameAsSymbol }

    // Next, lowercase the first character
    keyNameAsSymbol.replaceSubrange(keyNameAsSymbol.startIndex..<keyNameAsSymbol.index(keyNameAsSymbol.startIndex, offsetBy: 1),
                                    with: keyNameAsSymbol.first!.lowercased())

    // Finally, strip out disallowed punctuation
    keyNameAsSymbol = keyNameAsSymbol.replacingOccurrences(of: "-", with: "")
    keyNameAsSymbol = keyNameAsSymbol.replacingOccurrences(of: ".", with: "")
    keyNameAsSymbol = keyNameAsSymbol.replacingOccurrences(of: "{", with: "")
    keyNameAsSymbol = keyNameAsSymbol.replacingOccurrences(of: "}", with: "")
    keyNameAsSymbol = keyNameAsSymbol.replacingOccurrences(of: "$", with: "")
    return keyNameAsSymbol
}

for fullKey in stringsDictionary.keys.sorted() {
    guard !fullKey.isEmpty else { continue }
    let keyComponents: [String] = fullKey.components(separatedBy: " ")
    guard !keyComponents.isEmpty else { continue }
    let keyName = keyComponents.first!
    let keyNameAsSymbol = symbolizeKey(keyName)

    let isPlural = (keyNameAsSymbol.hasSuffix(pluralizedKeySuffix))
    guard !isPlural else { continue }

    let pluralizedKeyName = keyName.appending(pluralizedKeySuffix)
    let fullKeyWithPluralizedName: String = {
        var components = Array(keyComponents.dropFirst())
        components.insert(pluralizedKeyName, at: 0)
        return components.joined(separator: " ")
    }()
    let hasPluralizedVariant = (stringsDictionary[fullKeyWithPluralizedName] != nil)

    if keyComponents.count == 1 {
        let lines = SwiftUISymbolBuilder.symbolDefinition(for: keyNameAsSymbol, fullKey: fullKey,
                                                          hasPluralization: hasPluralizedVariant,
                                                          pluralizedSuffix: pluralizedKeySuffix)
        generatedCode.append(contentsOf: lines)
    } else {
        let formatSpecifiers = keyComponents.dropFirst(1).filter({ $0.starts(with: "%") })
        let lines = SwiftUISymbolBuilder.symbolDefinition(for: keyNameAsSymbol, keyName: keyName,
                                                          formatSpecifiers: formatSpecifiers,
                                                          hasPluralization: hasPluralizedVariant,
                                                          pluralizedSuffix: pluralizedKeySuffix)
        generatedCode.append(contentsOf: lines)
    }
}

generatedCode.append("}")
generatedCode.append("")

// Next, some plain strings.
generatedCode.append("struct \(tableName) {")

for fullKey in stringsDictionary.keys.sorted() {
    guard !fullKey.isEmpty else { continue }
    let keyComponents: [String] = fullKey.components(separatedBy: " ")
    guard !keyComponents.isEmpty else { continue }
    let keyName = keyComponents.first!
    let keyNameAsSymbol = symbolizeKey(keyName)
    
    let isPlural = (keyNameAsSymbol.hasSuffix(pluralizedKeySuffix))
    guard !isPlural else { continue }

    let pluralizedKeyName = keyName.appending(pluralizedKeySuffix)
    let fullKeyWithPluralizedName: String = {
        var components = Array(keyComponents.dropFirst())
        components.insert(pluralizedKeyName, at: 0)
        return components.joined(separator: " ")
    }()
    let hasPluralizedVariant = (stringsDictionary[fullKeyWithPluralizedName] != nil)

    if keyComponents.count == 1 {
        let lines = StringSymbolBuilder.symbolDefinition(for: keyNameAsSymbol, fullKey: fullKey,
                                                         hasPluralization: hasPluralizedVariant,
                                                         pluralizedFullKey: fullKeyWithPluralizedName)
        generatedCode.append(contentsOf: lines)
    } else {
        let formatSpecifiers = keyComponents.dropFirst(1).filter({ $0.starts(with: "%") })
        let lines = StringSymbolBuilder.symbolDefinition(for: keyNameAsSymbol, fullKey: fullKey,
                                                         formatSpecifiers: formatSpecifiers,
                                                         hasPluralization: hasPluralizedVariant,
                                                         pluralizedFullKey: fullKeyWithPluralizedName)
        generatedCode.append(contentsOf: lines)
    }
}

generatedCode.append("}")
generatedCode.append("")

// Finally, add the table name to the struct, which is useful in SwiftUI.
generatedCode.append("extension \(tableName) {")
generatedCode.append("    static var tableName: String { return \"\(tableName)\" }")
generatedCode.append("}")
generatedCode.append("")

// Write out file.
let output = generatedCode.joined(separator: "\n")
try output.write(to: URL(fileURLWithPath: outputPath), atomically: true, encoding: .utf8)

