import Foundation

let arguments = ProcessInfo().arguments
if arguments.count < 3 {
    print("missing arguments")
    exit(1)
}

let (inputPath, outputPath) = (arguments[1], arguments[2])

var generatedCode: [String] = [
    "import Foundation",
    "import SwiftUI",
    "",
    "// This file was autogenerated — do not modify.",
    ""
]

let stringsDictionary: [String: String] = {
    do {
        let stringsData = try Data(contentsOf: URL(fileURLWithPath: inputPath))
        let plistObject = try PropertyListSerialization.propertyList(from: stringsData, format: nil)
        guard let stringsDictionary = plistObject as? [String: String] else {
            print("Failed to parse strings file — is it valid?")
            exit(1)
        }
        return stringsDictionary
    } catch {
        print("Failed to parse strings file — is it valid?")
        exit(1)
    }
}()

let tableName = URL(fileURLWithPath: inputPath).deletingPathExtension().lastPathComponent

// First, let's make some SwiftUI keys.
generatedCode.append("extension LocalizedStringKey {")

for key in stringsDictionary.keys.sorted() {
    guard !key.isEmpty else { continue }
    let keyComponents: [String] = key.components(separatedBy: " ")
    guard !keyComponents.isEmpty else { continue }
    var keyAsSymbol = keyComponents.first!
    keyAsSymbol.replaceSubrange(keyAsSymbol.startIndex..<keyAsSymbol.index(keyAsSymbol.startIndex, offsetBy: 1),
                                with: keyAsSymbol.first!.lowercased())

    if keyComponents.count == 1 {
        generatedCode.append("    static let \(keyAsSymbol): LocalizedStringKey = LocalizedStringKey(\"\(key)\")")
    } else {

        let formatComponents = keyComponents.dropFirst(1).filter({ $0.starts(with: "%") })

        // Image func
        let imageParameterList: String = formatComponents.enumerated().map({ "imageValue value\($0.offset): Image" }).joined(separator: ", ")
        generatedCode.append("    static func \(keyAsSymbol)(\(imageParameterList)) -> LocalizedStringKey {")
        generatedCode.append("        var interpolation = LocalizedStringKey.StringInterpolation(literalCapacity: \(1 + formatComponents.count), interpolationCount: \(formatComponents.count))")
        generatedCode.append("        interpolation.appendLiteral(\"\(keyComponents.first!)\")")
        formatComponents.enumerated().forEach({
            generatedCode.append("        interpolation.appendLiteral(\" \")")
            generatedCode.append("        interpolation.appendInterpolation(value\($0.offset))")
        })
        generatedCode.append("        return LocalizedStringKey(stringInterpolation: interpolation)")
        generatedCode.append("    }")

        // String func
        let stringParameterList: String = formatComponents.enumerated().map({ "formatValue value\($0.offset): String" }).joined(separator: ", ")
        generatedCode.append("    static func \(keyAsSymbol)(\(stringParameterList)) -> LocalizedStringKey {")
        generatedCode.append("        var interpolation = LocalizedStringKey.StringInterpolation(literalCapacity: \(1 + formatComponents.count), interpolationCount: \(formatComponents.count))")
        generatedCode.append("        interpolation.appendLiteral(\"\(keyComponents.first!)\")")
        formatComponents.enumerated().forEach({
            generatedCode.append("        interpolation.appendLiteral(\" \")")
            generatedCode.append("        interpolation.appendInterpolation(value\($0.offset))")
        })
        generatedCode.append("        return LocalizedStringKey(stringInterpolation: interpolation)")
        generatedCode.append("    }")
    }
}

generatedCode.append("}")
generatedCode.append("")

// Next, some plain strings.
generatedCode.append("struct \(tableName) {")

for key in stringsDictionary.keys.sorted() {
    guard !key.isEmpty else { continue }
    let keyComponents: [String] = key.components(separatedBy: " ")
    guard !keyComponents.isEmpty else { continue }
    var keyAsSymbol = keyComponents.first!
    keyAsSymbol.replaceSubrange(keyAsSymbol.startIndex..<keyAsSymbol.index(keyAsSymbol.startIndex, offsetBy: 1),
                                with: keyAsSymbol.first!.lowercased())

    if keyComponents.count == 1 {
        generatedCode.append("    static var \(keyAsSymbol): String {")
        generatedCode.append("        return NSLocalizedString(\"\(key)\", tableName: \"\(tableName)\", comment: \"\")")
        generatedCode.append("    }")
    } else {
        let formatComponents = keyComponents.dropFirst(1).filter({ $0.starts(with: "%") })
        let stringParameterList: String = formatComponents.enumerated().map({ "formatValue value\($0.offset): String" }).joined(separator: ", ")
        let parameterFormatList: String = formatComponents.enumerated().map({ "value\($0.offset)" }).joined(separator: ", ")
        generatedCode.append("    static func \(keyAsSymbol)(\(stringParameterList)) -> String {")
        generatedCode.append("        let string = NSLocalizedString(\"\(key)\", tableName: \"\(tableName)\", comment: \"\")")
        generatedCode.append("        return String(format: string, \(parameterFormatList))")
        generatedCode.append("    }")
    }
}

generatedCode.append("}")
generatedCode.append("")

// Write out file.
let output = generatedCode.joined(separator: "\n")
try output.write(to: URL(fileURLWithPath: outputPath), atomically: true, encoding: .utf8)

