// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// This package implements the 32-bit cyclic redundancy check, or CRC-32, checksum.
// See http://en.wikipedia.org/wiki/Cyclic_redundancy_check for information.
package crc32

import (
	"hash";
	"os";
)

// The size of a CRC-32 checksum in bytes.
const Size = 4

// Predefined polynomials.
const (
	// Far and away the most common CRC-32 polynomial.
	// Used by ethernet (IEEE 802.3), v.42, fddi, gzip, zip, png, mpeg-2, ...
	IEEE	= 0xedb88320;

	// Castagnoli's polynomial, used in iSCSI.
	// Has better error detection characteristics than IEEE.
	// http://dx.doi.org/10.1109/26.231911
	Castagnoli	= 0x82f63b78;

	// Koopman's polynomial.
	// Also has better error detection characteristics than IEEE.
	// http://dx.doi.org/10.1109/DSN.2002.1028931
	Koopman	= 0xeb31d82e;
)

// Table is a 256-word table representing the polynomial for efficient processing.
type Table [256]uint32

// MakeTable returns the Table constructed from the specified polynomial.
func MakeTable(poly uint32) *Table {
	t := new(Table);
	for i := 0; i < 256; i++ {
		crc := uint32(i);
		for j := 0; j < 8; j++ {
			if crc&1 == 1 {
				crc = (crc>>1)^poly;
			} else {
				crc >>= 1;
			}
		}
		t[i] = crc;
	}
	return t;
}

// IEEETable is the table for the IEEE polynomial.
var IEEETable = MakeTable(IEEE)

// digest represents the partial evaluation of a checksum.
type digest struct {
	crc	uint32;
	tab	*Table;
}

// New creates a new Hash computing the CRC-32 checksum
// using the polynomial represented by the Table.
func New(tab *Table) hash.Hash32 {
	return &digest{0, tab};
}

// NewIEEE creates a new Hash computing the CRC-32 checksum
// using the IEEE polynomial.
func NewIEEE() hash.Hash32 {
	return New(IEEETable);
}

func (d *digest) Size() int {
	return Size;
}

func (d *digest) Reset() {
	d.crc = 0;
}

func update(crc uint32, tab *Table, p []byte) uint32 {
	crc = ^crc;
	for i := 0; i < len(p); i++ {
		crc = tab[byte(crc)^p[i]]^(crc>>8);
	}
	return ^crc;
}

func (d *digest) Write(p []byte) (n int, err os.Error) {
	d.crc = update(d.crc, d.tab, p);
	return len(p), nil;
}

func (d *digest) Sum32() uint32 {
	return d.crc;
}

func (d *digest) Sum() []byte {
	p := make([]byte, 4);
	s := d.Sum32();
	p[0] = byte(s>>24);
	p[1] = byte(s>>16);
	p[2] = byte(s>>8);
	p[3] = byte(s);
	return p;
}

// Checksum returns the CRC-32 checksum of data
// using the polynomial represented by the Table.
func Checksum(data []byte, tab *Table) uint32 {
	return update(0, tab, data);
}

// ChecksumIEEE returns the CRC-32 checksum of data
// using the IEEE polynomial.
func ChecksumIEEE(data []byte) uint32 {
	return update(0, IEEETable, data);
}
