"use strict";
/**
 * detect file info
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LocFile = void 0;
const fs = __importStar(require("fs-extra"));
const slash2_1 = __importDefault(require("slash2"));
const languages_1 = require("./languages");
const DefaultLine = {
    total: 0,
    code: 0,
    comment: 0,
};
const DefaultFileInfo = {
    name: '',
    languages: '',
    size: 0,
    lines: DefaultLine,
};
/**
 * Collect language info for a single file
 */
class LocFile {
    /**
     * Creates an instance of LocFile.
     */
    constructor(rawPath, debug = false) {
        Object.defineProperty(this, "debug", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: debug
        });
        Object.defineProperty(this, "path", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "rawPath", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: void 0
        });
        Object.defineProperty(this, "languages", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: new languages_1.Languages()
        });
        Object.defineProperty(this, "filterData", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: (data, regexes) => {
                const lines = data.split(/\n/);
                let commentLength = 0;
                let codeLength = lines.length;
                const total = codeLength;
                let inMultiLineComment = false;
                lines.forEach((origLine) => {
                    var _a, _b, _c, _d, _e, _f;
                    let lineType = 'code';
                    const line = origLine.trim();
                    if (inMultiLineComment) {
                        let noCode = true;
                        if ((_a = regexes.multiLineCommentClose) === null || _a === void 0 ? void 0 : _a.test(line)) {
                            // line contains the end of a multi-line comment
                            inMultiLineComment = false;
                            if (!((_b = regexes.multiLineCommentCloseEnd) === null || _b === void 0 ? void 0 : _b.test(line))) {
                                // the multiline comment does not end this line.
                                // there is real code on it.
                                noCode = false;
                            }
                        }
                        if (noCode) {
                            lineType = 'comm';
                            commentLength += 1;
                            codeLength -= 1;
                        }
                    }
                    else if (line) {
                        // non-empty line
                        if ((_c = regexes.multiLineCommentOpen) === null || _c === void 0 ? void 0 : _c.test(line)) {
                            // line contains the start of a multi-line comment
                            // might contain some real code, but we'll let that slide
                            if (!((_d = regexes.multiLineCommentOpenAndClose) === null || _d === void 0 ? void 0 : _d.test(line))) {
                                // comment is not also closed on this line
                                inMultiLineComment = true;
                            }
                            if ((_e = regexes.multiLineCommentOpenStart) === null || _e === void 0 ? void 0 : _e.test(line)) {
                                // The comment starts the line. There is no other code on this line
                                commentLength += 1;
                                codeLength -= 1;
                                lineType = 'comm';
                            }
                        }
                        else if ((_f = regexes.singleLineComment) === null || _f === void 0 ? void 0 : _f.test(line)) {
                            // line contains only a single line comment
                            commentLength += 1;
                            codeLength -= 1;
                            lineType = 'comm';
                        }
                    }
                    else {
                        // empty line
                        codeLength -= 1;
                        lineType = 'empt';
                    }
                    if (this.debug) {
                        console.log(lineType, line);
                    }
                });
                return {
                    ...DefaultLine,
                    total,
                    code: codeLength,
                    comment: commentLength,
                };
            }
        });
        this.path = slash2_1.default(rawPath);
        this.rawPath = rawPath;
    }
    /**
     * Get file info when LocFile init
     */
    async getFileInfo(data) {
        if (!(await fs.pathExists(this.rawPath))) {
            throw new Error(`Error: file ${this.rawPath} does not exist.`);
        }
        let newData = data;
        const info = { ...DefaultFileInfo };
        // note: do not use Path.sep here since we have already
        // translated the path to a posix path using slash(...).
        const name = this.path.split('/').pop() || '';
        try {
            const stat = await fs.stat(this.path);
            if (!stat.isFile()) {
                return info;
            }
            newData = data || (await fs.readFile(this.path, 'utf-8'));
            info.name = name;
            info.size = (stat && stat.size) || 0;
            info.languages = this.languages.getType(this.path);
            if (!info.languages) {
                return info;
            }
            if (newData) {
                const regexes = this.languages.getRegexes(info.languages);
                info.lines = this.filterData(newData, regexes);
            }
        }
        catch (err) {
            throw new Error('read file failed.');
        }
        return info;
    }
    getFileInfoByContent(name, data) {
        const info = { ...DefaultFileInfo };
        info.name = name;
        info.languages = this.languages.getType(name);
        info.lines = this.filterData(data, this.languages.getRegexes(info.languages));
        return info;
    }
}
exports.LocFile = LocFile;
//# sourceMappingURL=file.js.map