"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ava_1 = require("ava");
const index_1 = require("./index");
const path = require("path");
const originalEnv = process.env;
const originalWorkingDirectory = process.cwd();
const testResources = path.resolve(path.join("src", "index.test"));
ava_1.default.beforeEach(_ => {
    process.env = { ...originalEnv };
});
ava_1.default.afterEach(_ => {
    process.env = originalEnv;
    process.chdir(originalWorkingDirectory);
});
ava_1.default("relative path with forward-slash is returned as-is", async (t) => {
    process.env.PATH = path.join(testResources, "path");
    t.deepEqual(await index_1.safeWhich("./anything"), "./anything");
});
ava_1.default("absolute path with forward-slash is returned as-is", async (t) => {
    process.env.PATH = path.join(testResources, "path");
    t.deepEqual(await index_1.safeWhich("/usr/bin/anything"), "/usr/bin/anything");
});
ava_1.default("binaries in cwd are not returned", async (t) => {
    process.env.PATH = path.join(testResources, "empty");
    process.chdir(path.join(testResources, "path"));
    await t.throwsAsync(index_1.safeWhich("program"));
    await t.throwsAsync(index_1.safeWhich("has-an-extension"));
    await t.throwsAsync(index_1.safeWhich("has-an-extension.exe"));
});
if (index_1.isWindows) {
    ava_1.default("program is found if on path with correct extension preference", async (t) => {
        process.env.PATH = path.join(testResources, "path");
        process.env.PATHEXT = ".com;.exe";
        t.deepEqual(await index_1.safeWhich("has-an-extension"), path.join(testResources, "path", "has-an-extension.com"));
        process.env.PATHEXT = ".exe;.com";
        t.deepEqual(await index_1.safeWhich("has-an-extension"), path.join(testResources, "path", "has-an-extension.exe"));
    });
    ava_1.default("program is not found if no extension", async (t) => {
        process.env.PATH = path.join(testResources, "path");
        await t.throwsAsync(index_1.safeWhich("program"));
    });
    ava_1.default("relative path with backward-slash is returned as-is", async (t) => {
        process.env.PATH = path.join(testResources, "path");
        t.deepEqual(await index_1.safeWhich(".\\anything"), ".\\anything");
    });
    ava_1.default("absolute path with backward-slash is returned as-is", async (t) => {
        process.env.PATH = path.join(testResources, "path");
        t.deepEqual(await index_1.safeWhich("C:\\Python27\\python.exe"), "C:\\Python27\\python.exe");
    });
    ava_1.default("path order is respected", async (t) => {
        process.env.PATHEXT = ".com;.exe;.bat";
        process.env.PATH = path.join(testResources, "path") + ";" + path.join(testResources, "second-path");
        t.deepEqual(await index_1.safeWhich("has-an-extension"), path.join(testResources, "path", "has-an-extension.com"));
        process.env.PATH = path.join(testResources, "second-path") + ";" + path.join(testResources, "path");
        t.deepEqual(await index_1.safeWhich("has-an-extension"), path.join(testResources, "second-path", "has-an-extension.bat"));
    });
}
else {
    ava_1.default("program is found if on path and executable", async (t) => {
        process.env.PATH = path.join(testResources, "path");
        t.deepEqual(await index_1.safeWhich("program"), path.join(testResources, "path", "program"));
    });
    ava_1.default("program is not found if not executable", async (t) => {
        process.env.PATH = path.join(testResources, "path");
        await t.throwsAsync(index_1.safeWhich("non-executable-file"));
    });
    ava_1.default("path order is respected", async (t) => {
        process.env.PATH = path.join(testResources, "path") + ":" + path.join(testResources, "second-path");
        t.deepEqual(await index_1.safeWhich("program"), path.join(testResources, "path", "program"));
        process.env.PATH = path.join(testResources, "second-path") + ":" + path.join(testResources, "path");
        t.deepEqual(await index_1.safeWhich("program"), path.join(testResources, "second-path", "program"));
    });
}
//# sourceMappingURL=index.test.js.map