{
 * This unit was generated automatically. It incorporates a selection of source
 * code taken from the Code Snippets Database at
 * https://github.com/delphidabbler/code-snippets.
 * 
 * The unit is copyright  2005-2025 by Peter Johnson & Contributors and is
 * licensed under the MIT License (https://opensource.org/licenses/MIT).
 * 
 * Generated on : Fri, 04 Apr 2025 08:38:43 GMT.
 * Generated by : DelphiDabbler CodeSnip Release 4.24.0.
 * 
 * The latest version of CodeSnip is available from the CodeSnip GitHub project
 * at https://github.com/delphidabbler/codesnip.
}

unit UStructCatSnippets;

interface

uses
  Types, SysUtils, Math;

{
  Encapsulates a point with double precision floating point coordinates.
}
type
  TPointF = record
    X, Y: Double;  // x and y coordinates
  end;

{
  Encapsulates a rectangle with double precision floating point size and
  position.
}
type
  TRectF = record
    case Integer of
      0: (Left, Top, Right, Bottom: Double);
      1: (TopLeft, BottomRight: TPointF);
  end;

{
  Encapsulates the upper and lower bounds of a range of values.
}
type
  TRange = record
    Lower: Integer;  // lower bound of range
    Upper: Integer;  // upper bound of range
  end;

{
  An advanced record that encapsulates an integer range along with operations on
  it.
  The range is immutable, so the constructor must be used to instantiate a
  non-empty range.
}
type
  TIntegerRange = record
  strict private
    var
      fLowerBound: Integer;
      fUpperBound: Integer;
    function GetLowerBound: Integer;
    function GetUpperBound: Integer;
    function IsSubrangeOf(const ARange: TIntegerRange): Boolean;
  public
    // Constructs a range whose bounds are A and B. The lowest of the two
    // parameters is taken as the lower bound of the range with the other
    // parameter taken as the upper bound.
    // Valid bounds must fall in the range -MaxInt..MaxInt. An
    // EArgumentException exception is raised otherwise.
    constructor Create(const A, B: Integer);

    // Constructs an empty range.
    class function CreateEmpty: TIntegerRange; static;

    // Checks if the range is empty.
    function IsEmpty: Boolean;

    // Returns the length of the range, i.e. the number of integers in the range.
    function Length: Cardinal;

    // Constrains AValue to fall within this range. If the value lies within the
    // range it is returned unchanged. If it is outside the range then either
    // LowerBound or UpperBound is returned, depending on whether the value
    // falls below or above the range, respectively.
    // An EInvalidOpException exception is raised if the range is empty.
    function Constrain(const AValue: Integer): Integer;

    // Checks if this range overlaps with ARange, i.e. the interection of the
    // ranges is non empty. Empty ranges cannot overlap with any range.
    function OverlapsWith(const ARange: TIntegerRange): Boolean;

    // Checks if this range is immediately adjacent to ARange, with no overlap.
    // Empty ranges are never contiguous with other ranges or themselves.
    function IsContiguousWith(const ARange: TIntegerRange): Boolean;

    // Checks if the set of all values in this range and ARange form a
    // continuous sequence. This implies that a range is continuous with itself.
    // Since adding an empty range to a non-empty range doesn't change the
    // non-empty range we define empty ranges to be continuous with any range.
    function IsContinuousWith(const ARange: TIntegerRange): Boolean;

    // Checks if ranges A and B are the same
    class operator Equal(const A, B: TIntegerRange): Boolean;

    // Checks if ranges A and B are not the same
    class operator NotEqual(const A, B: TIntegerRange): Boolean;

    // Checks if range A is contained in, or is the same as, range B.
    // An empty range is deemed to be contained in any other range.
    class operator LessThanOrEqual(const A, B: TIntegerRange): Boolean;

    // Checks if range A is contains, or is the same as, range B.
    // A non-empty range is never contained in an empty range.
    class operator GreaterThanOrEqual(const A, B: TIntegerRange): Boolean;

    // Combine two ranges, A and B. The result is the smallest range that
    // contains both A and B.
    // If A and B are not continuous the resulting range will contain values
    // that were not in either A or B.
    // Combining any range either with itself or with an empty range is a no-op.
    class operator Add(const A, B: TIntegerRange): TIntegerRange;

    // Returns a range that is the intersection of ranges A and B.
    // Returns an empty range if A and B do not overlap.
    class operator Multiply(const A, B: TIntegerRange): TIntegerRange;

    // Checks if integer AValue is contained within range ARange.
    class operator In(const AValue: Integer; const ARange: TIntegerRange):
      Boolean;

    // Implicitly casts ARange to a string. If ARange is non-empty the string
    // has format [X..Y], where X and Y are the lower and upper bounds of
    // ARange respectively. If ARange is empty then [] is returned.
    // This means that ARange can be assigned directly to a string.
    class operator Implicit(const ARange: TIntegerRange): string;

    // Explicitly casts ARange to a string. If ARange is non-empty the string
    // has format [X..Y], where X and Y are the lower and upper bounds of
    // ARange respectively. If ARange is empty then [] is returned.
    // This means that ARange can be explicitly cast to a string using
    // string(ARange).
    class operator Explicit(const ARange: TIntegerRange): string;

    // The lower bound of a non-empty range.
    // EInvalidOpException is raised if the property is read when the range is
    // empty.
    property LowerBound: Integer read GetLowerBound;

    // The upper bound of a non-empty range.
    // EInvalidOpException is raised if the property is read when the range is
    // empty.
    property UpperBound: Integer read GetUpperBound;
  end;

{
  Encapsulates a range of integers with a methods to test whether a value falls
  within the range and to adjust the value to fit.
}
type
  TRangeEx = record
    // Minimum and maximum bounds of range.
    Min, Max: Integer;
    // Constructs record with given minimum and maximum bounds.
    constructor Create(AMin, AMax: Integer);
    // Checks if the given value is contained within the range.
    function Contains(const Value: Integer): Boolean;
    // Adjusts the given value to lie within the range, and returns it. If the
    // value is less than Min, Min is returned. If the value is greater than Max
    // then max is returned. If the value is in the range it is returned
    // unchanged.
    function Constrain(const Value: Integer): Integer;
  end;

{
  Record that represents the size, i.e. the width and height, of something.
  This is an extended version of the TSize record that features equality and
  inequality operator overloading, a test for zero size and a constructor.
  TSizeEx is assignment compatible and comparible with the Delphi RTL's TSize
  record.
}
type
  TSizeEx = record
  public
    // Width
    CX: Integer;
    // Height
    CY: Integer;
    // Constructs record with two given CX and CY field values
    constructor Create(ACX, ACY: Integer);
    // Enables TSize to be assigned to and compared with TSizeEx
    class operator Implicit(S: Types.TSize): TSizeEx;
    // Enables TSizeEx to be assigned to and compared with TSize
    class operator Implicit(S: TSizeEx): Types.TSize;
    // Tests for equality of TSizeEx records. Also works if one record is TSize.
    class operator Equal(S1, S2: TSizeEx): Boolean;
    // Tests for inequality of TSizeEx records. Also works if one record is
    // TSize.
    class operator NotEqual(S1, S2: TSizeEx): Boolean;
    // Tests if a TSizeEx instance is zero (i.e. one of fields is zero)
    function IsZero: Boolean;
  end;

{
  Constructs and returns a TRectF record with the given top-left coordinate,
  width and height.
}
function BoundsF(ALeft, ATop, AWidth, AHeight: Double): TRectF;

{
  Constructs and returns a TPointF record with the given x & y coordinates.
}
function PointF(const AX, AY: Double): TPointF;

{
  Constructs and returns a TRange record with bounds A and B.
  The smaller of A and B is used as the lower bound and the larger as the upper
  bound. If both values are equal then the range will be empty.
}
function Range(const A, B: Integer): TRange;

{
  Constructs and returns a TRectF record with the given left, top, right &
  bottom coordinates.
}
function RectF(const ALeft, ATop, ARight, ABottom: Double): TRectF;

{
  Constructs and returns a TSize record with the given dimensions.
}
function Size(const ACX, ACY: Integer): Types.TSize;

implementation

{
  Constructs and returns a TRectF record with the given top-left coordinate,
  width and height.
}
function BoundsF(ALeft, ATop, AWidth, AHeight: Double): TRectF;
begin
  Result.Left := ALeft;
  Result.Top := ATop;
  Result.Right := ALeft + AWidth;
  Result.Bottom :=  ATop + AHeight;
end;

{
  Constructs and returns a TPointF record with the given x & y coordinates.
}
function PointF(const AX, AY: Double): TPointF;
begin
  Result.X := AX;
  Result.Y := AY;
end;

{
  Constructs and returns a TRange record with bounds A and B.
  The smaller of A and B is used as the lower bound and the larger as the upper
  bound. If both values are equal then the range will be empty.
}
function Range(const A, B: Integer): TRange;
begin
  if A <= B then
  begin
    Result.Lower := A;
    Result.Upper := B;
  end
  else
  begin
    Result.Lower := B;
    Result.Upper := A;
  end;
end;

{
  Constructs and returns a TRectF record with the given left, top, right &
  bottom coordinates.
}
function RectF(const ALeft, ATop, ARight, ABottom: Double): TRectF;
begin
  Result.Left := ALeft;
  Result.Top := ATop;
  Result.Right := ARight;
  Result.Bottom := ABottom;
end;

{
  Constructs and returns a TSize record with the given dimensions.
}
function Size(const ACX, ACY: Integer): Types.TSize;
begin
  Result.cx := ACX;
  Result.cy := ACY;
end;

class operator TIntegerRange.Add(const A, B: TIntegerRange): TIntegerRange;
begin
  if A.IsEmpty then
    Exit(B);
  if B.IsEmpty then
    Exit(A);
  Result := TIntegerRange.Create(
    Math.Min(A.fLowerBound, B.fLowerBound),
    Math.Max(A.fUpperBound, B.fUpperBound)
  );
end;

function TIntegerRange.Constrain(const AValue: Integer): Integer;
begin
  if IsEmpty then
    raise Sysutils.EInvalidOpException.Create(
      'TIntegerRange.Constrain not valid for an empty range.'
    );
  Result := Math.EnsureRange(AValue, fLowerBound, fUpperBound);
end;

constructor TIntegerRange.Create(const A, B: Integer);
begin
  // Normalise range so that smallest parameter is the lower bound
  fLowerBound := Math.Min(A, B);
  fUpperBound := Math.Max(A, B);
  if fLowerBound = Low(Integer) then
    // This restriction is required to prevent the Length method's Cardinal
    // return value from wrapping around / overflowing
    raise SysUtils.EArgumentException.CreateFmt(
      'TIntegerRange.Create: Arguments must be greater than %d', [Low(Integer)]
    );
end;

class function TIntegerRange.CreateEmpty: TIntegerRange;
begin
  Result.fLowerBound := High(Integer);
  Result.fUpperBound := Low(Integer);
end;

class operator TIntegerRange.Equal(const A, B: TIntegerRange): Boolean;
begin
  if A.IsEmpty or B.IsEmpty then
    Exit(A.IsEmpty and B.IsEmpty);
  Result := (A.fLowerBound = B.fLowerBound) and (A.fUpperBound = B.fUpperBound);
end;

class operator TIntegerRange.Explicit(const ARange: TIntegerRange): string;
begin
  if ARange.IsEmpty then
    Exit('[]');
  Result := SysUtils.Format(
    '[%d..%d]', [ARange.fLowerBound, ARange.fUpperBound]
  );
end;

function TIntegerRange.GetLowerBound: Integer;
begin
  if IsEmpty then
    raise Sysutils.EInvalidOpException.Create(
      'TIntegerRange.LowerBound not valid for an empty range.'
    );
  Result := fLowerBound;
end;

function TIntegerRange.GetUpperBound: Integer;
begin
  if IsEmpty then
    raise Sysutils.EInvalidOpException.Create(
      'TIntegerRange.LowerBound not valid for an empty range.'
    );
  Result := fUpperBound;
end;

class operator TIntegerRange.GreaterThanOrEqual(const A, B: TIntegerRange):
  Boolean;
begin
  Result := B.IsSubrangeOf(A);
end;

class operator TIntegerRange.Implicit(const ARange: TIntegerRange): string;
begin
  Result := string(ARange); // calls Explicit cast operator
end;

class operator TIntegerRange.In(const AValue: Integer;
  const ARange: TIntegerRange): Boolean;
begin
  if ARange.IsEmpty then
    Exit(False);
  Result := (AValue >= ARange.fLowerBound) and (AValue <= ARange.fUpperBound);
end;

function TIntegerRange.IsContiguousWith(const ARange: TIntegerRange): Boolean;
begin
  if Self.IsEmpty or ARange.IsEmpty then
    Exit(False);
  Result := (Self + ARange).Length = (Self.Length + ARange.Length);
end;

function TIntegerRange.IsContinuousWith(const ARange: TIntegerRange): Boolean;
begin
  if Self.IsEmpty or ARange.IsEmpty then
    // Empty ranges are only continuous with other empty ranges
    Exit(True);
  Result := IsContiguousWith(ARange) or OverlapsWith(ARange);
end;

function TIntegerRange.IsEmpty: Boolean;
begin
  Result := fLowerBound > fUpperBound;
end;

function TIntegerRange.IsSubrangeOf(const ARange: TIntegerRange): Boolean;
begin
  if ARange.IsEmpty then
    Exit(Self.IsEmpty);
  Result := (Self.fLowerBound >= ARange.fLowerBound)
    and (Self.fUpperBound <= ARange.fUpperBound)
    or Self.IsEmpty
end;

function TIntegerRange.Length: Cardinal;
begin
  if IsEmpty then
    Exit(0);
  Result := fUpperBound - fLowerBound + 1
end;

class operator TIntegerRange.LessThanOrEqual(const A, B: TIntegerRange):
  Boolean;
begin
  Result := A.IsSubrangeOf(B);
end;

class operator TIntegerRange.Multiply(const A, B: TIntegerRange): TIntegerRange;
var
  Up, Lo: Integer;
begin
  if A.IsEmpty or B.IsEmpty then
    Exit(TIntegerRange.CreateEmpty);
  Lo := Math.Max(A.fLowerBound, B.fLowerBound);
  Up := Math.Min(A.fUpperBound, B.fUpperBound);
  if Lo <= Up then
    Result := TIntegerRange.Create(Lo, Up)
  else
    Result := TIntegerRange.CreateEmpty;
end;

class operator TIntegerRange.NotEqual(const A, B: TIntegerRange): Boolean;
begin
  if A.IsEmpty or B.IsEmpty then
    Exit(A.IsEmpty <> B.IsEmpty);
  Result := (A.fLowerBound <> B.fLowerBound)
    or (A.fUpperBound <> B.fUpperBound);
end;

function TIntegerRange.OverlapsWith(const ARange: TIntegerRange): Boolean;
begin
  Result := not (Self * ARange).IsEmpty;
end;

function TRangeEx.Constrain(const Value: Integer): Integer;
begin
  if Value < Min then
    Result := Min
  else if Value > Max then
    Result := Max
  else
    Result := Value;
end;

function TRangeEx.Contains(const Value: Integer): Boolean;
begin
  Result := Math.InRange(Value, Min, Max);
end;

constructor TRangeEx.Create(AMin, AMax: Integer);
begin
  Min := AMin;
  Max := AMax;
end;

constructor TSizeEx.Create(ACX, ACY: Integer);
begin
  CX := ACX;
  CY := ACY;
end;

class operator TSizeEx.Equal(S1, S2: TSizeEx): Boolean;
begin
  // zero records are special: can be zero when only one of CX or CY is zero
  if S1.IsZero and S2.IsZero then
  begin
    Result := True;
    Exit;
  end;
  Result := (S1.CX = S1.CX) and (S1.CY = S2.CY);
end;

class operator TSizeEx.Implicit(S: Types.TSize): TSizeEx;
begin
  Result.CX := S.cx;
  Result.CY := S.cy;
end;

class operator TSizeEx.Implicit(S: TSizeEx): Types.TSize;
begin
  Result.cx := S.CX;
  Result.cy := S.CY;
end;

function TSizeEx.IsZero: Boolean;
begin
  Result := (CX = 0) or (CY = 0);
end;

class operator TSizeEx.NotEqual(S1, S2: TSizeEx): Boolean;
begin
  Result := not (S1 = S2);
end;

end.
