unit TestUHexCatSnippets;
{

Delphi DUnit Test Case
----------------------
This unit contains a skeleton test case class generated by the Test Case Wizard.
Modify the generated code to correctly setup and call the methods from the unit
being tested.

}

interface

uses
  SysUtils, TestFramework, UHexCatSnippets;

type

  // Test snippets
  TestHexCatSnippets = class(TTestCase)
  strict private
    procedure HexToBufBadHexError;
    procedure HexToBufEmptyStringError;
    procedure HexToBytesBadHexError;
    procedure HexToBytesEmptyStringError;
    procedure HexToIntBadHexError;
    procedure HexToIntEmptyStringError;
    procedure HexToInt64BadHexError;
    procedure HexToInt64EmptyStringError;
  public
    procedure SetUp; override;
    procedure TearDown; override;
  published
    procedure TestBufToHex;
    procedure TestBytesToHex;
    procedure TestByteToHex;
    procedure TestWordToHex;
    procedure TestLongWordToHex;
    procedure TestQuadWordToHex;
    procedure TestStripHexPrefix;
    procedure TestAddHexPrefix;
    procedure TestHexByteSize;
    procedure TestTryHexToInt;
    procedure TestTryHexToInt64;
    procedure TestTryHexToBuf;
    procedure TestHexToBytes;
    procedure TestHexToBuf;
    procedure TestTryHexToBytes;
    procedure TestHexToInt;
    procedure TestHexToInt64;
    procedure TestHexToIntDef;
    procedure TestHexToInt64Def;
  end;

implementation

function EqualsByteArray(const A1, A2: array of Byte): Boolean;
var
  I: Integer;
begin
  Result := Length(A1) = Length(A2);
  if Result then
  begin
    for I := Low(A1) to High(A1) do
      if A1[I] <> A2[I] then
      begin
        Result := False;
        Exit;
      end;
  end;
end;

procedure TestHexCatSnippets.HexToBufBadHexError;
const
  cBadHex = '02AG46CD';
var
  Buf: array[0..3] of Byte;
begin
  HexToBuf(cBadHex, Buf);  // should raise exception
end;

procedure TestHexCatSnippets.HexToBufEmptyStringError;
var
  Buf: array[0..3] of Byte;
begin
  HexToBuf('', Buf);  // should raise exception
end;

procedure TestHexCatSnippets.HexToBytesBadHexError;
const
  cBadHex = '02AG46CD';
var
  Bytes: TDynByteArray;
begin
  Bytes := HexToBytes(cBadHex);  // should raise exception
end;

procedure TestHexCatSnippets.HexToBytesEmptyStringError;
var
  Bytes: TDynByteArray;
begin
  Bytes := HexToBytes('');  // should raise exception
end;

procedure TestHexCatSnippets.HexToInt64BadHexError;
const
  cBadHex = '0706050403020I00';
begin
  HexToInt64(cBadHex);
end;

procedure TestHexCatSnippets.HexToInt64EmptyStringError;
begin
  HexToInt64('');
end;

procedure TestHexCatSnippets.HexToIntBadHexError;
const
  cBadHex = '03020I00';
begin
  HexToInt(cBadHex);
end;

procedure TestHexCatSnippets.HexToIntEmptyStringError;
begin
  HexToInt('');
end;

procedure TestHexCatSnippets.SetUp;
begin
end;

procedure TestHexCatSnippets.TearDown;
begin
end;

procedure TestHexCatSnippets.TestAddHexPrefix;
const
  Result = '$42FACE';    // on Delphi - BCB should test for 0x42FACE
begin
  CheckEqualsString(Result, AddHexPrefix('42FACE'));
  CheckEqualsString(Result, AddHexPrefix('0x42FACE'));
  CheckEqualsString(Result, AddHexPrefix('$42FACE'));
end;

procedure TestHexCatSnippets.TestBufToHex;
const
  cTest1: array[0..6] of AnsiChar = 'ABCDEF.';
  cResult1 = '4142434445462E';
var
  Res: string;
  Buf: Pointer;
begin
  // general case
  GetMem(Buf, SizeOf(cTest1));
  try
    Move(cTest1, Buf^, Length(cTest1));
    Res := BufToHex(Buf^, Length(cTest1));
  finally
    FreeMem(Buf);
  end;
  CheckEqualsString(cResult1, Res);
  // empty buffer case
  Res := BufToHex(cTest1, 0);
  CheckEqualsString('', Res);
end;

procedure TestHexCatSnippets.TestBytesToHex;
const
  cTest2: array[1..6] of Byte = ($DE, $14, $81, $DA, $BB, $13);
  cResult2 = 'DE1481DABB13';
var
  Empty: array of Byte;
  Res: string;
begin
  Res := BytesToHex(cTest2);
  CheckEqualsString(cResult2, Res);
  SetLength(Empty, 0);
  Res := BytesToHex(Empty);
  CheckEqualsString('', Res);
end;

procedure TestHexCatSnippets.TestByteToHex;
var
  Res: string;
begin
  Res := ByteToHex(0);
  CheckEqualsString('00', Res);
  Res := ByteToHex($FF);
  CheckEqualsString('FF', Res);
  Res := ByteToHex($0A);
  CheckEqualsString('0A', Res);
  Res := ByteToHex($42);
  CheckEqualsString('42', Res);
end;

procedure TestHexCatSnippets.TestHexByteSize;
begin
  CheckEqualsBin(3, HexByteSize('42FACE'));
  CheckEqualsBin(3, HexByteSize('0x42FACE'));
  CheckEqualsBin(3, HexByteSize('$42FACE'));
  CheckEqualsBin(0, HexByteSize(''));
  CheckEqualsBin(0, HexByteSize('0x'));
  CheckEqualsBin(0, HexByteSize('$'));
  CheckEqualsBin(1, HexByteSize('12'));
  CheckEqualsBin(2, HexByteSize('123')); // equiv to '0123'
  CheckEqualsBin(2, HexByteSize('0123'));
end;

procedure TestHexCatSnippets.TestHexToBuf;
var
  Buf: array[0..3] of Byte;
  ResBuf: array[0..3] of Byte;
const
  cGoodHex = '02AB46CD';
begin
  ResBuf[0] := $02; ResBuf[1] := $AB; ResBuf[2] := $46; ResBuf[3] := $CD;
  HexToBuf(cGoodHex, Buf);
  Check(EqualsByteArray(Buf, ResBuf), 'Unexpected result from HexToBuf');
  CheckException(HexToBufBadHexError, EConvertError);
  CheckException(HexToBufEmptyStringError, EConvertError);
end;

procedure TestHexCatSnippets.TestHexToBytes;
var
  Bytes: TDynByteArray;
  Result: TDynByteArray;
const
  cGoodHex = '02AB46CD';
begin
  SetLength(Result, 4);
  Result[0] := $02; Result[1] := $AB; Result[2] := $46; Result[3] := $CD;
  Bytes := HexToBytes(cGoodHex);
  Check(EqualsByteArray(Bytes, Result), 'Unexpected result from HexToBytes');
  CheckException(HexToBytesBadHexError, EConvertError);
  CheckException(HexToBytesEmptyStringError, EConvertError);
end;

procedure TestHexCatSnippets.TestHexToInt;
const
  cHexStr1 = '019A03BC';
  cResult1 = $019A03BC;
  cHexStr2 = '1';
  cResult2 = 1;
  cHexStr3 = '0xFF';
  cResult3 = $FF;
begin
  Check(HexToInt(cHexStr1) = cResult1,
    Format('Expected %x from HexToInt', [cResult1]));
  Check(HexToInt(cHexStr2) = cResult2,
    Format('Expected %x from HexToInt', [cResult2]));
  Check(HexToInt(cHexStr3) = cResult3,
    Format('Expected %x from HexToInt', [cResult3]));
  CheckException(HexToIntBadHexError, EConvertError);
  CheckException(HexToIntEmptyStringError, EConvertError);
end;

procedure TestHexCatSnippets.TestHexToInt64;
const
  cHexStr1 = '019A03BC09456321';
  cResult1: Int64 = $019A03BC09456321;
  cHexStr2 = '1';
  cResult2: Int64 = 1;
  cHexStr3 = '0xFF';
  cResult3: Int64 = $FF;
begin
  Check(HexToInt64(cHexStr1) = cResult1,
    Format('Expected %x from HexToInt64', [cResult1]));
  Check(HexToInt64(cHexStr2) = cResult2,
    Format('Expected %x from HexToInt64', [cResult2]));
  Check(HexToInt64(cHexStr3) = cResult3,
    Format('Expected %x from HexToInt64', [cResult3]));
  CheckException(HexToInt64BadHexError, EConvertError);
  CheckException(HexToInt64EmptyStringError, EConvertError);
end;

procedure TestHexCatSnippets.TestHexToInt64Def;
const
  cHexStr1 = '019A03BC09456321';
  cResult1: Int64 = $019A03BC09456321;
  cHexStr2 = '1';
  cResult2: Int64 = 1;
  cHexStr3 = '0xFF';
  cResult3: Int64 = $FF;
  cDefault: Int64 = -1;
  cBadHex1 = 'ABCDEFG';
  cBadHex2 = '';
begin
  Check(HexToInt64Def(cHexStr1, cDefault) = cResult1,
    Format('Expected %x from HexToInt64Def', [cResult1]));
  Check(HexToInt64Def(cHexStr2, cDefault) = cResult2,
    Format('Expected %x from HexToInt64Def', [cResult2]));
  Check(HexToInt64Def(cHexStr3, cDefault) = cResult3,
    Format('Expected %x from HexToInt64Def', [cResult3]));
  Check(HexToInt64Def(cBadHex1, cDefault) = cDefault,
    Format('Expected %x from HexToInt64Def', [cDefault]));
  Check(HexToInt64Def(cBadHex2, cDefault) = cDefault,
    Format('Expected %x from HexToInt64Def', [cDefault]));
end;

procedure TestHexCatSnippets.TestHexToIntDef;
const
  cHexStr1 = '019A03BC';
  cResult1 = $019A03BC;
  cHexStr2 = '1';
  cResult2 = 1;
  cHexStr3 = '0xFF';
  cResult3 = $FF;
  cDefault = -1;
  cBadHex1 = 'ABCDEFG';
  cBadHex2 = '';
begin
  Check(HexToIntDef(cHexStr1, cDefault) = cResult1,
    Format('Expected %x from HexToIntDef', [cResult1]));
  Check(HexToIntDef(cHexStr2, cDefault) = cResult2,
    Format('Expected %x from HexToIntDef', [cResult2]));
  Check(HexToIntDef(cHexStr3, cDefault) = cResult3,
    Format('Expected %x from HexToIntDef', [cResult3]));
  Check(HexToIntDef(cBadHex1, cDefault) = cDefault,
    Format('Expected %x from HexToIntDef', [cDefault]));
  Check(HexToIntDef(cBadHex2, cDefault) = cDefault,
    Format('Expected %x from HexToIntDef', [cDefault]));
end;

procedure TestHexCatSnippets.TestLongWordToHex;
var
  Res: string;
begin
  Res := LongWordToHex(0);
  CheckEqualsString('00000000', Res);
  Res := LongWordToHex($FFFFFFFF);
  CheckEqualsString('FFFFFFFF', Res);
  Res := LongWordToHex($42);
  CheckEqualsString('00000042', Res);
  Res := LongWordToHex($AE842);
  CheckEqualsString('000AE842', Res);
  Res := LongWordToHex($AB42BC53);
  CheckEqualsString('AB42BC53', Res);
end;

procedure TestHexCatSnippets.TestQuadWordToHex;
var
  Res: string;
begin
  Res := QuadWordToHex(0);
  CheckEqualsString('0000000000000000', Res);
  Res := QuadWordToHex(UInt64(-1));
  CheckEqualsString('FFFFFFFFFFFFFFFF', Res);
  Res := QuadWordToHex($42);
  CheckEqualsString('0000000000000042', Res);
  Res := QuadWordToHex($12345678);
  CheckEqualsString('0000000012345678', Res);
  Res := QuadWordToHex($123456789ABCDEF0);
  CheckEqualsString('123456789ABCDEF0', Res);
end;

procedure TestHexCatSnippets.TestStripHexPrefix;
const
  Result = '42FACE';
begin
  CheckEqualsString(Result, StripHexPrefix('42FACE'));
  CheckEqualsString(Result, StripHexPrefix('0x42FACE'));
  CheckEqualsString(Result, StripHexPrefix('$42FACE'));
end;

procedure TestHexCatSnippets.TestTryHexToBuf;
var
  Buf: array[0..3] of Byte;
  ResBuf: array[0..3] of Byte;
  Flag: Boolean;
const
  cGoodHex = '02AB46CD';
  cBadHex = '02AG46CD';
begin
  ResBuf[0] := $02; ResBuf[1] := $AB; ResBuf[2] := $46; ResBuf[3] := $CD;
  Flag := TryHexToBuf(cGoodHex, Buf);
  Check(Flag,
    Format('Expecting True for TryHexToBuf(''%s'')', [cGoodHex]));
  Check(EqualsByteArray(Buf, ResBuf), 'Unexpected result from TryHexToBuf');
  Flag := TryHexToBuf(cBadHex, Buf);
  Check(not Flag,
    Format('Expecting False for TryHexToBuf(''%s'')', [cBadHex]));
  Flag := TryHexToBuf('', Buf);
  Check(not Flag,
    Format('Expecting False for TryHexToBuf(''%s'')', ['']));
end;

procedure TestHexCatSnippets.TestTryHexToBytes;
var
  Bytes: TDynByteArray;
  Result: TDynByteArray;
  Flag: Boolean;
const
  cGoodHex = '02AB46CD';
  cBadHex = '02AG46CD';
begin
  SetLength(Result, 4);
  Result[0] := $02; Result[1] := $AB; Result[2] := $46; Result[3] := $CD;
  Flag := TryHexToBytes(cGoodHex, Bytes);
  Check(Flag,
    Format('Expecting True for TryHexToBytes(''%s'')', [cGoodHex]));
  Check(EqualsByteArray(Bytes, Result), 'Unexpected result from TryHexToBytes');
  Flag := TryHexToBytes(cBadHex, Bytes);
  Check(not Flag,
    Format('Expecting Flase for TryHexToBytes(''%s'')', [cBadHex]));
  Flag := TryHexToBytes('', Bytes);
  Check(not Flag,
    Format('Expecting False for TryHexToBytes(''%s'')', ['']));
end;

procedure TestHexCatSnippets.TestTryHexToInt;
const
  cGoodHexIntStr1 = '$0ACE1234';
  cGoodHexIntStr2 = 'ACE1234';    // will pad out to 0ACE1234
  cGoodHexIntResult = $ACE1234;
  cBadHexIntStr = 'ABCG1234';
var
  Flag: Boolean;
  Value: Integer;
begin
  Flag := TryHexToInt(cGoodHexIntStr1, Value);
  Check(Flag,
    Format('Expecting True for TryHexToInt(''%s'')', [cGoodHexIntStr1]));
  CheckEqualsBin(cGoodHexIntResult, Value);
  Flag := TryHexToInt(cGoodHexIntStr2, Value);
  Check(Flag,
    Format('Expecting True for TryHexToInt(''%s'')', [cGoodHexIntStr2]));
  CheckEqualsBin(cGoodHexIntResult, Value);
  Flag := TryHexToInt(cBadHexIntStr, Value);
  Check(not Flag,
    Format('Expecting False for TryHexToInt(''%s'')', [cBadHexIntStr]));
  Flag := TryHexToInt('', Value);
  Check(not Flag,
    Format('Expecting False for TryHexToInt(''%s'')', ['']));
  CheckEqualsBin(0, Value);
end;

procedure TestHexCatSnippets.TestTryHexToInt64;
const
  cGoodHexInt64Str1 = '$0ACE1234BD789C';
  cGoodHexInt64Str2 = 'ACE1234BD789C';    // will pad out to 0ACE1234BD789C
  cGoodHexInt64Result = $0ACE1234BD789C;
  cBadHexInt64Str = '12345678ABCG1234';
var
  Flag: Boolean;
  Value: Int64;
begin
  Flag := TryHexToInt64(cGoodHexInt64Str1, Value);
  Check(Flag,
    Format('Expecting True for TryHexToInt64(''%s'')', [cGoodHexInt64Str1]));
  Check(cGoodHexInt64Result = Value,
    Format('Expecting %d for TryHexToInt64(''%s'')',
      [cGoodHexInt64Result, cGoodHexInt64Str1]));
  Flag := TryHexToInt64(cGoodHexInt64Str2, Value);
  Check(Flag,
    Format('Expecting True for TryHexToInt64(''%s'')', [cGoodHexInt64Str2]));
  Check(cGoodHexInt64Result = Value,
    Format('Expecting %d for TryHexToInt64(''%s'')',
      [cGoodHexInt64Result, cGoodHexInt64Str2]));
  Flag := TryHexToInt64(cBadHexInt64Str, Value);
  Check(not Flag,
    Format('Expecting False for TryHexToInt64(''%s'')', [cBadHexInt64Str]));
end;

procedure TestHexCatSnippets.TestWordToHex;
var
  Res: string;
begin
  Res := WordToHex(0);
  CheckEqualsString('0000', Res);
  Res := WordToHex($FFFF);
  CheckEqualsString('FFFF', Res);
  Res := WordToHex($42);
  CheckEqualsString('0042', Res);
  Res := WordToHex($842);
  CheckEqualsString('0842', Res);
  Res := WordToHex($AB42);
  CheckEqualsString('AB42', Res);
end;

initialization
  // Register any test cases with the test runner
  RegisterTest(TestHexCatSnippets.Suite);
end.

