---
title: Grid
name: Grid
description: Use our powerful mobile-first flexbox grid to build layouts of all shapes and sizes thanks to a twelve column system, six default responsive tiers, Sass variables and mixins, and dozens of predefined classes.
---

## Example

CoreUI's grid system uses a series of containers, rows, and columns to layout and align content. It's built with [flexbox](https://developer.mozilla.org/en-US/docs/Web/CSS/CSS_Flexible_Box_Layout/Basic_Concepts_of_Flexbox) and is fully responsive. Below is an example and an in-depth explanation for how the grid system comes together.

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow>
      <CCol sm="auto">
        One of three columns
      </CCol>
      <CCol sm="auto">
        One of three columns
      </CCol>
      <CCol sm="auto">
        One of three columns
      </CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow>
    <CCol sm="auto">
      One of three columns
    </CCol>
    <CCol sm="auto">
      One of three columns
    </CCol>
    <CCol sm="auto">
      One of three columns
    </CCol>
  </CRow>
</CContainer>
```

The above example creates three equal-width columns across all devices and viewports using our predefined grid classes. Those columns are centered in the page with the parent `<CContainer>`.

## How it works

Breaking it down, here's how the grid system comes together:

- **Our grid supports [six responsive breakpoints](./breakpoints).**  Breakpoints are based on `min-width` media queries, meaning they affect that breakpoint and all those above it (e.g., `sm="4"` applies to `sm`, `md`, `lg`, `xl`, and `xxl`). This means you can control container and column sizing and behavior by each breakpoint.

- **Containers center and horizontally pad your content.** Use `<CContainer>` for a responsive pixel width, `<CContainer fluid>` for `width: 100%` across all viewports and devices, or a responsive container (e.g., `<CContainer md>`) for a combination of fluid and pixel widths.

- **Rows are wrappers for columns.** Each column has horizontal `padding` (called a gutter) for controlling the space between them. This `padding` is then counteracted on the rows with negative margins to ensure the content in your columns is visually aligned down the left side. Rows also support modifier classes to [uniformly apply column sizing](#row-columns) and [gutter classes](../gutters) to change the spacing of your content.

- **Columns are incredibly flexible.** There are 12 template columns available per row, allowing you to create different combinations of elements that span any number of columns. Column classes indicate the number of template columns to span (e.g., `CCol xs="4"` spans four). `width`s are set in percentages so you always have the same relative sizing.

- **Gutters are also responsive and customizable.** [Gutter classes are available](../gutters) across all breakpoints, with all the same sizes as our [margin and padding spacing](https://coreui.io/docs/utilities/spacing/). Change horizontal gutters with `xs|sm|md|lg|xl|xxl="{ gutterX: * }"` classes, vertical gutters with `xs|sm|md|lg|xl|xxl="{ gutterY: * }"`, or all gutters with `xs|sm|md|lg|xl|xxl="{ gutter: * }"` classes. `xs|sm|md|lg|xl|xxl="{ gutter: 0 }"` is also available to remove gutters.

<!-- - **Sass variables, maps, and mixins power the grid.** If you don't want to use the predefined grid classes in CoreUI, you can use our [grid's source Sass](#sass) to create your own with more semantic markup. We also include some CSS custom properties to consume these Sass variables for even greater flexibility for you. -->

Be aware of the limitations and [bugs around flexbox](https://github.com/philipwalton/flexbugs), like the [inability to use some HTML elements as flex containers](https://github.com/philipwalton/flexbugs#flexbug-9).

## Grid options

CoreUI's grid system can adapt across all six default breakpoints, and any breakpoints you customize. The six default grid tiers are as follow:

- Extra small (xs)
- Small (sm)
- Medium (md)
- Large (lg)
- Extra large (xl)
- Extra extra large (xxl)

As noted above, each of these breakpoints have their own container, unique class prefix, and modifiers. Here's how the grid changes across these breakpoints:

<table class="table mb-4">
  <thead>
    <tr>
      <th scope="col"></th>
      <th scope="col">
        xs<br/>
        <span class="fw-normal">&lt;576px</span>
      </th>
      <th scope="col">
        sm<br/>
        <span class="fw-normal">&ge;576px</span>
      </th>
      <th scope="col">
        md<br/>
        <span class="fw-normal">&ge;768px</span>
      </th>
      <th scope="col">
        lg<br/>
        <span class="fw-normal">&ge;992px</span>
      </th>
      <th scope="col">
        xl<br/>
        <span class="fw-normal">&ge;1200px</span>
      </th>
      <th scope="col">
        xxl<br/>
        <span class="fw-normal">&ge;1400px</span>
      </th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th class="text-nowrap" scope="row">Container <code class="fw-normal">max-width</code></th>
      <td>None (auto)</td>
      <td>540px</td>
      <td>720px</td>
      <td>960px</td>
      <td>1140px</td>
      <td>1320px</td>
    </tr>
    <tr>
      <th class="text-nowrap" scope="row">Class prefix</th>
      <td><code>&lt;CCol xs=&gt;</code></td>
      <td><code>&lt;CCol sm=&gt;</code></td>
      <td><code>&lt;CCol md=&gt;</code></td>
      <td><code>&lt;CCol lg=&gt;</code></td>
      <td><code>&lt;CCol xl=&gt;</code></td>
      <td><code>&lt;CCol xxl=&gt;</code></td>
    </tr>
    <tr>
      <th class="text-nowrap" scope="row"># of columns</th>
      <td colspan="6">12</td>
    </tr>
    <tr>
      <th class="text-nowrap" scope="row">Gutter width</th>
      <td colspan="6">1.5rem (.75rem on left and right)</td>
    </tr>
    <tr>
      <th class="text-nowrap" scope="row">Custom gutters</th>
      <td colspan="6"><a href="./layout/gutters">Yes</a></td>
    </tr>
    <tr>
      <th class="text-nowrap" scope="row">Nestable</th>
      <td colspan="6"><a href="#nesting">Yes</a></td>
    </tr>
    <tr>
      <th class="text-nowrap" scope="row">Column ordering</th>
      <td colspan="6"><a href="./layout/columns">Yes</a></td>
    </tr>
  </tbody>
</table>

## Auto-layout columns

Utilize breakpoint-specific column classes for easy column sizing without an explicit numbered class like `<CCol sm={6}>`.

### Equal-width

For example, here are two grid layouts that apply to every device and viewport, from `xs` to `xxl`. Add any number of unit-less classes for each breakpoint you need and every column will be the same width.

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow>
      <CCol>
        1 of 2
      </CCol>
      <CCol>
        2 of 2
      </CCol>
    </CRow>
    <CRow>
      <CCol>
        1 of 3
      </CCol>
      <CCol>
        2 of 3
      </CCol>
      <CCol>
        3 of 3
      </CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow>
    <CCol>
      1 of 2
    </CCol>
    <CCol>
      2 of 2
    </CCol>
  </CRow>
  <CRow>
    <CCol>
      1 of 3
    </CCol>
    <CCol>
      2 of 3
    </CCol>
    <CCol>
      3 of 3
    </CCol>
  </CRow>
</CContainer>
```

### Setting one column width

Auto-layout for flexbox grid columns also means you can set the width of one column and have the sibling columns automatically resize around it. You may use predefined grid classes (as shown below), grid mixins, or inline widths. Note that the other columns will resize no matter the width of the center column.

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow>
      <CCol>
        1 of 3
      </CCol>
      <CCol xs="6">
        2 of 3 (wider)
      </CCol>
      <CCol>
        3 of 3
      </CCol>
    </CRow>
    <CRow>
      <CCol>
        1 of 3
      </CCol>
      <CCol xs="6">
        2 of 3 (wider)
      </CCol>
      <CCol>
        3 of 3
      </CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow>
    <CCol>
      1 of 3
    </CCol>
    <CCol xs="6">
      2 of 3 (wider)
    </CCol>
    <CCol>
      3 of 3
    </CCol>
  </CRow>
  <CRow>
    <CCol>
      1 of 3
    </CCol>
    <CCol xs="6">
      2 of 3 (wider)
    </CCol>
    <CCol>
      3 of 3
    </CCol>
  </CRow>
</CContainer>
```

### Variable width content

Use `<CCol {breakpoint}="auto"` props to size columns based on the natural width of their content.

::: demo
<div class="docs-example-row">
  <CContainer>
    <div class="row justify-content-md-center">
      <CCol xs lg="2">
        1 of 3
      </CCol>
      <CCol md="auto">
        Variable width content
      </CCol>
      <CCol xs lg="2">
        3 of 3
      </CCol>
    </div>
    <CRow>
      <CCol>
        1 of 3
      </CCol>
      <CCol md="auto">
        Variable width content
      </CCol>
      <CCol xs lg="2">
        3 of 3
      </CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <div class="row justify-content-md-center">
    <CCol xs lg="2">
      1 of 3
    </CCol>
    <CCol md="auto">
      Variable width content
    </CCol>
    <CCol xs lg="2">
      3 of 3
    </CCol>
  </div>
  <CRow>
    <CCol>
      1 of 3
    </CCol>
    <CCol md="auto">
      Variable width content
    </CCol>
    <CCol xs lg="2">
      3 of 3
    </CCol>
  </CRow>
</CContainer>
```

## Responsive classes

CoreUI's grid includes six tiers of predefined classes for building complex responsive layouts. Customize the size of your columns on extra small, small, medium, large, or extra large devices however you see fit.

### All breakpoints

For grids that are the same from the smallest of devices to the largest, use the `<CCol>` and `<CCol xs={*}>` classes. Specify a numbered class when you need a particularly sized column; otherwise, feel free to stick to `<CCol>`.

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow>
      <CCol>col</CCol>
      <CCol>col</CCol>
      <CCol>col</CCol>
      <CCol>col</CCol>
    </CRow>
    <CRow>
      <CCol xs="8">col-8</CCol>
      <CCol xs="4">col-4</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow>
    <CCol>col</CCol>
    <CCol>col</CCol>
    <CCol>col</CCol>
    <CCol>col</CCol>
  </CRow>
  <CRow>
    <CCol xs="8">col-8</CCol>
    <CCol xs="4">col-4</CCol>
  </CRow>
</CContainer>
```

### Stacked to horizontal

Using a single set of `<CCol sm={*}>` classes, you can create a basic grid system that starts out stacked and becomes horizontal at the small breakpoint (`sm`).

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow>
      <CCol sm="8">col-sm-8</CCol>
      <CCol sm="4">col-sm-4</CCol>
    </CRow>
    <CRow>
      <CCol sm>col-sm</CCol>
      <CCol sm>col-sm</CCol>
      <CCol sm>col-sm</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow>
    <CCol sm="8">col-sm-8</CCol>
    <CCol sm="4">col-sm-4</CCol>
  </CRow>
  <CRow>
    <CCol sm>col-sm</CCol>
    <CCol sm>col-sm</CCol>
    <CCol sm>col-sm</CCol>
  </CRow>
</CContainer>
```

### Mix and match

Don't want your columns to simply stack in some grid tiers? Use a combination of different classes for each tier as needed. See the example below for a better idea of how it all works.

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow>
      <CCol md="8">.col-md-8</CCol>
      <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
    </CRow>
    <CRow>
      <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
      <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
      <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
    </CRow>
    <CRow>
      <CCol xs="6">.col-6</CCol>
      <CCol xs="6">.col-6</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow>
    <CCol md="8">.col-md-8</CCol>
    <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
  </CRow>
  <CRow>
    <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
    <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
    <CCol xs="6" md="4">.col-6 .col-md-4</CCol>
  </CRow>
  <CRow>
    <CCol xs="6">.col-6</CCol>
    <CCol xs="6">.col-6</CCol>
  </CRow>
</CContainer>
```

### Row columns

Use the responsive `:{breakpoint}="{ cols: * }"` classes to quickly set the number of columns that best render your content and layout. Whereas normal `<CCol xs="*">` classes apply to the individual columns (e.g., `<CCol xs="4">`), the row columns classes are set on the parent `<CRow>` as a shortcut. With `:{breakpoint}="{ cols: 'auto' }"` you can give the columns their natural width.

Use these row columns classes to quickly create basic grid layouts or to control your card layouts.

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow :xs="{ cols: 2 }">
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow :xs="{ cols: 2 }">
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
  </CRow>
</CContainer>
```

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow :xs="{ cols: 3 }">
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow :xs="{ cols: 3 }">
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
  </CRow>
</CContainer>
```

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow :xs="{ cols: 'auto' }">
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow :xs="{ cols: 'auto' }">
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
  </CRow>
</CContainer>
```

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow :xs="{ cols: 4 }">
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow :xs="{ cols: 4 }">
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
  </CRow>
</CContainer>
```

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow :xs="{ cols: 4 }">
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol xs="6">Column</CCol>
      <CCol>Column</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow :xs="{ cols: 4 }">
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol xs="6">Column</CCol>
    <CCol>Column</CCol>
  </CRow>
</CContainer>
```

::: demo
<div class="docs-example-row">
  <CContainer>
    <CRow :xs="{ cols: 1 }" :sm="{ cols: 2 }" :md="{ cols: 4 }">
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
      <CCol>Column</CCol>
    </CRow>
  </CContainer>
</div>
:::
```vue
<CContainer>
  <CRow :xs="{ cols: 1 }" :sm="{ cols: 2 }" :md="{ cols: 4 }">
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
    <CCol>Column</CCol>
  </CRow>
</CContainer>
```

## API

!!!include(./docs/api/grid/CContainer.api.md)!!!

!!!include(./docs/api/grid/CRow.api.md)!!!

!!!include(./docs/api/grid/CCol.api.md)!!!