import { NgClass } from '@angular/common';
import { Component, computed, input, InputSignal } from '@angular/core';

import { Colors, Shapes, Sizes, TextColors } from '../coreui.types';
import { TextColorDirective } from '../utilities';

@Component({
  selector: 'c-avatar',
  templateUrl: './avatar.component.html',
  styleUrls: ['./avatar.component.scss'],
  imports: [NgClass],
  hostDirectives: [
    {
      directive: TextColorDirective,
      inputs: ['cTextColor: textColor']
    }
  ],
  host: { class: 'avatar', '[class]': 'hostClasses()' }
})
export class AvatarComponent {
  /**
   * Sets the background color context of the component to one of CoreUI’s themed colors.
   * @type Colors
   */
  readonly color: InputSignal<Colors | undefined> = input();

  /**
   * Select the shape of the component.
   * @type Shapes
   */
  readonly shape: InputSignal<Shapes | undefined> = input();

  /**
   * Size the component small, large, or extra large.
   * @default 'md'
   */
  readonly size = input<Omit<Sizes, 'xxl'>>('');

  /**
   * The alt attribute for the img element alternate text.
   * @type string
   */
  readonly alt: InputSignal<string> = input('');

  /**
   * The src attribute for the img element.
   * @type string
   */
  readonly src: InputSignal<string | undefined> = input();

  /**
   * Sets the color context of the status indicator to one of CoreUI’s themed colors.
   * @type Colors
   */
  readonly status: InputSignal<Colors | undefined> = input();

  /**
   * Sets the text color of the component to one of CoreUI’s themed colors.
   * via TextColorDirective
   * @type TextColors
   */
  readonly textColor: InputSignal<TextColors | undefined> = input();

  readonly statusClass = computed(() => {
    return {
      'avatar-status': true,
      [`bg-${this.status()}`]: !!this.status()
    } as Record<string, boolean>;
  });

  readonly hostClasses = computed(() => {
    const size = this.size();
    const color = this.color();
    const shape = this.shape();

    return {
      avatar: true,
      [`avatar-${size}`]: !!size,
      [`bg-${color}`]: !!color,
      [`${shape}`]: !!shape
    } as Record<string, boolean>;
  });
}
