package com.contentstack.sdk;

import okhttp3.Request;
import okhttp3.ResponseBody;
import org.junit.jupiter.api.*;
import org.json.JSONArray;
import org.json.JSONObject;
import retrofit2.Response;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.*;

class TestCSHttpConnection {

    private CSHttpConnection connection;
    private MockIRequestModelHTTP mockRequest;

    static class MockIRequestModelHTTP implements IRequestModelHTTP {
        public JSONObject error;
        public int statusCode;
        public boolean requestFinishedCalled = false;

        @Override
        public void sendRequest() {
            // Do nothing
        }

        @Override
        public void onRequestFailed(JSONObject error, int statusCode, ResultCallBack callBackObject) {
            this.error = error;
            this.statusCode = statusCode;
        }

        @Override
        public void onRequestFinished(CSHttpConnection request) {
            requestFinishedCalled = true;
        }
    }

    @BeforeEach
    void setUp() {
        mockRequest = new MockIRequestModelHTTP();
        connection = new CSHttpConnection("https://api.contentstack.io/test", mockRequest);
    }

    @Test
    void testValidJsonErrorResponse() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError("{\"error_message\": \"Invalid API Key\", \"error_code\": \"401\"}");

        Assertions.assertEquals("Invalid API Key", csConnectionRequest.error.getString("error_message"));
        Assertions.assertEquals(401, csConnectionRequest.error.getInt("error_code"));
    }

    @Test
    void testInvalidJsonErrorResponse() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError("This is not a JSON");

        Assertions.assertEquals("This is not a JSON", csConnectionRequest.error.getString("error_message"));
        Assertions.assertEquals(0, csConnectionRequest.statusCode);
    }

    @Test
    void testMissingErrorMessageField() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError("{\"error_code\": \"500\"}");

        Assertions.assertTrue(csConnectionRequest.error.has("error_message"));
        Assertions.assertEquals("An unknown error occurred.", csConnectionRequest.error.optString("error_message", "An unknown error occurred"));
        Assertions.assertEquals(500, csConnectionRequest.error.getInt("error_code"));
    }

    @Test
    void testMissingErrorCodeField() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError("{\"error_message\": \"Server error\"}");

        Assertions.assertEquals("Server error", csConnectionRequest.error.getString("error_message"));
        Assertions.assertEquals(0, csConnectionRequest.statusCode); // Default value when error_code is missing
    }

    @Test
    void testCompletelyEmptyErrorResponse() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError("{}");

        Assertions.assertEquals("An unknown error occurred.", csConnectionRequest.error.optString("error_message", "An unknown error occurred"));
        Assertions.assertEquals(0, csConnectionRequest.statusCode);
    }

    @Test
    void testNullErrorResponse() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError(null);

        Assertions.assertEquals("Unexpected error: No response received from server.", csConnectionRequest.error.getString("error_message"));
        Assertions.assertEquals(0, csConnectionRequest.statusCode);
    }

    @Test
    void testErrorResponseWithAdditionalFields() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();

        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        connection.setError("{\"error_message\": \"Bad Request\", \"error_code\": \"400\", \"errors\": \"Missing parameter\"}");

        Assertions.assertEquals("Bad Request", csConnectionRequest.error.getString("error_message"));
        Assertions.assertEquals(400, csConnectionRequest.error.getInt("error_code"));
        Assertions.assertEquals("Missing parameter", csConnectionRequest.error.getString("errors"));
    }

    @Test
    void testErrorResponseWithNonNumericErrorCode() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();
        CSHttpConnection connection = new CSHttpConnection("https://www.example.com", csConnectionRequest);
        
        // This should trigger NumberFormatException
        connection.setError("{\"error_message\": \"Bad Request\", \"error_code\": \"invalid_code\"}");
        
        assertEquals("Bad Request", csConnectionRequest.error.getString("error_message"));
        assertEquals(0, csConnectionRequest.statusCode); // Should default to 0 when parsing fails
    }

    // ========== GETTER TESTS ==========

    @Test
    void testGetHeaders() {
        LinkedHashMap<String, Object> headers = new LinkedHashMap<>();
        headers.put("api_key", "test_key");
        headers.put("access_token", "test_token");
        
        connection.setHeaders(headers);
        
        LinkedHashMap<String, Object> retrievedHeaders = connection.getHeaders();
        assertNotNull(retrievedHeaders);
        assertEquals("test_key", retrievedHeaders.get("api_key"));
        assertEquals("test_token", retrievedHeaders.get("access_token"));
    }

    @Test
    void testGetInfo() {
        connection.setInfo("TEST_REQUEST");
        
        String info = connection.getInfo();
        assertNotNull(info);
        assertEquals("TEST_REQUEST", info);
    }

    @Test
    void testGetController() {
        connection.setController("QUERY");
        
        String controller = connection.getController();
        assertNotNull(controller);
        assertEquals("QUERY", controller);
    }

    // ========== SET FORM PARAMS GET TESTS ==========

    @Test
    void testSetFormParamsGETWithQueryController() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("environment", "production");
        params.put("locale", "en-us");
        
        String result = connection.setFormParamsGET(params);
        
        assertNotNull(result);
        assertTrue(result.startsWith("?"));
        assertTrue(result.contains("environment=production"));
        assertTrue(result.contains("locale=en-us"));
    }

    @Test
    void testSetFormParamsGETWithEntryController() throws Exception {
        connection.setInfo("ENTRY");
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("include_count", "true");
        
        String result = connection.setFormParamsGET(params);
        
        assertNotNull(result);
        assertTrue(result.startsWith("?"));
    }

    @Test
    void testSetFormParamsGETWithOtherController() {
        connection.setInfo("ASSET");
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("key1", "value1");
        params.put("key2", "value2");
        
        String result = connection.setFormParamsGET(params);
        
        assertNotNull(result);
        assertTrue(result.contains("key1=value1"));
        assertTrue(result.contains("key2=value2"));
    }

    @Test
    void testSetFormParamsGETWithNullParams() {
        String result = connection.setFormParamsGET(null);
        assertNull(result);
    }

    @Test
    void testSetFormParamsGETWithEmptyParams() {
        HashMap<String, Object> params = new HashMap<>();
        String result = connection.setFormParamsGET(params);
        assertNull(result);
    }

    // ========== GET PARAMS TESTS (via reflection) ==========

    @Test
    void testGetParamsWithIncludeArray() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        JSONArray includeArray = new JSONArray();
        includeArray.put("title");
        includeArray.put("description");
        params.put("include[]", includeArray);
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("title"));
        assertTrue(result.contains("description"));
    }

    @Test
    void testGetParamsWithOnlyBaseArray() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        JSONArray onlyArray = new JSONArray();
        onlyArray.put("uid");
        onlyArray.put("title");
        params.put("only[BASE][]", onlyArray);
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("uid"));
        assertTrue(result.contains("title"));
    }

    @Test
    void testGetParamsWithExceptBaseArray() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        JSONArray exceptArray = new JSONArray();
        exceptArray.put("created_at");
        exceptArray.put("updated_at");
        params.put("except[BASE][]", exceptArray);
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("created_at"));
        assertTrue(result.contains("updated_at"));
    }

    @Test
    void testGetParamsWithOnlyObject() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        JSONObject onlyJSON = new JSONObject();
        JSONArray fieldsArray = new JSONArray();
        fieldsArray.put("title");
        fieldsArray.put("description");
        onlyJSON.put("reference_field", fieldsArray);
        params.put("only", onlyJSON);
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("only"));
        assertTrue(result.contains("reference_field"));
    }

    @Test
    void testGetParamsWithExceptObject() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        JSONObject exceptJSON = new JSONObject();
        JSONArray fieldsArray = new JSONArray();
        fieldsArray.put("large_field");
        fieldsArray.put("metadata");
        exceptJSON.put("reference_field", fieldsArray);
        params.put("except", exceptJSON);
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("except"));
        assertTrue(result.contains("reference_field"));
    }

    @Test
    void testGetParamsWithQueryObject() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        JSONObject queryJSON = new JSONObject();
        queryJSON.put("title", "Test Title");
        queryJSON.put("status", "published");
        params.put("query", queryJSON);
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("query="));
    }

    @Test
    void testGetParamsWithRegularKeyValue() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("environment", "production");
        params.put("locale", "en-us");
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("environment=production"));
        assertTrue(result.contains("locale=en-us"));
    }

    @Test
    void testGetParamsWithMultipleTypes() throws Exception {
        connection.setInfo("QUERY");
        
        HashMap<String, Object> params = new HashMap<>();
        
        // Add include[]
        JSONArray includeArray = new JSONArray();
        includeArray.put("title");
        params.put("include[]", includeArray);
        
        // Add regular param
        params.put("environment", "staging");
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        assertTrue(result.contains("title"));
        assertTrue(result.contains("environment=staging"));
    }

    // ========== CONVERT URL PARAM TESTS ==========

    @Test
    void testConvertUrlParam() throws Exception {
        Method convertUrlParamMethod = CSHttpConnection.class.getDeclaredMethod("convertUrlParam", String.class, Object.class, String.class);
        convertUrlParamMethod.setAccessible(true);
        
        JSONArray array = new JSONArray();
        array.put("field1");
        array.put("field2");
        array.put("field3");
        
        String result = (String) convertUrlParamMethod.invoke(connection, "?", array, "include[]");
        
        assertNotNull(result);
        assertTrue(result.contains("field1"));
        assertTrue(result.contains("field2"));
        assertTrue(result.contains("field3"));
    }

    @Test
    void testConvertUrlParamWithExistingParams() throws Exception {
        Method convertUrlParamMethod = CSHttpConnection.class.getDeclaredMethod("convertUrlParam", String.class, Object.class, String.class);
        convertUrlParamMethod.setAccessible(true);
        
        JSONArray array = new JSONArray();
        array.put("value1");
        array.put("value2");
        
        String result = (String) convertUrlParamMethod.invoke(connection, "?existing=param&", array, "test_key");
        
        assertNotNull(result);
        assertTrue(result.contains("value1"));
        assertTrue(result.contains("value2"));
        assertTrue(result.contains("&"));
    }

    // ========== CREATE ORDERED JSON OBJECT TEST ==========

    @Test
    void testCreateOrderedJSONObject() throws Exception {
        Method createOrderedJSONObjectMethod = CSHttpConnection.class.getDeclaredMethod("createOrderedJSONObject", Map.class);
        createOrderedJSONObjectMethod.setAccessible(true);
        
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("uid", "test_uid");
        map.put("title", "Test Title");
        map.put("count", 42);
        map.put("active", true);
        
        JSONObject result = (JSONObject) createOrderedJSONObjectMethod.invoke(connection, map);
        
        assertNotNull(result);
        assertEquals("test_uid", result.getString("uid"));
        assertEquals("Test Title", result.getString("title"));
        assertEquals(42, result.getInt("count"));
        assertEquals(true, result.getBoolean("active"));
    }

    @Test
    void testCreateOrderedJSONObjectWithEmptyMap() throws Exception {
        Method createOrderedJSONObjectMethod = CSHttpConnection.class.getDeclaredMethod("createOrderedJSONObject", Map.class);
        createOrderedJSONObjectMethod.setAccessible(true);
        
        Map<String, Object> map = new LinkedHashMap<>();
        
        JSONObject result = (JSONObject) createOrderedJSONObjectMethod.invoke(connection, map);
        
        assertNotNull(result);
        assertEquals(0, result.length());
    }

    @Test
    void testCreateOrderedJSONObjectWithNestedObjects() throws Exception {
        Method createOrderedJSONObjectMethod = CSHttpConnection.class.getDeclaredMethod("createOrderedJSONObject", Map.class);
        createOrderedJSONObjectMethod.setAccessible(true);
        
        Map<String, Object> innerMap = new LinkedHashMap<>();
        innerMap.put("nested_key", "nested_value");
        
        Map<String, Object> map = new LinkedHashMap<>();
        map.put("outer_key", "outer_value");
        map.put("nested", innerMap);
        
        JSONObject result = (JSONObject) createOrderedJSONObjectMethod.invoke(connection, map);
        
        assertNotNull(result);
        assertEquals("outer_value", result.getString("outer_key"));
        assertTrue(result.has("nested"));
    }

    // ========== HANDLE JSON ARRAY TESTS (Live Preview) ==========

    @Test
    void testHandleJSONArrayWithEntries() throws Exception {
        // Create a config with livePreviewEntry
        Config config = new Config();
        JSONObject livePreviewEntry = new JSONObject();
        livePreviewEntry.put("uid", "preview_uid");
        livePreviewEntry.put("title", "Preview Title");
        config.setLivePreviewEntry(livePreviewEntry);
        
        connection.setConfig(config);
        
        // Create response with entries
        JSONObject responseJSON = new JSONObject();
        JSONArray entries = new JSONArray();
        
        JSONObject entry1 = new JSONObject();
        entry1.put("uid", "preview_uid"); // Matches live preview
        entry1.put("title", "Original Title");
        entries.put(entry1);
        
        JSONObject entry2 = new JSONObject();
        entry2.put("uid", "other_uid");
        entry2.put("title", "Other Title");
        entries.put(entry2);
        
        responseJSON.put("entries", entries);
        
        // Set responseJSON via reflection
        Field responseField = CSHttpConnection.class.getDeclaredField("responseJSON");
        responseField.setAccessible(true);
        responseField.set(connection, responseJSON);
        
        // Call handleJSONArray
        Method handleJSONArrayMethod = CSHttpConnection.class.getDeclaredMethod("handleJSONArray");
        handleJSONArrayMethod.setAccessible(true);
        handleJSONArrayMethod.invoke(connection);
        
        // Get the updated responseJSON
        JSONObject updatedResponse = (JSONObject) responseField.get(connection);
        
        assertNotNull(updatedResponse);
        assertTrue(updatedResponse.has("entries"));
    }

    @Test
    void testHandleJSONArrayWithSingleEntry() throws Exception {
        // Create a config with livePreviewEntry
        Config config = new Config();
        JSONObject livePreviewEntry = new JSONObject();
        livePreviewEntry.put("uid", "preview_uid");
        livePreviewEntry.put("title", "Preview Title");
        config.setLivePreviewEntry(livePreviewEntry);
        
        connection.setConfig(config);
        
        // Create response with single entry
        JSONObject responseJSON = new JSONObject();
        JSONObject entry = new JSONObject();
        entry.put("uid", "preview_uid"); // Matches live preview
        entry.put("title", "Original Title");
        responseJSON.put("entry", entry);
        
        // Set responseJSON via reflection
        Field responseField = CSHttpConnection.class.getDeclaredField("responseJSON");
        responseField.setAccessible(true);
        responseField.set(connection, responseJSON);
        
        // Call handleJSONArray
        Method handleJSONArrayMethod = CSHttpConnection.class.getDeclaredMethod("handleJSONArray");
        handleJSONArrayMethod.setAccessible(true);
        handleJSONArrayMethod.invoke(connection);
        
        // Get the updated responseJSON
        JSONObject updatedResponse = (JSONObject) responseField.get(connection);
        
        assertNotNull(updatedResponse);
        assertTrue(updatedResponse.has("entry"));
    }

    @Test
    void testHandleJSONObjectWithMatchingUid() throws Exception {
        // Create a config with livePreviewEntry
        Config config = new Config();
        JSONObject livePreviewEntry = new JSONObject();
        livePreviewEntry.put("uid", "preview_uid");
        livePreviewEntry.put("title", "Preview Title");
        config.setLivePreviewEntry(livePreviewEntry);
        
        connection.setConfig(config);
        
        JSONArray arrayEntry = new JSONArray();
        JSONObject jsonObj = new JSONObject();
        jsonObj.put("uid", "preview_uid");
        jsonObj.put("title", "Original Title");
        arrayEntry.put(jsonObj);
        
        // Call handleJSONObject via reflection
        Method handleJSONObjectMethod = CSHttpConnection.class.getDeclaredMethod("handleJSONObject", JSONArray.class, JSONObject.class, int.class);
        handleJSONObjectMethod.setAccessible(true);
        handleJSONObjectMethod.invoke(connection, arrayEntry, jsonObj, 0);
        
        // Verify responseJSON was updated
        Field responseField = CSHttpConnection.class.getDeclaredField("responseJSON");
        responseField.setAccessible(true);
        JSONObject updatedResponse = (JSONObject) responseField.get(connection);
        
        assertNotNull(updatedResponse);
    }

    @Test
    void testHandleJSONObjectWithNonMatchingUid() throws Exception {
        // Create a config with livePreviewEntry
        Config config = new Config();
        JSONObject livePreviewEntry = new JSONObject();
        livePreviewEntry.put("uid", "preview_uid");
        livePreviewEntry.put("title", "Preview Title");
        config.setLivePreviewEntry(livePreviewEntry);
        
        connection.setConfig(config);
        
        JSONArray arrayEntry = new JSONArray();
        JSONObject jsonObj = new JSONObject();
        jsonObj.put("uid", "different_uid"); // Does NOT match
        jsonObj.put("title", "Original Title");
        arrayEntry.put(jsonObj);
        
        // Call handleJSONObject via reflection
        Method handleJSONObjectMethod = CSHttpConnection.class.getDeclaredMethod("handleJSONObject", JSONArray.class, JSONObject.class, int.class);
        handleJSONObjectMethod.setAccessible(true);
        handleJSONObjectMethod.invoke(connection, arrayEntry, jsonObj, 0);
        
        // Verify responseJSON was still updated (even though uid doesn't match, responseJSON is set)
        Field responseField = CSHttpConnection.class.getDeclaredField("responseJSON");
        responseField.setAccessible(true);
        JSONObject updatedResponse = (JSONObject) responseField.get(connection);
        
        assertNotNull(updatedResponse);
    }

    @Test
    void testHandleJSONObjectWithEmptyObject() throws Exception {
        Config config = new Config();
        JSONObject livePreviewEntry = new JSONObject();
        livePreviewEntry.put("uid", "preview_uid");
        config.setLivePreviewEntry(livePreviewEntry);
        
        connection.setConfig(config);
        
        JSONArray arrayEntry = new JSONArray();
        JSONObject jsonObj = new JSONObject(); // Empty object
        arrayEntry.put(jsonObj);
        
        // Call handleJSONObject via reflection
        Method handleJSONObjectMethod = CSHttpConnection.class.getDeclaredMethod("handleJSONObject", JSONArray.class, JSONObject.class, int.class);
        handleJSONObjectMethod.setAccessible(true);
        
        // Should not throw exception even with empty object
        assertDoesNotThrow(() -> {
            try {
                handleJSONObjectMethod.invoke(connection, arrayEntry, jsonObj, 0);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    // ========== SEND METHOD TESTS ==========
    // Note: The send() method and getService() make actual network calls via Retrofit.
    // These methods are covered by integration tests. Unit tests can only verify
    // basic setup and exception handling paths.

    // ========== SETTER TESTS FOR COVERAGE ==========

    @Test
    void testSetAPIService() throws IllegalAccessException {
        Stack stack = Contentstack.stack("test_key", "test_token", "test_env");
        
        // Create an APIService instance (this is used for actual network calls)
        assertDoesNotThrow(() -> connection.setAPIService(stack.service));
    }

    @Test
    void testSetStack() throws IllegalAccessException {
        Stack stack = Contentstack.stack("test_key", "test_token", "test_env");
        
        assertDoesNotThrow(() -> connection.setStack(stack));
    }

    @Test
    void testSetConfig() throws IllegalAccessException {
        Stack stack = Contentstack.stack("test_key", "test_token", "test_env");
        Config config = stack.config;
        
        connection.setConfig(config);
        
        // Verify config was set by checking if we can use it
        assertNotNull(config);
    }

    @Test
    void testSetCallBackObject() {
        ResultCallBack callback = new ResultCallBack() {
            @Override
            public void onRequestFail(ResponseType responseType, Error error) {
                // Mock implementation
            }
        };
        
        connection.setCallBackObject(callback);
        ResultCallBack retrieved = connection.getCallBackObject();
        
        assertNotNull(retrieved);
        assertEquals(callback, retrieved);
    }

    @Test
    void testGetResponse() {
        // Initially, response should be null
        JSONObject response = connection.getResponse();
        
        // Response is null until a request is made
        assertNull(response);
    }

    // ========== PLUGIN-RELATED TESTS ==========
    // Note: Plugin methods (pluginRequestImp and pluginResponseImp) are called within
    // getService(), which makes actual network calls. These are covered by integration tests.

    @Test
    void testPluginCreation() throws Exception {
        Stack stack = Contentstack.stack("test_key", "test_token", "test_env");
        
        // Create a mock plugin
        ContentstackPlugin mockPlugin = new ContentstackPlugin() {
            @Override
            public Request onRequest(Stack stack, Request request) {
                // Return the request (default plugin behavior)
                return request;
            }
            
            @Override
            public Response<ResponseBody> onResponse(Stack stack, Request request, Response<ResponseBody> response) {
                // Return the response (default plugin behavior)
                return response;
            }
        };
        
        // Set up config with plugin
        Config config = stack.config;
        java.util.List<ContentstackPlugin> plugins = new java.util.ArrayList<>();
        plugins.add(mockPlugin);
        config.setPlugins(plugins);
        
        // Verify plugin was added
        assertNotNull(config.plugins);
        assertEquals(1, config.plugins.size());
        assertEquals(mockPlugin, config.plugins.get(0));
    }

    // ========== EXCEPTION HANDLING TESTS ==========
    // Note: Exception handling in getService() and send() requires actual network calls
    // and is covered by integration tests.

    @Test
    void testFormParamsHandling() {
        HashMap<String, Object> params = new HashMap<>();
        params.put("environment", "production");
        params.put("locale", "en-us");
        params.put("include_count", true);
        
        connection.setFormParams(params);
        connection.setInfo("QUERY");
        
        // Verify that form params are processed correctly
        String result = connection.setFormParamsGET(params);
        
        assertNotNull(result);
        assertTrue(result.contains("environment=production"));
    }

    @Test
    void testConnectionWithAllFieldsSet() throws IllegalAccessException {
        Stack stack = Contentstack.stack("test_key", "test_token", "test_env");
        
        connection.setController("QUERY");
        connection.setInfo("TEST_INFO");
        connection.setConfig(stack.config);
        connection.setStack(stack);
        connection.setAPIService(stack.service);
        
        LinkedHashMap<String, Object> headers = new LinkedHashMap<>();
        headers.put("api_key", "test_key");
        headers.put("access_token", "test_token");
        connection.setHeaders(headers);
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("environment", "test_env");
        connection.setFormParams(params);
        
        ResultCallBack callback = new ResultCallBack() {
            @Override
            public void onRequestFail(ResponseType responseType, Error error) {}
        };
        connection.setCallBackObject(callback);
        
        // Verify all getters return expected values
        assertEquals("QUERY", connection.getController());
        assertEquals("TEST_INFO", connection.getInfo());
        assertNotNull(connection.getHeaders());
        assertNotNull(connection.getCallBackObject());
        
        // Note: send() is not called here as it requires actual network infrastructure
        // The complete flow with send() is covered by integration tests
    }

    // ========== ADDITIONAL BRANCH COVERAGE TESTS ==========

    @Test
    void testSetFormParamsGETWithNullResult() {
        HashMap<String, Object> params = null;
        
        String result = connection.setFormParamsGET(params);
        
        assertNull(result);
    }

    @Test
    void testSetFormParamsGETWithEmptyParamsReturnsNull() {
        HashMap<String, Object> params = new HashMap<>();
        
        String result = connection.setFormParamsGET(params);
        
        assertNull(result);
    }

    @Test
    void testSetFormParamsGETWithNonQueryNonEntryController() {
        connection.setInfo("ASSET");
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("key1", "value1");
        params.put("key2", "value2");
        
        String result = connection.setFormParamsGET(params);
        
        assertNotNull(result);
        assertTrue(result.contains("key1=value1"));
        assertTrue(result.contains("key2=value2"));
    }

    @Test
    void testSetFormParamsGETWithMultipleParams() {
        connection.setInfo("OTHER");
        
        HashMap<String, Object> params = new HashMap<>();
        params.put("param1", "value1");
        params.put("param2", "value2");
        params.put("param3", "value3");
        
        String result = connection.setFormParamsGET(params);
        
        assertNotNull(result);
        assertTrue(result.startsWith("?"));
        assertTrue(result.contains("param1=value1"));
        assertTrue(result.contains("&"));
    }

    @Test
    void testGetParamsExceptionHandling() throws Exception {
        connection.setInfo("QUERY");
        
        // Create a params map with a value that will cause encoding issues
        HashMap<String, Object> params = new HashMap<>();
        
        // Add a mock object that will cause ClassCastException when treated as JSONObject
        params.put("query", new Object() {
            @Override
            public String toString() {
                return "{invalid}";
            }
        });
        
        Method getParamsMethod = CSHttpConnection.class.getDeclaredMethod("getParams", HashMap.class);
        getParamsMethod.setAccessible(true);
        
        // This should handle the exception and log it, returning a partial URL
        String result = (String) getParamsMethod.invoke(connection, params);
        
        assertNotNull(result);
        // The method should continue despite the exception
        assertTrue(result.startsWith("?"));
    }

    @Test
    void testSendWithNullParams() {
        connection.setInfo("QUERY");
        connection.setFormParams(null);
        
        // Verify send can be called with null params without throwing
        // Note: This will fail at network call, but that's expected in unit test
        assertDoesNotThrow(() -> {
            try {
                // Setup minimal required fields
                LinkedHashMap<String, Object> headers = new LinkedHashMap<>();
                headers.put("api_key", "test");
                connection.setHeaders(headers);
                
                Stack stack = Contentstack.stack("test", "test", "test");
                connection.setConfig(stack.config);
                connection.setAPIService(stack.service);
                connection.setStack(stack);
                
                // This will fail at network level, but params handling is tested
                connection.send();
            } catch (Exception e) {
                // Expected - network call will fail in unit test
            }
        });
    }

    @Test
    void testSendWithEmptyParams() {
        connection.setInfo("QUERY");
        connection.setFormParams(new HashMap<>());
        
        assertDoesNotThrow(() -> {
            try {
                LinkedHashMap<String, Object> headers = new LinkedHashMap<>();
                headers.put("api_key", "test");
                connection.setHeaders(headers);
                
                Stack stack = Contentstack.stack("test", "test", "test");
                connection.setConfig(stack.config);
                connection.setAPIService(stack.service);
                connection.setStack(stack);
                
                connection.send();
            } catch (Exception e) {
                // Expected
            }
        });
    }

    @Test
    void testConvertUrlParamWithSingleElement() throws Exception {
        Method convertUrlParamMethod = CSHttpConnection.class.getDeclaredMethod("convertUrlParam", 
            String.class, Object.class, String.class);
        convertUrlParamMethod.setAccessible(true);
        
        JSONArray array = new JSONArray();
        array.put("single_value");
        
        String result = (String) convertUrlParamMethod.invoke(connection, "?", array, "test_key");
        
        assertNotNull(result);
        assertTrue(result.contains("test_key=single_value"));
    }

    @Test
    void testCreateOrderedJSONObjectWithMultipleEntries() throws Exception {
        Method createOrderedMethod = CSHttpConnection.class.getDeclaredMethod("createOrderedJSONObject", Map.class);
        createOrderedMethod.setAccessible(true);
        
        LinkedHashMap<String, Object> map = new LinkedHashMap<>();
        map.put("key1", "value1");
        map.put("key2", 123);
        map.put("key3", true);
        map.put("key4", "value4");
        
        JSONObject result = (JSONObject) createOrderedMethod.invoke(connection, map);
        
        assertNotNull(result);
        assertEquals("value1", result.get("key1"));
        assertEquals(123, result.get("key2"));
        assertEquals(true, result.get("key3"));
        assertEquals("value4", result.get("key4"));
        assertEquals(4, result.length());
    }

    @Test
    void testSetErrorWithEmptyString() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();
        CSHttpConnection conn = new CSHttpConnection("https://test.com", csConnectionRequest);
        
        conn.setError("");
        
        assertNotNull(csConnectionRequest.error);
        assertTrue(csConnectionRequest.error.has("error_message"));
        String errorMsg = csConnectionRequest.error.getString("error_message");
        assertEquals("Unexpected error: No response received from server.", errorMsg);
    }

    @Test
    void testSetErrorWithWhitespaceOnly() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();
        CSHttpConnection conn = new CSHttpConnection("https://test.com", csConnectionRequest);
        
        conn.setError("   ");
        
        assertNotNull(csConnectionRequest.error);
        assertTrue(csConnectionRequest.error.has("error_message"));
    }

    @Test
    void testSetErrorWithValidJSONButMissingAllFields() {
        MockIRequestModelHTTP csConnectionRequest = new MockIRequestModelHTTP();
        CSHttpConnection conn = new CSHttpConnection("https://test.com", csConnectionRequest);
        
        conn.setError("{\"some_field\": \"some_value\"}");
        
        assertNotNull(csConnectionRequest.error);
        assertEquals("An unknown error occurred.", csConnectionRequest.error.getString("error_message"));
        assertEquals("0", csConnectionRequest.error.getString("error_code"));
        assertEquals("No additional error details available.", csConnectionRequest.error.getString("errors"));
    }
}
