import path from 'path'
import libsass from 'sass'
import cssnano from 'cssnano'
import autoprefixer from 'autoprefixer'
import rgbaCompat from 'postcss-hexrgba'
import hslCompat from 'postcss-color-hsl'
import MiniCssExtractPlugin from 'mini-css-extract-plugin'
import RemoveEmptyScriptsPlugin from 'webpack-remove-empty-scripts'
import { glob } from 'glob'
import { RtlCssPlugin } from './RtlCssPlugin'
import type { Configuration, EntryObject } from 'webpack'
import type { Config as PostCssConfig } from 'postcss-load-config'

const postcssOptions: PostCssConfig = {
	plugins: [
		rgbaCompat(),
		hslCompat(),
		autoprefixer(),
		cssnano({
			preset: ['default', { discardComments: { removeAll: true } }]
		})
	]
}

const entriesFromFiles = (patterns: string | string[], entry: (filename: string) => string): EntryObject =>
	Object.fromEntries(
		glob.sync(patterns)
			.map(filename => [entry(filename), `./${filename}`])
	)

export const cssWebpackConfig: Configuration = {
	entry: {
		...entriesFromFiles(
			['src/css/*.scss', '!src/css/**/_*.scss'],
			filename => `${path.parse(filename).name}-css`
		),
		...entriesFromFiles(
			'node_modules/codemirror/theme/*.css',
			filename => `codemirror-theme-${path.parse(filename).name}`
		)
	},
	module: {
		rules: [
			{
				test: /\.scss$/,
				exclude: /node_modules/,
				use: [
					MiniCssExtractPlugin.loader,
					{
						loader: 'css-loader',
						options: {
							sourceMap: true,
							importLoaders: 2
						}
					},
					{
						loader: 'postcss-loader',
						options: {
							postcssOptions
						}
					},
					{
						loader: 'sass-loader',
						options: {
							implementation: libsass,
							sourceMap: true,
							sassOptions: {
								loadPaths: ['node_modules']
							}
						}
					}
				]
			},
			{
				test: /\.css$/,
				use: [
					MiniCssExtractPlugin.loader,
					{
						loader: 'css-loader',
						options: {
							sourceMap: false,
							importLoaders: 1
						}
					},
					{
						loader: 'postcss-loader',
						options: {
							postcssOptions: {
								plugins: [cssnano()]
							}
						}
					}
				]
			}
		]
	},
	plugins: [
		new RemoveEmptyScriptsPlugin(),
		new MiniCssExtractPlugin({
			filename: ({ chunk }) =>
				chunk?.name
					? `${chunk.name}.css`
						.replace(/^codemirror-theme-/, 'editor-themes/')
						.replace(/-css\.css$/, '.css')
					: '[name].css'
		}),
		new RtlCssPlugin({
			entries: new Set(['manage-css', 'edit-css'])
		})
	]
}
