package com.coder.toolbox.sdk.interceptors

import com.coder.toolbox.CoderToolboxContext
import com.coder.toolbox.util.getArch
import com.coder.toolbox.util.getHeaders
import com.coder.toolbox.util.getOS
import okhttp3.Interceptor
import java.net.URL

/**
 * Factory of okhttp interceptors
 */
object Interceptors {

    /**
     * Creates a token authentication interceptor
     */
    fun tokenAuth(token: String): Interceptor {
        return Interceptor { chain ->
            chain.proceed(
                chain.request().newBuilder()
                    .addHeader("Coder-Session-Token", token)
                    .build()
            )
        }
    }

    /**
     * Creates a User-Agent header interceptor
     */
    fun userAgent(pluginVersion: String): Interceptor {
        return Interceptor { chain ->
            chain.proceed(
                chain.request().newBuilder()
                    .addHeader("User-Agent", "Coder Toolbox/$pluginVersion (${getOS()}; ${getArch()})")
                    .build()
            )
        }
    }

    /**
     * Adds headers generated by executing a native command
     */
    fun externalHeaders(context: CoderToolboxContext, url: URL): Interceptor {
        val settings = context.settingsStore.readOnly()
        return Interceptor { chain ->
            var request = chain.request()
            val headers = getHeaders(url, settings.headerCommand)
            if (headers.isNotEmpty()) {
                val reqBuilder = request.newBuilder()
                headers.forEach { h -> reqBuilder.addHeader(h.key, h.value) }
                request = reqBuilder.build()
            }
            chain.proceed(request)
        }
    }

    /**
     * Creates a logging interceptor
     */
    fun logging(context: CoderToolboxContext): Interceptor {
        return LoggingInterceptor(context)
    }
}