/*
 * DISCLAIMER
 *
 * Copyright 2016 ArangoDB GmbH, Cologne, Germany
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright holder is ArangoDB GmbH, Cologne, Germany
 */

package com.arangodb.internal;

import com.arangodb.ArangoDBException;
import com.arangodb.entity.MetaAware;
import com.arangodb.internal.net.CommunicationProtocol;
import com.arangodb.internal.net.HostHandle;
import com.arangodb.internal.serde.InternalSerde;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.lang.reflect.Type;

/**
 * @author Mark Vollmary
 */
public class ArangoExecutorSync extends ArangoExecutor {

    private static final Logger LOG = LoggerFactory.getLogger(ArangoExecutorSync.class);

    private final CommunicationProtocol protocol;

    public ArangoExecutorSync(final CommunicationProtocol protocol, final InternalSerde util,
                              final QueueTimeMetricsImpl qtMetrics, final int timeoutMs) {
        super(util, qtMetrics, timeoutMs);
        this.protocol = protocol;
    }

    public <T> T execute(final InternalRequest request, final Type type) {
        return execute(request, type, null);
    }

    public <T> T execute(final InternalRequest request, final Type type, final HostHandle hostHandle) {
        return execute(request, response -> createResult(type, response), hostHandle);
    }

    public <T> T execute(final InternalRequest request, final ResponseDeserializer<T> responseDeserializer) {
        return execute(request, responseDeserializer, null);
    }

    public <T> T execute(
            final InternalRequest request,
            final ResponseDeserializer<T> responseDeserializer,
            final HostHandle hostHandle) {

        final InternalResponse response = protocol.execute(interceptRequest(request), hostHandle);
        interceptResponse(response);
        T deserialize = responseDeserializer.deserialize(response);

        if (deserialize instanceof MetaAware) {
            LOG.debug("Response is MetaAware {}", deserialize.getClass().getName());
            ((MetaAware) deserialize).setMeta(response.getMeta());
        }

        return deserialize;
    }

    public void disconnect() {
        try {
            protocol.close();
        } catch (final IOException e) {
            throw new ArangoDBException(e);
        }
    }

    public void setJwt(String jwt) {
        protocol.setJwt(jwt);
    }

}
