# Manipulating indexes

These functions implement the
[HTTP API for manipulating indexes](https://docs.arangodb.com/latest/HTTP/Indexes/index.html).

## ArangoCollection.ensureHashIndex

`ArangoCollection.ensureHashIndex(Iterable<String> fields, HashIndexOptions options) : IndexEntity`

Creates a hash index for the collection if it does not already exist.

**Arguments**

- **fields**: `Iterable<String>`

  A list of attribute paths

- **options**: `HashIndexOptions`

  - **name**: `String`

    Specify a custom name during index creation (optional). When running AQL queries you might then refer
    to these name as your preferred index for lookups (e.g. Index Hints).

  - **unique**: `Boolean`

    If true, then create a unique index

  - **sparse**: `Boolean`

    If true, then create a sparse index

  - **deduplicate**: `Boolean`

    If false, the de-duplication of array values is turned off.

  - **inBackground**: `Boolean`

    Indexes created with the 'inBackground' option, will not hold an exclusive collection
    lock for the entire index creation period (rocksdb only).

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

IndexEntity index = collection.ensureHashIndex(Arrays.asList("a", "b.c"));
// the index has been created with the handle `index.getId()`
```

## ArangoCollection.ensureSkipListIndex

`ArangoCollection.ensureSkipListIndex(Iterable<String> fields, SkipListIndexOptions options) : IndexEntity`

Creates a skip-list index for the collection if it does not already exist.

**Arguments**

- **fields**: `Iterable<String>`

  A list of attribute paths

- **options**: `SkipListIndexOptions`

  - **name**: `String`

    Specify a custom name during index creation (optional). When running AQL queries you might then refer
    to these name as your preferred index for lookups (e.g. Index Hints).

  - **unique**: `Boolean`

    If true, then create a unique index

  - **sparse**: `Boolean`

    If true, then create a sparse index

  - **deduplicate**: `Boolean`

    If false, the de-duplication of array values is turned off.

  - **inBackground**: `Boolean`

    Indexes created with the 'inBackground' option, will not hold an exclusive collection
    lock for the entire index creation period (rocksdb only).

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

IndexEntity index = collection.ensureSkipListIndex(
  Arrays.asList("a", "b.c")
);
// the index has been created with the handle `index.getId()`
```

## ArangoCollection.ensureGeoIndex

`ArangoCollection.ensureGeoIndex(Iterable<String> fields, GeoIndexOptions options) : IndexEntity`

Creates a geo index for the collection if it does not already exist.

**Arguments**

- **fields**: `Iterable<String>`

  A list of attribute paths

- **options**: `GeoIndexOptions`

  - **name**: `String`

    Specify a custom name during index creation (optional). When running AQL queries you might then refer
    to these name as your preferred index for lookups (e.g. Index Hints).

  - **geoJson**: `Boolean`

    If a geo-spatial index on a location is constructed and geoJson is true,
    then the order within the array is longitude followed by latitude.
    This corresponds to the format described in.

  - **inBackground**: `Boolean`

    Indexes created with the 'inBackground' option, will not hold an exclusive collection
    lock for the entire index creation period (rocksdb only).

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

IndexEntity index = collection.ensureGeoIndex(
  Arrays.asList("latitude", "longitude")
);
// the index has been created with the handle `index.getId()`
```

## ArangoCollection.ensureFulltextIndex

`ArangoCollection.ensureFulltextIndex(Iterable<String> fields, FulltextIndexOptions options) : IndexEntity`

Creates a fulltext index for the collection if it does not already exist.

**Arguments**

- **fields**: `Iterable<String>`

  A list of attribute paths

- **options**: `FulltextIndexOptions`

  - **name**: `String`

    Specify a custom name during index creation (optional). When running AQL queries you might then refer
    to these name as your preferred index for lookups (e.g. Index Hints).

  - **minLength**: `Integer`

    Minimum character length of words to index. Will default to a server-defined
    value if unspecified. It is thus recommended to set this value explicitly
    when creating the index.

  - **inBackground**: `Boolean`

    Indexes created with the 'inBackground' option, will not hold an exclusive collection
    lock for the entire index creation period (rocksdb only).

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

IndexEntity index = collection.ensureFulltextIndex(
  Arrays.asList("description")
);
// the index has been created with the handle `index.getId()`
```

## ArangoCollection.ensurePersistentIndex

`ArangoCollection.ensurePersistentIndex(Iterable<String> fields, PersistentIndexOptions options) : IndexEntity`

Creates a persistent index for the collection if it does not already exist.

**Arguments**

- **fields**: `Iterable<String>`

  A list of attribute paths

- **options**: `PersistentIndexOptions`

  - **name**: `String`

    Specify a custom name during index creation (optional). When running AQL queries you might then refer
    to these name as your preferred index for lookups (e.g. Index Hints).

  - **unique**: `Boolean`

    If true, then create a unique index

  - **sparse**: `Boolean`

    If true, then create a sparse index

  - **inBackground**: `Boolean`

    Indexes created with the 'inBackground' option, will not hold an exclusive collection
    lock for the entire index creation period (rocksdb only).

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

IndexEntity index = collection.ensurePersistentIndex(Arrays.asList("a", "b.c"));
// the index has been created with the handle `index.getId()`
```

## ArangoCollection.ensureTtlIndex

`ArangoCollection.ensureTtlIndex(Iterable<String> fields, TtlIndexOptions options) : IndexEntity`

Creates a ttl index for the collection if it does not already exist.

**Arguments**

- **fields**: `Iterable<String>`

  A list of attribute paths

- **options**: `PersistentIndexOptions`

  - **name**: `String`

    Specify a custom name during index creation (optional). When running AQL queries you might then refer
    to these name as your preferred index for lookups (e.g. Index Hints).

  - **expireAfter**: `Integer`

    The time (in seconds) after a document's creation after which the documents count as "expired".

  - **inBackground**: `Boolean`

    Indexes created with the 'inBackground' option, will not hold an exclusive collection
    lock for the entire index creation period (rocksdb only).

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

final TtlIndexOptions options = new TtlIndexOptions();
		options.name("myTtlIndex");
		options.expireAfter(3600);

IndexEntity index = collection.ensureTtlIndex(Arrays.asList("a", "b.c"), options);
// the index has been created with the handle `index.getId()`
```

## ArangoCollection.getIndex

`ArangoCollection.getIndex(String id) : IndexEntity`

Fetches information about the index with the given _id_ and returns it.

**Arguments**

- **id**: `String`

  The index-handle

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

IndexEntity index = collection.getIndex("some-index");
```

## ArangoCollection.getIndexes

`ArangoCollection.getIndexes() : Collection<IndexEntity>`

Fetches a list of all indexes on this collection.

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

Collection<IndexEntity> indexes = collection.getIndexs();
```

## ArangoCollection.deleteIndex

`ArangoCollection.deleteIndex(String id) : String`

Deletes the index with the given _id_ from the collection.

**Arguments**

- **id**: `String`

  The index-handle

**Examples**

```Java
ArangoDB arango = new ArangoDB.Builder().build();
ArangoDatabase db = arango.db("myDB");
ArangoCollection collection = db.collection("some-collection");

collection.deleteIndex("some-index");
```
