/*
 * DISCLAIMER
 *
 * Copyright 2016 ArangoDB GmbH, Cologne, Germany
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright holder is ArangoDB GmbH, Cologne, Germany
 */

package com.arangodb.velocypack;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Map.Entry;

import org.junit.Test;

import com.arangodb.velocypack.exception.VPackException;
import com.arangodb.velocypack.exception.VPackValueTypeException;

/**
 * @author Mark - mark at arangodb.com
 *
 */
public class VPackSliceTest {

	@Test
	public void isNone() {
		final byte[] vpack = { 0x00 };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isNone(), is(true));
	}

	@Test
	public void isNull() {
		final byte[] vpack = { 0x18 };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isNull(), is(true));
	}

	@Test
	public void isIllegal() {
		final byte[] vpack = { 0x17 };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isIllegal(), is(true));
	}

	@Test
	public void booleanTrue() {
		final byte[] vpack = { 0x1a };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isBoolean(), is(true));
		assertThat(slice.isTrue(), is(true));
		assertThat(slice.isFalse(), is(false));
		assertThat(slice.getByteSize(), is(1));
	}

	@Test
	public void booleanFalse() {
		final byte[] vpack = { 0x19 };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isBoolean(), is(true));
		assertThat(slice.isFalse(), is(true));
		assertThat(slice.isTrue(), is(false));
		assertThat(slice.getByteSize(), is(1));
	}

	@Test
	public void isArray() {
		checkArray(new byte[] { 0x01 });
		checkArray(new byte[] { 0x02 });
		checkArray(new byte[] { 0x03 });
		checkArray(new byte[] { 0x04 });
		checkArray(new byte[] { 0x05 });
		checkArray(new byte[] { 0x06 });
		checkArray(new byte[] { 0x07 });
		checkArray(new byte[] { 0x08 });
		checkArray(new byte[] { 0x09 });
		checkArray(new byte[] { 0x13 });
	}

	private void checkArray(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isArray(), is(true));
	}

	@Test
	public void isObject() {
		checkObject(new byte[] { 0x0b });
		checkObject(new byte[] { 0x0c });
		checkObject(new byte[] { 0x0d });
		checkObject(new byte[] { 0x0e });
		checkObject(new byte[] { 0x0f });
		checkObject(new byte[] { 0x10 });
		checkObject(new byte[] { 0x11 });
		checkObject(new byte[] { 0x12 });
		checkObject(new byte[] { 0x14 });
	}

	private void checkObject(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isObject(), is(true));
	}

	@Test
	public void isDouble() {
		final byte[] vpack = { 0x1b };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isDouble(), is(true));
		assertThat(slice.isNumber(), is(true));
	}

	@Test
	public void isUTCDate() {
		final byte[] vpack = { 0x1c };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isDate(), is(true));
	}

	@Test
	public void isExternal() {
		final byte[] vpack = { 0x1d };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isExternal(), is(true));
	}

	@Test
	public void isMinKey() {
		final byte[] vpack = { 0x1e };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isMinKey(), is(true));
	}

	@Test
	public void isMaxKey() {
		final byte[] vpack = { 0x1f };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isMaxKey(), is(true));
	}

	@Test
	public void isInt() {
		checkInt(new byte[] { 0x20 });
		checkInt(new byte[] { 0x21 });
		checkInt(new byte[] { 0x22 });
		checkInt(new byte[] { 0x23 });
		checkInt(new byte[] { 0x24 });
		checkInt(new byte[] { 0x25 });
		checkInt(new byte[] { 0x26 });
		checkInt(new byte[] { 0x27 });
	}

	private void checkInt(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isInt(), is(true));
		assertThat(slice.isInteger(), is(true));
		assertThat(slice.isNumber(), is(true));
	}

	@Test
	public void isUInt() {
		checkUInt(new byte[] { 0x28 });
		checkUInt(new byte[] { 0x29 });
		checkUInt(new byte[] { 0x2a });
		checkUInt(new byte[] { 0x2b });
		checkUInt(new byte[] { 0x2c });
		checkUInt(new byte[] { 0x2d });
		checkUInt(new byte[] { 0x2e });
		checkUInt(new byte[] { 0x2f });
	}

	private void checkUInt(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isUInt(), is(true));
		assertThat(slice.isInteger(), is(true));
		assertThat(slice.isNumber(), is(true));
	}

	@Test
	public void isSmallInt() {
		checkSmallInt(new byte[] { 0x30 });
		checkSmallInt(new byte[] { 0x31 });
		checkSmallInt(new byte[] { 0x32 });
		checkSmallInt(new byte[] { 0x33 });
		checkSmallInt(new byte[] { 0x34 });
		checkSmallInt(new byte[] { 0x35 });
		checkSmallInt(new byte[] { 0x36 });
		checkSmallInt(new byte[] { 0x37 });
		checkSmallInt(new byte[] { 0x38 });
		checkSmallInt(new byte[] { 0x39 });
		checkSmallInt(new byte[] { 0x3a });
		checkSmallInt(new byte[] { 0x3b });
		checkSmallInt(new byte[] { 0x3c });
		checkSmallInt(new byte[] { 0x3d });
		checkSmallInt(new byte[] { 0x3e });
		checkSmallInt(new byte[] { 0x3f });
	}

	private void checkSmallInt(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isSmallInt(), is(true));
		assertThat(slice.isInteger(), is(true));
		assertThat(slice.isNumber(), is(true));
	}

	@Test
	public void isString() {
		checkString(new byte[] { 0x40 });
		checkString(new byte[] { 0x41 });
		checkString(new byte[] { 0x42 });
		checkString(new byte[] { 0x43 });
		checkString(new byte[] { 0x44 });
		checkString(new byte[] { 0x45 });
		checkString(new byte[] { 0x46 });
		checkString(new byte[] { 0x47 });
		checkString(new byte[] { 0x48 });
		checkString(new byte[] { 0x49 });
		checkString(new byte[] { 0x4a });
		checkString(new byte[] { 0x4b });
		checkString(new byte[] { 0x4c });
		checkString(new byte[] { 0x4d });
		checkString(new byte[] { 0x4e });
		checkString(new byte[] { 0x4f });
		checkString(new byte[] { 0x50 });
		checkString(new byte[] { 0x51 });
		checkString(new byte[] { 0x52 });
		checkString(new byte[] { 0x53 });
		checkString(new byte[] { 0x54 });
		checkString(new byte[] { 0x55 });
		checkString(new byte[] { 0x56 });
		checkString(new byte[] { 0x57 });
		checkString(new byte[] { 0x58 });
		checkString(new byte[] { 0x59 });
		checkString(new byte[] { 0x5a });
		checkString(new byte[] { 0x5b });
		checkString(new byte[] { 0x5c });
		checkString(new byte[] { 0x5d });
		checkString(new byte[] { 0x5e });
		checkString(new byte[] { 0x5f });
		checkString(new byte[] { 0x60 });
		checkString(new byte[] { 0x61 });
		checkString(new byte[] { 0x62 });
		checkString(new byte[] { 0x63 });
		checkString(new byte[] { 0x64 });
		checkString(new byte[] { 0x65 });
		checkString(new byte[] { 0x66 });
		checkString(new byte[] { 0x67 });
		checkString(new byte[] { 0x68 });
		checkString(new byte[] { 0x69 });
		checkString(new byte[] { 0x6a });
		checkString(new byte[] { 0x6b });
		checkString(new byte[] { 0x6c });
		checkString(new byte[] { 0x6d });
		checkString(new byte[] { 0x6e });
		checkString(new byte[] { 0x6f });
		checkString(new byte[] { 0x70 });
		checkString(new byte[] { 0x71 });
		checkString(new byte[] { 0x72 });
		checkString(new byte[] { 0x73 });
		checkString(new byte[] { 0x74 });
		checkString(new byte[] { 0x75 });
		checkString(new byte[] { 0x76 });
		checkString(new byte[] { 0x77 });
		checkString(new byte[] { 0x78 });
		checkString(new byte[] { 0x79 });
		checkString(new byte[] { 0x7a });
		checkString(new byte[] { 0x7b });
		checkString(new byte[] { 0x7c });
		checkString(new byte[] { 0x7d });
		checkString(new byte[] { 0x7e });
		checkString(new byte[] { 0x7f });
		checkString(new byte[] { (byte) 0x80 });
		checkString(new byte[] { (byte) 0x81 });
		checkString(new byte[] { (byte) 0x82 });
		checkString(new byte[] { (byte) 0x83 });
		checkString(new byte[] { (byte) 0x84 });
		checkString(new byte[] { (byte) 0x85 });
		checkString(new byte[] { (byte) 0x86 });
		checkString(new byte[] { (byte) 0x87 });
		checkString(new byte[] { (byte) 0x88 });
		checkString(new byte[] { (byte) 0x89 });
		checkString(new byte[] { (byte) 0x8a });
		checkString(new byte[] { (byte) 0x8b });
		checkString(new byte[] { (byte) 0x8c });
		checkString(new byte[] { (byte) 0x8d });
		checkString(new byte[] { (byte) 0x8e });
		checkString(new byte[] { (byte) 0x8f });
		checkString(new byte[] { (byte) 0x90 });
		checkString(new byte[] { (byte) 0x91 });
		checkString(new byte[] { (byte) 0x92 });
		checkString(new byte[] { (byte) 0x93 });
		checkString(new byte[] { (byte) 0x94 });
		checkString(new byte[] { (byte) 0x95 });
		checkString(new byte[] { (byte) 0x96 });
		checkString(new byte[] { (byte) 0x97 });
		checkString(new byte[] { (byte) 0x98 });
		checkString(new byte[] { (byte) 0x99 });
		checkString(new byte[] { (byte) 0x9a });
		checkString(new byte[] { (byte) 0x9b });
		checkString(new byte[] { (byte) 0x9c });
		checkString(new byte[] { (byte) 0x9d });
		checkString(new byte[] { (byte) 0x9e });
		checkString(new byte[] { (byte) 0x9f });
		checkString(new byte[] { (byte) 0xa0 });
		checkString(new byte[] { (byte) 0xa1 });
		checkString(new byte[] { (byte) 0xa2 });
		checkString(new byte[] { (byte) 0xa3 });
		checkString(new byte[] { (byte) 0xa4 });
		checkString(new byte[] { (byte) 0xa5 });
		checkString(new byte[] { (byte) 0xa6 });
		checkString(new byte[] { (byte) 0xa7 });
		checkString(new byte[] { (byte) 0xa8 });
		checkString(new byte[] { (byte) 0xa9 });
		checkString(new byte[] { (byte) 0xaa });
		checkString(new byte[] { (byte) 0xab });
		checkString(new byte[] { (byte) 0xac });
		checkString(new byte[] { (byte) 0xad });
		checkString(new byte[] { (byte) 0xae });
		checkString(new byte[] { (byte) 0xaf });
		checkString(new byte[] { (byte) 0xb0 });
		checkString(new byte[] { (byte) 0xb1 });
		checkString(new byte[] { (byte) 0xb2 });
		checkString(new byte[] { (byte) 0xb3 });
		checkString(new byte[] { (byte) 0xb4 });
		checkString(new byte[] { (byte) 0xb5 });
		checkString(new byte[] { (byte) 0xb6 });
		checkString(new byte[] { (byte) 0xb7 });
		checkString(new byte[] { (byte) 0xb8 });
		checkString(new byte[] { (byte) 0xb9 });
		checkString(new byte[] { (byte) 0xba });
		checkString(new byte[] { (byte) 0xbb });
		checkString(new byte[] { (byte) 0xbc });
		checkString(new byte[] { (byte) 0xbd });
		checkString(new byte[] { (byte) 0xbe });
		checkString(new byte[] { (byte) 0xbf });
	}

	private void checkString(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isString(), is(true));
	}

	@Test
	public void isBinary() {
		checkBinary(new byte[] { (byte) 0xc0 });
		checkBinary(new byte[] { (byte) 0xc1 });
		checkBinary(new byte[] { (byte) 0xc2 });
		checkBinary(new byte[] { (byte) 0xc3 });
		checkBinary(new byte[] { (byte) 0xc4 });
		checkBinary(new byte[] { (byte) 0xc5 });
		checkBinary(new byte[] { (byte) 0xc6 });
		checkBinary(new byte[] { (byte) 0xc7 });
	}

	private void checkBinary(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isBinary(), is(true));
	}

	@Test
	public void isBCD() {
		checkBCD(new byte[] { (byte) 0xc8 });
		checkBCD(new byte[] { (byte) 0xc9 });
		checkBCD(new byte[] { (byte) 0xca });
		checkBCD(new byte[] { (byte) 0xcb });
		checkBCD(new byte[] { (byte) 0xcc });
		checkBCD(new byte[] { (byte) 0xcd });
		checkBCD(new byte[] { (byte) 0xce });
		checkBCD(new byte[] { (byte) 0xcf });
		checkBCD(new byte[] { (byte) 0xd0 });
		checkBCD(new byte[] { (byte) 0xd1 });
		checkBCD(new byte[] { (byte) 0xd2 });
		checkBCD(new byte[] { (byte) 0xd3 });
		checkBCD(new byte[] { (byte) 0xd4 });
		checkBCD(new byte[] { (byte) 0xd5 });
		checkBCD(new byte[] { (byte) 0xd6 });
		checkBCD(new byte[] { (byte) 0xd7 });
	}

	private void checkBCD(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isBCD(), is(true));
	}

	@Test
	public void isCustom() {
		checkCustom(new byte[] { (byte) 0xf0 });
		checkCustom(new byte[] { (byte) 0xf1 });
		checkCustom(new byte[] { (byte) 0xf2 });
		checkCustom(new byte[] { (byte) 0xf3 });
		checkCustom(new byte[] { (byte) 0xf4 });
		checkCustom(new byte[] { (byte) 0xf5 });
		checkCustom(new byte[] { (byte) 0xf6 });
		checkCustom(new byte[] { (byte) 0xf7 });
		checkCustom(new byte[] { (byte) 0xf8 });
		checkCustom(new byte[] { (byte) 0xf9 });
		checkCustom(new byte[] { (byte) 0xfa });
		checkCustom(new byte[] { (byte) 0xfb });
		checkCustom(new byte[] { (byte) 0xfc });
		checkCustom(new byte[] { (byte) 0xfd });
		checkCustom(new byte[] { (byte) 0xfe });
		checkCustom(new byte[] { (byte) 0xff });
	}

	private void checkCustom(final byte[] vpack) {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.isCustom(), is(true));
	}

	@Test
	public void getBooleanTrue() throws VPackException {
		final byte[] vpack = { 0x1a };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsBoolean(), is(true));
	}

	@Test
	public void getBooleanFalse() throws VPackException {
		final byte[] vpack = { 0x19 };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsBoolean(), is(false));
	}

	@Test(expected = VPackValueTypeException.class)
	public void getBooleanFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsBoolean();
	}

	@Test
	public void getDouble() throws VPackException {
		{
			final byte[] vpack = { 0x1b, 0x66, 0x66, 0x66, 0x66, 0x66, (byte) 0xb6, 0x60, 0x40 };
			final VPackSlice slice = new VPackSlice(vpack);
			assertThat(slice.getAsDouble(), is(133.7));
		}
		{
			final byte[] vpack = { 0x1b, 0x66, 0x66, 0x66, 0x66, 0x66, (byte) 0xb6, 0x60, (byte) 0xc0 };
			final VPackSlice slice = new VPackSlice(vpack);
			assertThat(slice.getAsDouble(), is(-133.7));
		}
	}

	@Test
	public void getDoubleAsNumber() throws VPackException {
		{
			final byte[] vpack = { 0x1b, 0x66, 0x66, 0x66, 0x66, 0x66, (byte) 0xb6, 0x60, 0x40 };
			final VPackSlice slice = new VPackSlice(vpack);
			assertThat(slice.getAsNumber().doubleValue(), is(133.7));
		}
		{
			final byte[] vpack = { 0x1b, 0x66, 0x66, 0x66, 0x66, 0x66, (byte) 0xb6, 0x60, (byte) 0xc0 };
			final VPackSlice slice = new VPackSlice(vpack);
			assertThat(slice.getAsNumber().doubleValue(), is(-133.7));
		}
	}

	@Test(expected = VPackValueTypeException.class)
	public void getDoubleFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsDouble();
	}

	@Test(expected = VPackValueTypeException.class)
	public void getNumberFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsNumber();
	}

	@Test
	public void getSmallInt() throws VPackException {
		checkSmallInt(0, new byte[] { 0x30 });
		checkSmallInt(1, new byte[] { 0x31 });
		checkSmallInt(2, new byte[] { 0x32 });
		checkSmallInt(3, new byte[] { 0x33 });
		checkSmallInt(4, new byte[] { 0x34 });
		checkSmallInt(5, new byte[] { 0x35 });
		checkSmallInt(6, new byte[] { 0x36 });
		checkSmallInt(7, new byte[] { 0x37 });
		checkSmallInt(8, new byte[] { 0x38 });
		checkSmallInt(9, new byte[] { 0x39 });
		checkSmallInt(-6, new byte[] { 0x3a });
		checkSmallInt(-5, new byte[] { 0x3b });
		checkSmallInt(-4, new byte[] { 0x3c });
		checkSmallInt(-3, new byte[] { 0x3d });
		checkSmallInt(-2, new byte[] { 0x3e });
		checkSmallInt(-1, new byte[] { 0x3f });
	}

	private void checkSmallInt(final int expecteds, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsInt(), is(expecteds));
	}

	@Test
	public void getInt() throws VPackException {
		checkInt(Short.MAX_VALUE, new byte[] { 0x29, (byte) 0xff, 0x7f });
		checkInt(Integer.MAX_VALUE, new byte[] { 0x2b, (byte) 0xff, (byte) 0xff, (byte) 0xff, 0x7f });
		checkInt(Long.MAX_VALUE, new byte[] { 0x27, -1, -1, -1, -1, -1, -1, -1, 127 });
	}

	private void checkInt(final long expextedValue, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsLong(), is(expextedValue));
	}

	@Test
	public void getUInt() throws VPackException {
		checkUInt(Short.MAX_VALUE, new byte[] { 0x29, -1, 127 });
		checkUInt(Integer.MAX_VALUE, new byte[] { 0x2b, -1, -1, -1, 127 });
		checkUInt(Long.MAX_VALUE, new byte[] { 0x2f, -1, -1, -1, -1, -1, -1, -1, 127 });
	}

	@Test
	public void getUIntAsBigInteger() throws VPackException {
		checkUIntAsBigInteger(new BigInteger(String.valueOf(Short.MAX_VALUE)), new byte[] { 0x29, -1, 127 });
		checkUIntAsBigInteger(new BigInteger(String.valueOf(Integer.MAX_VALUE)), new byte[] { 0x2b, -1, -1, -1, 127 });
		final BigInteger longMax = new BigInteger(String.valueOf(Long.MAX_VALUE));
		checkUIntAsBigInteger(longMax, new byte[] { 0x2f, -1, -1, -1, -1, -1, -1, -1, 127 });
		checkUIntAsBigInteger(longMax.add(longMax), new byte[] { 0x2f, -2, -1, -1, -1, -1, -1, -1, -1 });
	}

	private void checkUInt(final long expecteds, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsLong(), is(expecteds));
	}

	private void checkUIntAsBigInteger(final BigInteger expecteds, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsBigInteger(), is(expecteds));
	}

	@Test(expected = VPackValueTypeException.class)
	public void getBigIntegerFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsBigInteger();
	}

	@Test
	public void getDate() throws VPackException {
		final byte[] vpack = { 0x1c, 0, 83, 115, 5, -114, 0, 0, 0, };
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsDate(), is(new Date(609976800000l)));
	}

	@Test(expected = VPackValueTypeException.class)
	public void getDateFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsDate();
	}

	@Test
	public void getString() throws VPackException {
		checkString("Hallo Welt!", new byte[] { 0x4b, 72, 97, 108, 108, 111, 32, 87, 101, 108, 116, 33 });
		checkString("Hello World!", new byte[] { 0x4c, 72, 101, 108, 108, 111, 32, 87, 111, 114, 108, 100, 33 });
		checkString(
			"Lorem ipsum dolor sit amet, consectetuer adipiscing elit. Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. Vivamus elementum semper nisi. Aenean vulputate eleifend tellus.",
			new byte[] { (byte) 0xbf, 0x37, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4c, 0x6f, 0x72, 0x65, 0x6d,
					0x20, 0x69, 0x70, 0x73, 0x75, 0x6d, 0x20, 0x64, 0x6f, 0x6c, 0x6f, 0x72, 0x20, 0x73, 0x69, 0x74,
					0x20, 0x61, 0x6d, 0x65, 0x74, 0x2c, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x65, 0x63, 0x74, 0x65, 0x74,
					0x75, 0x65, 0x72, 0x20, 0x61, 0x64, 0x69, 0x70, 0x69, 0x73, 0x63, 0x69, 0x6e, 0x67, 0x20, 0x65,
					0x6c, 0x69, 0x74, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61, 0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x6d,
					0x6f, 0x64, 0x6f, 0x20, 0x6c, 0x69, 0x67, 0x75, 0x6c, 0x61, 0x20, 0x65, 0x67, 0x65, 0x74, 0x20,
					0x64, 0x6f, 0x6c, 0x6f, 0x72, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61, 0x6e, 0x20, 0x6d, 0x61,
					0x73, 0x73, 0x61, 0x2e, 0x20, 0x43, 0x75, 0x6d, 0x20, 0x73, 0x6f, 0x63, 0x69, 0x69, 0x73, 0x20,
					0x6e, 0x61, 0x74, 0x6f, 0x71, 0x75, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x61, 0x74, 0x69, 0x62, 0x75,
					0x73, 0x20, 0x65, 0x74, 0x20, 0x6d, 0x61, 0x67, 0x6e, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x20,
					0x70, 0x61, 0x72, 0x74, 0x75, 0x72, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x6d, 0x6f, 0x6e, 0x74, 0x65,
					0x73, 0x2c, 0x20, 0x6e, 0x61, 0x73, 0x63, 0x65, 0x74, 0x75, 0x72, 0x20, 0x72, 0x69, 0x64, 0x69,
					0x63, 0x75, 0x6c, 0x75, 0x73, 0x20, 0x6d, 0x75, 0x73, 0x2e, 0x20, 0x44, 0x6f, 0x6e, 0x65, 0x63,
					0x20, 0x71, 0x75, 0x61, 0x6d, 0x20, 0x66, 0x65, 0x6c, 0x69, 0x73, 0x2c, 0x20, 0x75, 0x6c, 0x74,
					0x72, 0x69, 0x63, 0x69, 0x65, 0x73, 0x20, 0x6e, 0x65, 0x63, 0x2c, 0x20, 0x70, 0x65, 0x6c, 0x6c,
					0x65, 0x6e, 0x74, 0x65, 0x73, 0x71, 0x75, 0x65, 0x20, 0x65, 0x75, 0x2c, 0x20, 0x70, 0x72, 0x65,
					0x74, 0x69, 0x75, 0x6d, 0x20, 0x71, 0x75, 0x69, 0x73, 0x2c, 0x20, 0x73, 0x65, 0x6d, 0x2e, 0x20,
					0x4e, 0x75, 0x6c, 0x6c, 0x61, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x65, 0x71, 0x75, 0x61, 0x74, 0x20,
					0x6d, 0x61, 0x73, 0x73, 0x61, 0x20, 0x71, 0x75, 0x69, 0x73, 0x20, 0x65, 0x6e, 0x69, 0x6d, 0x2e,
					0x20, 0x44, 0x6f, 0x6e, 0x65, 0x63, 0x20, 0x70, 0x65, 0x64, 0x65, 0x20, 0x6a, 0x75, 0x73, 0x74,
					0x6f, 0x2c, 0x20, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x69, 0x6c, 0x6c, 0x61, 0x20, 0x76, 0x65, 0x6c,
					0x2c, 0x20, 0x61, 0x6c, 0x69, 0x71, 0x75, 0x65, 0x74, 0x20, 0x6e, 0x65, 0x63, 0x2c, 0x20, 0x76,
					0x75, 0x6c, 0x70, 0x75, 0x74, 0x61, 0x74, 0x65, 0x20, 0x65, 0x67, 0x65, 0x74, 0x2c, 0x20, 0x61,
					0x72, 0x63, 0x75, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x65, 0x6e, 0x69, 0x6d, 0x20, 0x6a, 0x75, 0x73,
					0x74, 0x6f, 0x2c, 0x20, 0x72, 0x68, 0x6f, 0x6e, 0x63, 0x75, 0x73, 0x20, 0x75, 0x74, 0x2c, 0x20,
					0x69, 0x6d, 0x70, 0x65, 0x72, 0x64, 0x69, 0x65, 0x74, 0x20, 0x61, 0x2c, 0x20, 0x76, 0x65, 0x6e,
					0x65, 0x6e, 0x61, 0x74, 0x69, 0x73, 0x20, 0x76, 0x69, 0x74, 0x61, 0x65, 0x2c, 0x20, 0x6a, 0x75,
					0x73, 0x74, 0x6f, 0x2e, 0x20, 0x4e, 0x75, 0x6c, 0x6c, 0x61, 0x6d, 0x20, 0x64, 0x69, 0x63, 0x74,
					0x75, 0x6d, 0x20, 0x66, 0x65, 0x6c, 0x69, 0x73, 0x20, 0x65, 0x75, 0x20, 0x70, 0x65, 0x64, 0x65,
					0x20, 0x6d, 0x6f, 0x6c, 0x6c, 0x69, 0x73, 0x20, 0x70, 0x72, 0x65, 0x74, 0x69, 0x75, 0x6d, 0x2e,
					0x20, 0x49, 0x6e, 0x74, 0x65, 0x67, 0x65, 0x72, 0x20, 0x74, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x75,
					0x6e, 0x74, 0x2e, 0x20, 0x43, 0x72, 0x61, 0x73, 0x20, 0x64, 0x61, 0x70, 0x69, 0x62, 0x75, 0x73,
					0x2e, 0x20, 0x56, 0x69, 0x76, 0x61, 0x6d, 0x75, 0x73, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e,
					0x74, 0x75, 0x6d, 0x20, 0x73, 0x65, 0x6d, 0x70, 0x65, 0x72, 0x20, 0x6e, 0x69, 0x73, 0x69, 0x2e,
					0x20, 0x41, 0x65, 0x6e, 0x65, 0x61, 0x6e, 0x20, 0x76, 0x75, 0x6c, 0x70, 0x75, 0x74, 0x61, 0x74,
					0x65, 0x20, 0x65, 0x6c, 0x65, 0x69, 0x66, 0x65, 0x6e, 0x64, 0x20, 0x74, 0x65, 0x6c, 0x6c, 0x75,
					0x73, 0x2e });
	}

	private void checkString(final String expecteds, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsString(), is(expecteds));
	}

	@Test
	public void getStringLength() throws VPackException {
		checkStringLength(11, new byte[] { 0x4b, 72, 97, 108, 108, 111, 32, 87, 101, 108, 116, 33 });
		checkStringLength(12, new byte[] { 0x4c, 72, 101, 108, 108, 111, 32, 87, 111, 114, 108, 100, 33 });
		checkStringLength(567, new byte[] { (byte) 0xbf, 0x37, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4c, 0x6f,
				0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d, 0x20, 0x64, 0x6f, 0x6c, 0x6f, 0x72, 0x20, 0x73,
				0x69, 0x74, 0x20, 0x61, 0x6d, 0x65, 0x74, 0x2c, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x65, 0x63, 0x74, 0x65,
				0x74, 0x75, 0x65, 0x72, 0x20, 0x61, 0x64, 0x69, 0x70, 0x69, 0x73, 0x63, 0x69, 0x6e, 0x67, 0x20, 0x65,
				0x6c, 0x69, 0x74, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61, 0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x6f,
				0x64, 0x6f, 0x20, 0x6c, 0x69, 0x67, 0x75, 0x6c, 0x61, 0x20, 0x65, 0x67, 0x65, 0x74, 0x20, 0x64, 0x6f,
				0x6c, 0x6f, 0x72, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61, 0x6e, 0x20, 0x6d, 0x61, 0x73, 0x73, 0x61,
				0x2e, 0x20, 0x43, 0x75, 0x6d, 0x20, 0x73, 0x6f, 0x63, 0x69, 0x69, 0x73, 0x20, 0x6e, 0x61, 0x74, 0x6f,
				0x71, 0x75, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x61, 0x74, 0x69, 0x62, 0x75, 0x73, 0x20, 0x65, 0x74, 0x20,
				0x6d, 0x61, 0x67, 0x6e, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73, 0x20, 0x70, 0x61, 0x72, 0x74, 0x75, 0x72,
				0x69, 0x65, 0x6e, 0x74, 0x20, 0x6d, 0x6f, 0x6e, 0x74, 0x65, 0x73, 0x2c, 0x20, 0x6e, 0x61, 0x73, 0x63,
				0x65, 0x74, 0x75, 0x72, 0x20, 0x72, 0x69, 0x64, 0x69, 0x63, 0x75, 0x6c, 0x75, 0x73, 0x20, 0x6d, 0x75,
				0x73, 0x2e, 0x20, 0x44, 0x6f, 0x6e, 0x65, 0x63, 0x20, 0x71, 0x75, 0x61, 0x6d, 0x20, 0x66, 0x65, 0x6c,
				0x69, 0x73, 0x2c, 0x20, 0x75, 0x6c, 0x74, 0x72, 0x69, 0x63, 0x69, 0x65, 0x73, 0x20, 0x6e, 0x65, 0x63,
				0x2c, 0x20, 0x70, 0x65, 0x6c, 0x6c, 0x65, 0x6e, 0x74, 0x65, 0x73, 0x71, 0x75, 0x65, 0x20, 0x65, 0x75,
				0x2c, 0x20, 0x70, 0x72, 0x65, 0x74, 0x69, 0x75, 0x6d, 0x20, 0x71, 0x75, 0x69, 0x73, 0x2c, 0x20, 0x73,
				0x65, 0x6d, 0x2e, 0x20, 0x4e, 0x75, 0x6c, 0x6c, 0x61, 0x20, 0x63, 0x6f, 0x6e, 0x73, 0x65, 0x71, 0x75,
				0x61, 0x74, 0x20, 0x6d, 0x61, 0x73, 0x73, 0x61, 0x20, 0x71, 0x75, 0x69, 0x73, 0x20, 0x65, 0x6e, 0x69,
				0x6d, 0x2e, 0x20, 0x44, 0x6f, 0x6e, 0x65, 0x63, 0x20, 0x70, 0x65, 0x64, 0x65, 0x20, 0x6a, 0x75, 0x73,
				0x74, 0x6f, 0x2c, 0x20, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x69, 0x6c, 0x6c, 0x61, 0x20, 0x76, 0x65, 0x6c,
				0x2c, 0x20, 0x61, 0x6c, 0x69, 0x71, 0x75, 0x65, 0x74, 0x20, 0x6e, 0x65, 0x63, 0x2c, 0x20, 0x76, 0x75,
				0x6c, 0x70, 0x75, 0x74, 0x61, 0x74, 0x65, 0x20, 0x65, 0x67, 0x65, 0x74, 0x2c, 0x20, 0x61, 0x72, 0x63,
				0x75, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x65, 0x6e, 0x69, 0x6d, 0x20, 0x6a, 0x75, 0x73, 0x74, 0x6f, 0x2c,
				0x20, 0x72, 0x68, 0x6f, 0x6e, 0x63, 0x75, 0x73, 0x20, 0x75, 0x74, 0x2c, 0x20, 0x69, 0x6d, 0x70, 0x65,
				0x72, 0x64, 0x69, 0x65, 0x74, 0x20, 0x61, 0x2c, 0x20, 0x76, 0x65, 0x6e, 0x65, 0x6e, 0x61, 0x74, 0x69,
				0x73, 0x20, 0x76, 0x69, 0x74, 0x61, 0x65, 0x2c, 0x20, 0x6a, 0x75, 0x73, 0x74, 0x6f, 0x2e, 0x20, 0x4e,
				0x75, 0x6c, 0x6c, 0x61, 0x6d, 0x20, 0x64, 0x69, 0x63, 0x74, 0x75, 0x6d, 0x20, 0x66, 0x65, 0x6c, 0x69,
				0x73, 0x20, 0x65, 0x75, 0x20, 0x70, 0x65, 0x64, 0x65, 0x20, 0x6d, 0x6f, 0x6c, 0x6c, 0x69, 0x73, 0x20,
				0x70, 0x72, 0x65, 0x74, 0x69, 0x75, 0x6d, 0x2e, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x67, 0x65, 0x72, 0x20,
				0x74, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x75, 0x6e, 0x74, 0x2e, 0x20, 0x43, 0x72, 0x61, 0x73, 0x20, 0x64,
				0x61, 0x70, 0x69, 0x62, 0x75, 0x73, 0x2e, 0x20, 0x56, 0x69, 0x76, 0x61, 0x6d, 0x75, 0x73, 0x20, 0x65,
				0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x75, 0x6d, 0x20, 0x73, 0x65, 0x6d, 0x70, 0x65, 0x72, 0x20, 0x6e,
				0x69, 0x73, 0x69, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61, 0x6e, 0x20, 0x76, 0x75, 0x6c, 0x70, 0x75,
				0x74, 0x61, 0x74, 0x65, 0x20, 0x65, 0x6c, 0x65, 0x69, 0x66, 0x65, 0x6e, 0x64, 0x20, 0x74, 0x65, 0x6c,
				0x6c, 0x75, 0x73, 0x2e });
	}

	private void checkStringLength(final int expected, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getLength(), is(expected));
		assertThat(slice.getByteSize(), is(vpack.length));
	}

	@Test(expected = VPackValueTypeException.class)
	public void getStringFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsString();
	}

	@Test
	public void getBinary() throws VPackException {
		final byte[] expected = new byte[] { 49, 50, 51, 52, 53, 54, 55, 56, 57 };
		checkBinary(expected, new byte[] { (byte) 0xc0, 9, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc1, 9, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc2, 9, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc3, 9, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc4, 9, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc5, 9, 0, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc6, 9, 0, 0, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc7, 9, 0, 0, 0, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
	}

	private void checkBinary(final byte[] expected, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getAsBinary(), is(expected));
		assertThat(slice.getByteSize(), is(vpack.length));
	}

	@Test
	public void getBinaryLength() throws VPackException {
		final int expected = 9;
		checkBinary(expected, new byte[] { (byte) 0xc0, 9, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc1, 9, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc2, 9, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc3, 9, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc4, 9, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc5, 9, 0, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc6, 9, 0, 0, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
		checkBinary(expected, new byte[] { (byte) 0xc7, 9, 0, 0, 0, 0, 0, 0, 0, 49, 50, 51, 52, 53, 54, 55, 56, 57 });
	}

	private void checkBinary(final int expected, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getBinaryLength(), is(expected));
		assertThat(slice.getByteSize(), is(vpack.length));
	}

	@Test(expected = VPackValueTypeException.class)
	public void getBinaryFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getAsBinary();
	}

	@Test(expected = VPackValueTypeException.class)
	public void getBinaryLengthFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getBinaryLength();
	}

	@Test(expected = VPackValueTypeException.class)
	public void notArrayAt() throws VPackException {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x1a });
		slice.get(1);
	}

	@Test
	public void arrayEmpty() throws VPackException {
		checkArray(new long[] {}, new byte[] { 0x01 });
	}

	@Test(expected = IndexOutOfBoundsException.class)
	public void arrayEmptyAt() throws VPackException {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x01 });
		slice.get(1);
	}

	@Test(expected = IndexOutOfBoundsException.class)
	public void arrayWrongIndex() throws VPackException {
		// [1, 2 ,3]
		final VPackSlice slice = new VPackSlice(new byte[] { 0x02, 0x05, 0x31, 0x32, 0x33 });
		slice.get(4);
	}

	@Test
	public void arrayWithoutIndexTable() throws VPackException {
		final long[] expected = new long[] { 1, 2, 3 };
		checkArray(expected, new byte[] { 0x02, 0x05, 0x31, 0x32, 0x33 });
		checkArray(expected, new byte[] { 0x03, 0x06, 0x00, 0x31, 0x32, 0x33 });
		checkArray(expected, new byte[] { 0x04, 0x08, 0x00, 0x00, 0x00, 0x31, 0x32, 0x33 });
		checkArray(expected, new byte[] { 0x05, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31, 0x32, 0x33 });
	}

	@Test
	public void arrayWithIndexTable() throws VPackException {
		final long[] expected = new long[] { 1, 2, 3 };
		checkArray(expected, new byte[] { 0x06, 0x09, 0x03, 0x31, 0x32, 0x33, 0x03, 0x04, 0x05 });
		checkArray(expected,
			new byte[] { 0x07, 0x0e, 0x00, 0x03, 0x00, 0x31, 0x32, 0x33, 0x05, 0x00, 0x06, 0x00, 0x07, 0x00 });
		checkArray(expected, new byte[] { 0x08, 0x18, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x31, 0x32, 0x33, 0x09,
				0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00 });
	}

	@Test
	public void arrayWithIndexTable8bytes() throws VPackException {
		final long[] expected = new long[] { 1, 2, 3 };
		checkArray(expected,
			new byte[] { 0x09, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x31, 0x32, 0x33, 0x09, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 });
	}

	@Test
	public void arrayCompact() throws VPackException {
		final long[] expected = { 1, 16 };
		checkArray(expected, new byte[] { 0x13, 0x06, 0x31, 0x28, 0x10, 0x02 });
	}

	@Test(expected = IndexOutOfBoundsException.class)
	public void arrayCompactWrongIndex() throws VPackException {
		// [1, 16]
		final VPackSlice slice = new VPackSlice(new byte[] { 0x13, 0x06, 0x31, 0x28, 0x10, 0x02 });
		slice.get(3);
	}

	@Test
	public void arrayIncludesArray() throws VPackException {
		final long[][] expected = { { 1, 2, 3 }, { 1, 2, 3 } };
		checkArray(expected, new byte[] { 0x02, 0x0c, 0x02, 0x05, 0x31, 0x32, 0x33, 0x02, 0x05, 0x31, 0x32, 0x33 });
	}

	@Test
	public void arrayIncludesArrayCompact() throws VPackException {
		final long[][] expected = { { 1, 2, 3 }, { 1, 2, 3 } };
		checkArray(expected,
			new byte[] { 0x02, 0x0e, 0x13, 0x06, 0x31, 0x32, 0x33, 0x03, 0x13, 0x06, 0x31, 0x32, 0x33, 0x03 });
	}

	@Test
	public void arrayIncludesObject() throws VPackException {
		// [{"a": 12, "b": true, "c": "xyz"},{"a": 12, "b": true, "c": "xyz"}]
		checkLength(2,
			new byte[] { 0x13, 0x23, 0x14, 0x10, 0x41, 0x61, 0x28, 0x0c, 0x41, 0x62, 0x1a, 0x41, 0x63, 0x43, 0x78, 0x79,
					0x7a, 0x03, 0x14, 0x10, 0x41, 0x61, 0x28, 0x0c, 0x41, 0x62, 0x1a, 0x41, 0x63, 0x43, 0x78, 0x79,
					0x7a, 0x03, 0x02 });
	}

	@Test
	public void arrayIncludesLongString() throws VPackException {
		checkLength(1, new byte[] { 0x03, 0x49, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (byte) 0xbf, 0x37, 0x02, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x4c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d, 0x20,
				0x64, 0x6f, 0x6c, 0x6f, 0x72, 0x20, 0x73, 0x69, 0x74, 0x20, 0x61, 0x6d, 0x65, 0x74, 0x2c, 0x20, 0x63,
				0x6f, 0x6e, 0x73, 0x65, 0x63, 0x74, 0x65, 0x74, 0x75, 0x65, 0x72, 0x20, 0x61, 0x64, 0x69, 0x70, 0x69,
				0x73, 0x63, 0x69, 0x6e, 0x67, 0x20, 0x65, 0x6c, 0x69, 0x74, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61,
				0x6e, 0x20, 0x63, 0x6f, 0x6d, 0x6d, 0x6f, 0x64, 0x6f, 0x20, 0x6c, 0x69, 0x67, 0x75, 0x6c, 0x61, 0x20,
				0x65, 0x67, 0x65, 0x74, 0x20, 0x64, 0x6f, 0x6c, 0x6f, 0x72, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61,
				0x6e, 0x20, 0x6d, 0x61, 0x73, 0x73, 0x61, 0x2e, 0x20, 0x43, 0x75, 0x6d, 0x20, 0x73, 0x6f, 0x63, 0x69,
				0x69, 0x73, 0x20, 0x6e, 0x61, 0x74, 0x6f, 0x71, 0x75, 0x65, 0x20, 0x70, 0x65, 0x6e, 0x61, 0x74, 0x69,
				0x62, 0x75, 0x73, 0x20, 0x65, 0x74, 0x20, 0x6d, 0x61, 0x67, 0x6e, 0x69, 0x73, 0x20, 0x64, 0x69, 0x73,
				0x20, 0x70, 0x61, 0x72, 0x74, 0x75, 0x72, 0x69, 0x65, 0x6e, 0x74, 0x20, 0x6d, 0x6f, 0x6e, 0x74, 0x65,
				0x73, 0x2c, 0x20, 0x6e, 0x61, 0x73, 0x63, 0x65, 0x74, 0x75, 0x72, 0x20, 0x72, 0x69, 0x64, 0x69, 0x63,
				0x75, 0x6c, 0x75, 0x73, 0x20, 0x6d, 0x75, 0x73, 0x2e, 0x20, 0x44, 0x6f, 0x6e, 0x65, 0x63, 0x20, 0x71,
				0x75, 0x61, 0x6d, 0x20, 0x66, 0x65, 0x6c, 0x69, 0x73, 0x2c, 0x20, 0x75, 0x6c, 0x74, 0x72, 0x69, 0x63,
				0x69, 0x65, 0x73, 0x20, 0x6e, 0x65, 0x63, 0x2c, 0x20, 0x70, 0x65, 0x6c, 0x6c, 0x65, 0x6e, 0x74, 0x65,
				0x73, 0x71, 0x75, 0x65, 0x20, 0x65, 0x75, 0x2c, 0x20, 0x70, 0x72, 0x65, 0x74, 0x69, 0x75, 0x6d, 0x20,
				0x71, 0x75, 0x69, 0x73, 0x2c, 0x20, 0x73, 0x65, 0x6d, 0x2e, 0x20, 0x4e, 0x75, 0x6c, 0x6c, 0x61, 0x20,
				0x63, 0x6f, 0x6e, 0x73, 0x65, 0x71, 0x75, 0x61, 0x74, 0x20, 0x6d, 0x61, 0x73, 0x73, 0x61, 0x20, 0x71,
				0x75, 0x69, 0x73, 0x20, 0x65, 0x6e, 0x69, 0x6d, 0x2e, 0x20, 0x44, 0x6f, 0x6e, 0x65, 0x63, 0x20, 0x70,
				0x65, 0x64, 0x65, 0x20, 0x6a, 0x75, 0x73, 0x74, 0x6f, 0x2c, 0x20, 0x66, 0x72, 0x69, 0x6e, 0x67, 0x69,
				0x6c, 0x6c, 0x61, 0x20, 0x76, 0x65, 0x6c, 0x2c, 0x20, 0x61, 0x6c, 0x69, 0x71, 0x75, 0x65, 0x74, 0x20,
				0x6e, 0x65, 0x63, 0x2c, 0x20, 0x76, 0x75, 0x6c, 0x70, 0x75, 0x74, 0x61, 0x74, 0x65, 0x20, 0x65, 0x67,
				0x65, 0x74, 0x2c, 0x20, 0x61, 0x72, 0x63, 0x75, 0x2e, 0x20, 0x49, 0x6e, 0x20, 0x65, 0x6e, 0x69, 0x6d,
				0x20, 0x6a, 0x75, 0x73, 0x74, 0x6f, 0x2c, 0x20, 0x72, 0x68, 0x6f, 0x6e, 0x63, 0x75, 0x73, 0x20, 0x75,
				0x74, 0x2c, 0x20, 0x69, 0x6d, 0x70, 0x65, 0x72, 0x64, 0x69, 0x65, 0x74, 0x20, 0x61, 0x2c, 0x20, 0x76,
				0x65, 0x6e, 0x65, 0x6e, 0x61, 0x74, 0x69, 0x73, 0x20, 0x76, 0x69, 0x74, 0x61, 0x65, 0x2c, 0x20, 0x6a,
				0x75, 0x73, 0x74, 0x6f, 0x2e, 0x20, 0x4e, 0x75, 0x6c, 0x6c, 0x61, 0x6d, 0x20, 0x64, 0x69, 0x63, 0x74,
				0x75, 0x6d, 0x20, 0x66, 0x65, 0x6c, 0x69, 0x73, 0x20, 0x65, 0x75, 0x20, 0x70, 0x65, 0x64, 0x65, 0x20,
				0x6d, 0x6f, 0x6c, 0x6c, 0x69, 0x73, 0x20, 0x70, 0x72, 0x65, 0x74, 0x69, 0x75, 0x6d, 0x2e, 0x20, 0x49,
				0x6e, 0x74, 0x65, 0x67, 0x65, 0x72, 0x20, 0x74, 0x69, 0x6e, 0x63, 0x69, 0x64, 0x75, 0x6e, 0x74, 0x2e,
				0x20, 0x43, 0x72, 0x61, 0x73, 0x20, 0x64, 0x61, 0x70, 0x69, 0x62, 0x75, 0x73, 0x2e, 0x20, 0x56, 0x69,
				0x76, 0x61, 0x6d, 0x75, 0x73, 0x20, 0x65, 0x6c, 0x65, 0x6d, 0x65, 0x6e, 0x74, 0x75, 0x6d, 0x20, 0x73,
				0x65, 0x6d, 0x70, 0x65, 0x72, 0x20, 0x6e, 0x69, 0x73, 0x69, 0x2e, 0x20, 0x41, 0x65, 0x6e, 0x65, 0x61,
				0x6e, 0x20, 0x76, 0x75, 0x6c, 0x70, 0x75, 0x74, 0x61, 0x74, 0x65, 0x20, 0x65, 0x6c, 0x65, 0x69, 0x66,
				0x65, 0x6e, 0x64, 0x20, 0x74, 0x65, 0x6c, 0x6c, 0x75, 0x73, 0x2e });
	}

	private void checkArray(final long[][] expected, final byte[] vpack) throws VPackException {
		checkLength(expected.length, vpack);
		final VPackSlice slice = new VPackSlice(vpack);
		for (int i = 0; i < expected.length; i++) {
			final long[] childArray = expected[i];
			final VPackSlice at = slice.get(i);
			assertThat(at.isArray(), is(true));
			assertThat(at.getLength(), is(childArray.length));
			for (int j = 0; j < childArray.length; j++) {
				final VPackSlice atat = at.get(j);
				assertThat(atat.isInteger(), is(true));
				assertThat(atat.getAsLong(), is(childArray[j]));
			}
		}
	}

	private void checkArray(final long[] expected, final byte[] vpack) throws VPackException {
		checkLength(expected.length, vpack);
		final VPackSlice slice = new VPackSlice(vpack);
		for (int i = 0; i < expected.length; i++) {
			final VPackSlice at = slice.get(i);
			assertThat(at.isInteger(), is(true));
			assertThat(at.getAsLong(), is(expected[i]));
		}
	}

	@Test
	public void arrayIterator() throws VPackException {
		final Collection<String> expected = Arrays.asList("a", "b", "c", "d", "e", "f");
		final VPackSlice slice = new VPackSlice(new byte[] { 0x13, 0x0f, 0x41, 0x61, 0x41, 0x62, 0x41, 0x63, 0x41, 0x64,
				0x41, 0x65, 0x41, 0x66, 0x06 });
		final Iterator<VPackSlice> iteratorSlice = slice.arrayIterator();
		for (final Iterator<String> iterator = expected.iterator(); iterator.hasNext();) {
			final String string = iterator.next();
			final VPackSlice next = iteratorSlice.next();
			assertThat(next.isString(), is(true));
			assertThat(next.getAsString(), is(string));
		}
	}

	@Test
	public void objectIterator() throws VPackException {
		// {"a":"test","b":"test","c":"test"}
		final String[] fields = new String[] { "a", "b", "c" };
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, 0x1b, 0x03, 0x41, 0x61, 0x44, 0x74, 0x65, 0x73, 0x74,
				0x41, 0x62, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x63, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11 });
		int i = 0;
		for (final Iterator<Entry<String, VPackSlice>> iterator = slice.objectIterator(); iterator.hasNext();) {
			final Entry<String, VPackSlice> next = iterator.next();
			assertThat(next.getKey(), is(fields[i++]));
			assertThat(next.getValue().getAsString(), is("test"));
		}
	}

	@Test(expected = VPackValueTypeException.class)
	public void nonArrayIterator() {
		final VPackSlice vpack = new VPackSlice(new byte[] { 0x1a });
		vpack.arrayIterator();
	}

	@Test(expected = VPackValueTypeException.class)
	public void nonObjectIterator() {
		final VPackSlice vpack = new VPackSlice(new byte[] { 0x1a });
		vpack.objectIterator();
	}

	@Test
	public void objectEmpty() throws VPackException {
		checkLength(0, new byte[] { 0x0a });
	}

	@Test
	public void objectLength() throws VPackException {
		// {"a": 12, "b": true, "c": "xyz"}
		final int expected = 3;
		checkLength(expected, new byte[] { 0x0b, 0x13, 0x03, 0x41, 0x62, 0x1a, 0x41, 0x61, 0x28, 0x0c, 0x41, 0x63, 0x43,
				0x78, 0x79, 0x7a, 0x06, 0x03, 0x0a });
		checkLength(expected,
			new byte[] { 0x0d, 0x22, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x41, 0x62, 0x03, 0x41, 0x61, 0x28, 0x0c,
					0x41, 0x63, 0x43, 0x78, 0x79, 0x7a, 0x0c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x10, 0x00,
					0x00, 0x00 });
		checkLength(expected, new byte[] { 0x0f, 0x13, 0x03, 0x41, 0x62, 0x03, 0x41, 0x61, 0x28, 0x0c, 0x41, 0x63, 0x43,
				0x78, 0x79, 0x7a, 0x03, 0x06, 0x0a });
	}

	@Test
	public void objectCompactLength() throws VPackException {
		// {"a":1, "b":16}
		checkLength(2, new byte[] { 0x14, 0x0a, 0x41, 0x61, 0x31, 0x41, 0x62, 0x28, 0x10, 0x02 });
	}

	private void checkLength(final int expected, final byte[] vpack) throws VPackException {
		final VPackSlice slice = new VPackSlice(vpack);
		assertThat(slice.getLength(), is(expected));
		assertThat(slice.getByteSize(), is(vpack.length));
	}

	@Test
	public void objectEmptyGet() throws VPackException {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0a });
		final VPackSlice slice2 = slice.get("abc");
		assertThat(slice2.isNone(), is(true));
	}

	@Test
	public void objectSingleEntryString() throws VPackException {
		// {"a":"b"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, 0x07, 0x01, 0x41, 0x61, 0x41, 0x62 });
		final VPackSlice sliceNone = slice.get("abc");
		assertThat(sliceNone.isNone(), is(true));
		final VPackSlice sliceA = slice.get("a");
		assertThat(sliceA.isString(), is(true));
	}

	@Test
	public void objectSorted4Entries() throws VPackException {
		// {"a":"b","c":"d","e":"f","g":"h"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, 0x17, 0x04, 0x41, 0x61, 0x41, 0x62, 0x41, 0x63, 0x41,
				0x64, 0x41, 0x65, 0x41, 0x66, 0x41, 0x67, 0x41, 0x68, 0x03, 0x07, 0x0b, 0x0f });
		checkString("b", slice.get("a"));
		checkString("d", slice.get("c"));
		checkString("f", slice.get("e"));
		checkString("h", slice.get("g"));
	}

	@Test
	public void objectSortedUnder4Entries() throws VPackException {
		// {"a":"b","c":"d","e":"f"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, 0x12, 0x03, 0x41, 0x61, 0x41, 0x62, 0x41, 0x63, 0x41,
				0x64, 0x41, 0x65, 0x41, 0x66, 0x03, 0x07, 0x0b });
		checkString("b", slice.get("a"));
		checkString("d", slice.get("c"));
		checkString("f", slice.get("e"));
	}

	@Test
	public void objectCompact() throws VPackException {
		// {"a":"b","c":"d","e":"f"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x14, 0x0f, 0x41, 0x61, 0x41, 0x62, 0x41, 0x63, 0x41, 0x64,
				0x41, 0x65, 0x41, 0x66, 0x03 });
		checkString("b", slice.get("a"));
		checkString("d", slice.get("c"));
		checkString("f", slice.get("e"));
	}

	private void checkString(final String expected, final VPackSlice slice) throws VPackException {
		assertThat(slice.isString(), is(true));
		assertThat(slice.getAsString(), is(expected));
	}

	@Test
	public void objectKeyValueAtIndex() throws VPackException {
		// {"a":"b","c":"d","e":"f"}
		final String[] keys = { "a", "c", "e" };
		final String[] values = { "b", "d", "f" };
		final VPackSlice slice = new VPackSlice(new byte[] { 0x14, 0x0f, 0x41, 0x61, 0x41, 0x62, 0x41, 0x63, 0x41, 0x64,
				0x41, 0x65, 0x41, 0x66, 0x03 });
		for (int i = 0; i < 3; i++) {
			checkString(keys[i], slice.keyAt(i));
			checkString(values[i], slice.valueAt(i));
		}
	}

	@Test
	public void object1ByteOffset() throws VPackException {
		/*
		 * {"0":{"0":"test","1":"test","2":"test","3":"test","4":"test"},
		 * "1":{"0":"test","1":"test","2":"test","3":"test","4":"test"},
		 * "2":{"0":"test","1":"test","2":"test","3":"test","4":"test"},
		 * "3":{"0":"test","1":"test","2":"test","3":"test","4":"test"},
		 * "4":{"0":"test","1":"test","2":"test","3":"test","4":"test"}}
		 */
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, (byte) 0xe9, 0x05, 0x41, 0x30, 0x0b, 0x2b, 0x05,
				0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x32, 0x44,
				0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x41, 0x31, 0x0b, 0x2b, 0x05, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f,
				0x41, 0x32, 0x0b, 0x2b, 0x05, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x34, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x41, 0x33, 0x0b, 0x2b, 0x05, 0x41,
				0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x32, 0x44, 0x74,
				0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34, 0x44, 0x74, 0x65, 0x73, 0x74,
				0x03, 0x0a, 0x11, 0x18, 0x1f, 0x41, 0x34, 0x0b, 0x2b, 0x05, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74,
				0x41, 0x31, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44,
				0x74, 0x65, 0x73, 0x74, 0x41, 0x34, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x03,
				0x30, 0x5d, (byte) 0x8a, (byte) 0xb7 });
		final int size = 5;
		assertThat(slice.isObject(), is(true));
		assertThat(slice.getLength(), is(size));
		for (int i = 0; i < size; i++) {
			final VPackSlice attr = slice.get(String.valueOf(i));
			assertThat(attr.isObject(), is(true));
			assertThat(attr.getLength(), is(size));
			for (int j = 0; j < size; j++) {
				final VPackSlice childAttr = attr.get(String.valueOf(j));
				assertThat(childAttr.isString(), is(true));
			}
		}
	}

	@Test
	public void object2ByteOffset() throws VPackException {
		/*
		 * {"0":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "1":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "2":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "3":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "4":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "5":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "6":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "7":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "8":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"},
		 * "9":{"0":"test","1":"test","2":"test","3":"test","4":"test","5":
		 * "test","6":"test","7":"test","8":"test","9":"test"}}
		 */
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0c, 0x6f, 0x03, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x41,
				0x30, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x31, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x32, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x33, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x34, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x35, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x36, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x37, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x38, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x41,
				0x39, 0x0b, 0x53, 0x0a, 0x41, 0x30, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x31, 0x44, 0x74, 0x65, 0x73,
				0x74, 0x41, 0x32, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x33, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x34,
				0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x35, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x36, 0x44, 0x74, 0x65,
				0x73, 0x74, 0x41, 0x37, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41, 0x38, 0x44, 0x74, 0x65, 0x73, 0x74, 0x41,
				0x39, 0x44, 0x74, 0x65, 0x73, 0x74, 0x03, 0x0a, 0x11, 0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x09,
				0x00, 0x5e, 0x00, (byte) 0xb3, 0x00, 0x08, 0x01, 0x5d, 0x01, (byte) 0xb2, 0x01, 0x07, 0x02, 0x5c, 0x02,
				(byte) 0xb1, 0x02, 0x06, 0x03 });
		final int size = 10;
		assertThat(slice.isObject(), is(true));
		assertThat(slice.getLength(), is(size));
		for (int i = 0; i < size; i++) {
			final VPackSlice attr = slice.get(String.valueOf(i));
			assertThat(attr.isObject(), is(true));
			assertThat(attr.getLength(), is(size));
			for (int j = 0; j < size; j++) {
				final VPackSlice childAttr = attr.get(String.valueOf(j));
				assertThat(childAttr.isString(), is(true));
			}
		}
	}

	@Test
	public void attributeAdapter() throws VPackException {
		// {1:"a"}
		// {"_key":"a"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, 0x06, 0x01, 0x31, 0x41, 0x61 });
		assertThat(slice.isObject(), is(true));
		assertThat(slice.getLength(), is(1));
		final VPackSlice key = slice.get("_key");
		assertThat(key.isString(), is(true));
		assertThat(key.getAsString(), is("a"));
	}

	@Test
	public void attributeAdapterDefaults() throws VPackException {
		// {1:"a",2:"b",3:"c",4:"d",5:"e"}
		// {"_key":"a","_rev":"b","_id":"c","_from":"d","_to":"e"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x14, 0x12, 0x31, 0x41, 0x61, 0x32, 0x41, 0x62, 0x33, 0x41,
				0x63, 0x34, 0x41, 0x64, 0x35, 0x41, 0x65, 0x05 });
		assertThat(slice.isObject(), is(true));
		assertThat(slice.getLength(), is(5));
		{
			final VPackSlice key = slice.get("_key");
			assertThat(key.isString(), is(true));
			assertThat(key.getAsString(), is("a"));
		}
		{
			final VPackSlice rev = slice.get("_rev");
			assertThat(rev.isString(), is(true));
			assertThat(rev.getAsString(), is("b"));
		}
		{
			final VPackSlice id = slice.get("_id");
			assertThat(id.isString(), is(true));
			assertThat(id.getAsString(), is("c"));
		}
		{
			final VPackSlice from = slice.get("_from");
			assertThat(from.isString(), is(true));
			assertThat(from.getAsString(), is("d"));
		}
		{
			final VPackSlice to = slice.get("_to");
			assertThat(to.isString(), is(true));
			assertThat(to.getAsString(), is("e"));
		}
	}

	@Test
	public void attributeAdapterObjectWithIndexTable() throws VPackException {
		// {1:"a",2:"b",3:"c",4:"d",5:"e"}
		// {"_key":"a","_rev":"b","_id":"c","_from":"d","_to":"e"}
		final VPackSlice slice = new VPackSlice(new byte[] { 0x0b, 0x21, 0x05, 0x31, 0x41, 0x61, 0x32, 0x42, 0x62, 0x62,
				0x33, 0x43, 0x63, 0x63, 0x63, 0x34, 0x44, 0x64, 0x64, 0x64, 0x64, 0x35, 0x45, 0x65, 0x65, 0x65, 0x65,
				0x65, 0x0f, 0x0a, 0x03, 0x06, 0x15 });
		assertThat(slice.isObject(), is(true));
		assertThat(slice.getLength(), is(5));
		{
			final VPackSlice key = slice.get("_key");
			assertThat(key.isString(), is(true));
			assertThat(key.getAsString(), is("a"));
		}
		{
			final VPackSlice rev = slice.get("_rev");
			assertThat(rev.isString(), is(true));
			assertThat(rev.getAsString(), is("bb"));
		}
		{
			final VPackSlice id = slice.get("_id");
			assertThat(id.isString(), is(true));
			assertThat(id.getAsString(), is("ccc"));
		}
		{
			final VPackSlice from = slice.get("_from");
			assertThat(from.isString(), is(true));
			assertThat(from.getAsString(), is("dddd"));
		}
		{
			final VPackSlice to = slice.get("_to");
			assertThat(to.isString(), is(true));
			assertThat(to.getAsString(), is("eeeee"));
		}
	}

	@Test(expected = VPackValueTypeException.class)
	public void getLengthFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.getLength();
	}

	@Test(expected = VPackValueTypeException.class)
	public void getNoObjectGetFail() throws VPackException {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.get("a");
	}

	@Test(expected = VPackValueTypeException.class)
	public void getNoObjectKeyAtFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.keyAt(0);
	}

	@Test(expected = VPackValueTypeException.class)
	public void getNoObjectValueAtFail() {
		final VPackSlice slice = new VPackSlice(new byte[] { 0x00 });
		slice.valueAt(0);
	}
}
