//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift Atomics open source project
//
// Copyright (c) 2020 - 2023 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//


// #############################################################################
// #                                                                           #
// #            DO NOT EDIT THIS FILE; IT IS AUTOGENERATED.                    #
// #                                                                           #
// #############################################################################


import XCTest
import Atomics

/// Exercises all operations in a single-threaded context, verifying
/// they provide the expected results.
class BasicAtomicUnmanagedTests: XCTestCase {
  private let _bar1 = Unmanaged<Bar>.passRetained(Bar(1))
  private let _bar2 = Unmanaged<Bar>.passRetained(Bar(2))

  deinit {
    _bar1.release()
    _bar2.release()
  }

  func test_create_destroy() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .relaxed), _bar2)
  }

  func test_load_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .relaxed), _bar2)
  }

  func test_load_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .acquiring), _bar1)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .acquiring), _bar2)
  }

  func test_load_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .sequentiallyConsistent), _bar1)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .sequentiallyConsistent), _bar2)
  }


  func test_store_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    v.store(_bar2, ordering: .relaxed)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    w.store(_bar1, ordering: .relaxed)
    XCTAssertEqual(w.load(ordering: .relaxed), _bar1)
  }

  func test_store_releasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    v.store(_bar2, ordering: .releasing)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    w.store(_bar1, ordering: .releasing)
    XCTAssertEqual(w.load(ordering: .relaxed), _bar1)
  }

  func test_store_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }
    v.store(_bar2, ordering: .sequentiallyConsistent)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    let w: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar2)
    defer { w.destroy() }
    w.store(_bar1, ordering: .sequentiallyConsistent)
    XCTAssertEqual(w.load(ordering: .relaxed), _bar1)
  }


  func test_exchange_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_bar1, ordering: .relaxed), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    XCTAssertEqual(v.exchange(_bar2, ordering: .relaxed), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    XCTAssertEqual(v.exchange(_bar2, ordering: .relaxed), _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)
  }

  func test_exchange_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_bar1, ordering: .acquiring), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    XCTAssertEqual(v.exchange(_bar2, ordering: .acquiring), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    XCTAssertEqual(v.exchange(_bar2, ordering: .acquiring), _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)
  }

  func test_exchange_releasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_bar1, ordering: .releasing), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    XCTAssertEqual(v.exchange(_bar2, ordering: .releasing), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    XCTAssertEqual(v.exchange(_bar2, ordering: .releasing), _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)
  }

  func test_exchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_bar1, ordering: .acquiringAndReleasing), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    XCTAssertEqual(v.exchange(_bar2, ordering: .acquiringAndReleasing), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    XCTAssertEqual(v.exchange(_bar2, ordering: .acquiringAndReleasing), _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)
  }

  func test_exchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_bar1, ordering: .sequentiallyConsistent), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    XCTAssertEqual(v.exchange(_bar2, ordering: .sequentiallyConsistent), _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    XCTAssertEqual(v.exchange(_bar2, ordering: .sequentiallyConsistent), _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)
  }


  func test_compareExchange_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_releasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .releasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .releasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .acquiringAndReleasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .acquiringAndReleasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }


  func test_weakCompareExchange_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        ordering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        ordering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        ordering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        ordering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_releasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        ordering: .releasing)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        ordering: .releasing)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        ordering: .acquiringAndReleasing)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        ordering: .acquiringAndReleasing)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        ordering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        ordering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }


  func test_compareExchange_relaxed_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_relaxed_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_relaxed_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiring_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiring_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiring_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_releasing_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_releasing_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_releasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiringAndReleasing_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiringAndReleasing_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_acquiringAndReleasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_sequentiallyConsistent_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_sequentiallyConsistent_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_compareExchange_sequentiallyConsistent_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.compareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }


  func test_weakCompareExchange_relaxed_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .relaxed,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .relaxed,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_relaxed_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .relaxed,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .relaxed,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_relaxed_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .relaxed,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .relaxed,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiring_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .acquiring,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .acquiring,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiring_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .acquiring,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .acquiring,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiring_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .acquiring,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .acquiring,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_releasing_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .releasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .releasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_releasing_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .releasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .releasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_releasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .releasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .releasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiringAndReleasing_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiringAndReleasing_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_sequentiallyConsistent_relaxed() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_sequentiallyConsistent_acquiring() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }

  func test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent() {
    let v: UnsafeAtomic<Unmanaged<Bar>> = .create(_bar1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Unmanaged<Bar>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar1,
        desired: _bar2,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar1,
      desired: _bar2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _bar2,
        desired: _bar1,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _bar2)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _bar2,
      desired: _bar1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _bar1)
    XCTAssertEqual(v.load(ordering: .relaxed), _bar1)
  }





#if MANUAL_TEST_DISCOVERY
  public static var allTests: [(String, (BasicAtomicUnmanagedTests) -> () -> ())] = [
    ("test_create_destroy", test_create_destroy),
    ("test_load_relaxed", test_load_relaxed),
    ("test_load_acquiring", test_load_acquiring),
    ("test_load_sequentiallyConsistent", test_load_sequentiallyConsistent),
    ("test_store_relaxed", test_store_relaxed),
    ("test_store_releasing", test_store_releasing),
    ("test_store_sequentiallyConsistent", test_store_sequentiallyConsistent),
    ("test_exchange_relaxed", test_exchange_relaxed),
    ("test_exchange_acquiring", test_exchange_acquiring),
    ("test_exchange_releasing", test_exchange_releasing),
    ("test_exchange_acquiringAndReleasing", test_exchange_acquiringAndReleasing),
    ("test_exchange_sequentiallyConsistent", test_exchange_sequentiallyConsistent),
    ("test_compareExchange_relaxed", test_compareExchange_relaxed),
    ("test_compareExchange_acquiring", test_compareExchange_acquiring),
    ("test_compareExchange_releasing", test_compareExchange_releasing),
    ("test_compareExchange_acquiringAndReleasing", test_compareExchange_acquiringAndReleasing),
    ("test_compareExchange_sequentiallyConsistent", test_compareExchange_sequentiallyConsistent),
    ("test_compareExchange_relaxed_relaxed", test_compareExchange_relaxed_relaxed),
    ("test_compareExchange_relaxed_acquiring", test_compareExchange_relaxed_acquiring),
    ("test_compareExchange_relaxed_sequentiallyConsistent", test_compareExchange_relaxed_sequentiallyConsistent),
    ("test_compareExchange_acquiring_relaxed", test_compareExchange_acquiring_relaxed),
    ("test_compareExchange_acquiring_acquiring", test_compareExchange_acquiring_acquiring),
    ("test_compareExchange_acquiring_sequentiallyConsistent", test_compareExchange_acquiring_sequentiallyConsistent),
    ("test_compareExchange_releasing_relaxed", test_compareExchange_releasing_relaxed),
    ("test_compareExchange_releasing_acquiring", test_compareExchange_releasing_acquiring),
    ("test_compareExchange_releasing_sequentiallyConsistent", test_compareExchange_releasing_sequentiallyConsistent),
    ("test_compareExchange_acquiringAndReleasing_relaxed", test_compareExchange_acquiringAndReleasing_relaxed),
    ("test_compareExchange_acquiringAndReleasing_acquiring", test_compareExchange_acquiringAndReleasing_acquiring),
    ("test_compareExchange_acquiringAndReleasing_sequentiallyConsistent", test_compareExchange_acquiringAndReleasing_sequentiallyConsistent),
    ("test_compareExchange_sequentiallyConsistent_relaxed", test_compareExchange_sequentiallyConsistent_relaxed),
    ("test_compareExchange_sequentiallyConsistent_acquiring", test_compareExchange_sequentiallyConsistent_acquiring),
    ("test_compareExchange_sequentiallyConsistent_sequentiallyConsistent", test_compareExchange_sequentiallyConsistent_sequentiallyConsistent),
    ("test_weakCompareExchange_relaxed_relaxed", test_weakCompareExchange_relaxed_relaxed),
    ("test_weakCompareExchange_relaxed_acquiring", test_weakCompareExchange_relaxed_acquiring),
    ("test_weakCompareExchange_relaxed_sequentiallyConsistent", test_weakCompareExchange_relaxed_sequentiallyConsistent),
    ("test_weakCompareExchange_acquiring_relaxed", test_weakCompareExchange_acquiring_relaxed),
    ("test_weakCompareExchange_acquiring_acquiring", test_weakCompareExchange_acquiring_acquiring),
    ("test_weakCompareExchange_acquiring_sequentiallyConsistent", test_weakCompareExchange_acquiring_sequentiallyConsistent),
    ("test_weakCompareExchange_releasing_relaxed", test_weakCompareExchange_releasing_relaxed),
    ("test_weakCompareExchange_releasing_acquiring", test_weakCompareExchange_releasing_acquiring),
    ("test_weakCompareExchange_releasing_sequentiallyConsistent", test_weakCompareExchange_releasing_sequentiallyConsistent),
    ("test_weakCompareExchange_acquiringAndReleasing_relaxed", test_weakCompareExchange_acquiringAndReleasing_relaxed),
    ("test_weakCompareExchange_acquiringAndReleasing_acquiring", test_weakCompareExchange_acquiringAndReleasing_acquiring),
    ("test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent", test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent),
    ("test_weakCompareExchange_sequentiallyConsistent_relaxed", test_weakCompareExchange_sequentiallyConsistent_relaxed),
    ("test_weakCompareExchange_sequentiallyConsistent_acquiring", test_weakCompareExchange_sequentiallyConsistent_acquiring),
    ("test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent", test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent),
  ]
#endif
}

