//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift Atomics open source project
//
// Copyright (c) 2020 - 2023 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//


// #############################################################################
// #                                                                           #
// #            DO NOT EDIT THIS FILE; IT IS AUTOGENERATED.                    #
// #                                                                           #
// #############################################################################


import XCTest
import Atomics

/// Exercises all operations in a single-threaded context, verifying
/// they provide the expected results.
class BasicAtomicRawRepresentableTests: XCTestCase {

  func test_create_destroy() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .relaxed), Fred.two)
  }

  func test_load_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .relaxed), Fred.two)
  }

  func test_load_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .acquiring), Fred.one)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .acquiring), Fred.two)
  }

  func test_load_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .sequentiallyConsistent), Fred.one)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .sequentiallyConsistent), Fred.two)
  }


  func test_store_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    v.store(Fred.two, ordering: .relaxed)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    w.store(Fred.one, ordering: .relaxed)
    XCTAssertEqual(w.load(ordering: .relaxed), Fred.one)
  }

  func test_store_releasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    v.store(Fred.two, ordering: .releasing)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    w.store(Fred.one, ordering: .releasing)
    XCTAssertEqual(w.load(ordering: .relaxed), Fred.one)
  }

  func test_store_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }
    v.store(Fred.two, ordering: .sequentiallyConsistent)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    let w: UnsafeAtomic<Fred> = .create(Fred.two)
    defer { w.destroy() }
    w.store(Fred.one, ordering: .sequentiallyConsistent)
    XCTAssertEqual(w.load(ordering: .relaxed), Fred.one)
  }


  func test_exchange_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(Fred.one, ordering: .relaxed), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .relaxed), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .relaxed), Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)
  }

  func test_exchange_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(Fred.one, ordering: .acquiring), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .acquiring), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .acquiring), Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)
  }

  func test_exchange_releasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(Fred.one, ordering: .releasing), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .releasing), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .releasing), Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)
  }

  func test_exchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(Fred.one, ordering: .acquiringAndReleasing), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .acquiringAndReleasing), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .acquiringAndReleasing), Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)
  }

  func test_exchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(Fred.one, ordering: .sequentiallyConsistent), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .sequentiallyConsistent), Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    XCTAssertEqual(v.exchange(Fred.two, ordering: .sequentiallyConsistent), Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)
  }


  func test_compareExchange_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_releasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .releasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .releasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .acquiringAndReleasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .acquiringAndReleasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }


  func test_weakCompareExchange_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        ordering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        ordering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        ordering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        ordering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_releasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        ordering: .releasing)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        ordering: .releasing)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        ordering: .acquiringAndReleasing)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        ordering: .acquiringAndReleasing)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        ordering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        ordering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }


  func test_compareExchange_relaxed_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_relaxed_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_relaxed_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiring_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiring_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiring_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_releasing_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_releasing_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_releasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiringAndReleasing_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiringAndReleasing_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_acquiringAndReleasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_sequentiallyConsistent_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_sequentiallyConsistent_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_compareExchange_sequentiallyConsistent_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.compareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }


  func test_weakCompareExchange_relaxed_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .relaxed,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .relaxed,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_relaxed_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .relaxed,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .relaxed,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_relaxed_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .relaxed,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .relaxed,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiring_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .acquiring,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .acquiring,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiring_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .acquiring,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .acquiring,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiring_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .acquiring,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .acquiring,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_releasing_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .releasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .releasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_releasing_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .releasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .releasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_releasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .releasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .releasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiringAndReleasing_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiringAndReleasing_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_sequentiallyConsistent_relaxed() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_sequentiallyConsistent_acquiring() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }

  func test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent() {
    let v: UnsafeAtomic<Fred> = .create(Fred.one)
    defer { v.destroy() }

    var (exchanged, original): (Bool, Fred)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.one,
        desired: Fred.two,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.one,
      desired: Fred.two,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.two)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: Fred.two,
        desired: Fred.one,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, Fred.two)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)

    (exchanged, original) = v.weakCompareExchange(
      expected: Fred.two,
      desired: Fred.one,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, Fred.one)
    XCTAssertEqual(v.load(ordering: .relaxed), Fred.one)
  }





#if MANUAL_TEST_DISCOVERY
  public static var allTests: [(String, (BasicAtomicRawRepresentableTests) -> () -> ())] = [
    ("test_create_destroy", test_create_destroy),
    ("test_load_relaxed", test_load_relaxed),
    ("test_load_acquiring", test_load_acquiring),
    ("test_load_sequentiallyConsistent", test_load_sequentiallyConsistent),
    ("test_store_relaxed", test_store_relaxed),
    ("test_store_releasing", test_store_releasing),
    ("test_store_sequentiallyConsistent", test_store_sequentiallyConsistent),
    ("test_exchange_relaxed", test_exchange_relaxed),
    ("test_exchange_acquiring", test_exchange_acquiring),
    ("test_exchange_releasing", test_exchange_releasing),
    ("test_exchange_acquiringAndReleasing", test_exchange_acquiringAndReleasing),
    ("test_exchange_sequentiallyConsistent", test_exchange_sequentiallyConsistent),
    ("test_compareExchange_relaxed", test_compareExchange_relaxed),
    ("test_compareExchange_acquiring", test_compareExchange_acquiring),
    ("test_compareExchange_releasing", test_compareExchange_releasing),
    ("test_compareExchange_acquiringAndReleasing", test_compareExchange_acquiringAndReleasing),
    ("test_compareExchange_sequentiallyConsistent", test_compareExchange_sequentiallyConsistent),
    ("test_compareExchange_relaxed_relaxed", test_compareExchange_relaxed_relaxed),
    ("test_compareExchange_relaxed_acquiring", test_compareExchange_relaxed_acquiring),
    ("test_compareExchange_relaxed_sequentiallyConsistent", test_compareExchange_relaxed_sequentiallyConsistent),
    ("test_compareExchange_acquiring_relaxed", test_compareExchange_acquiring_relaxed),
    ("test_compareExchange_acquiring_acquiring", test_compareExchange_acquiring_acquiring),
    ("test_compareExchange_acquiring_sequentiallyConsistent", test_compareExchange_acquiring_sequentiallyConsistent),
    ("test_compareExchange_releasing_relaxed", test_compareExchange_releasing_relaxed),
    ("test_compareExchange_releasing_acquiring", test_compareExchange_releasing_acquiring),
    ("test_compareExchange_releasing_sequentiallyConsistent", test_compareExchange_releasing_sequentiallyConsistent),
    ("test_compareExchange_acquiringAndReleasing_relaxed", test_compareExchange_acquiringAndReleasing_relaxed),
    ("test_compareExchange_acquiringAndReleasing_acquiring", test_compareExchange_acquiringAndReleasing_acquiring),
    ("test_compareExchange_acquiringAndReleasing_sequentiallyConsistent", test_compareExchange_acquiringAndReleasing_sequentiallyConsistent),
    ("test_compareExchange_sequentiallyConsistent_relaxed", test_compareExchange_sequentiallyConsistent_relaxed),
    ("test_compareExchange_sequentiallyConsistent_acquiring", test_compareExchange_sequentiallyConsistent_acquiring),
    ("test_compareExchange_sequentiallyConsistent_sequentiallyConsistent", test_compareExchange_sequentiallyConsistent_sequentiallyConsistent),
    ("test_weakCompareExchange_relaxed_relaxed", test_weakCompareExchange_relaxed_relaxed),
    ("test_weakCompareExchange_relaxed_acquiring", test_weakCompareExchange_relaxed_acquiring),
    ("test_weakCompareExchange_relaxed_sequentiallyConsistent", test_weakCompareExchange_relaxed_sequentiallyConsistent),
    ("test_weakCompareExchange_acquiring_relaxed", test_weakCompareExchange_acquiring_relaxed),
    ("test_weakCompareExchange_acquiring_acquiring", test_weakCompareExchange_acquiring_acquiring),
    ("test_weakCompareExchange_acquiring_sequentiallyConsistent", test_weakCompareExchange_acquiring_sequentiallyConsistent),
    ("test_weakCompareExchange_releasing_relaxed", test_weakCompareExchange_releasing_relaxed),
    ("test_weakCompareExchange_releasing_acquiring", test_weakCompareExchange_releasing_acquiring),
    ("test_weakCompareExchange_releasing_sequentiallyConsistent", test_weakCompareExchange_releasing_sequentiallyConsistent),
    ("test_weakCompareExchange_acquiringAndReleasing_relaxed", test_weakCompareExchange_acquiringAndReleasing_relaxed),
    ("test_weakCompareExchange_acquiringAndReleasing_acquiring", test_weakCompareExchange_acquiringAndReleasing_acquiring),
    ("test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent", test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent),
    ("test_weakCompareExchange_sequentiallyConsistent_relaxed", test_weakCompareExchange_sequentiallyConsistent_relaxed),
    ("test_weakCompareExchange_sequentiallyConsistent_acquiring", test_weakCompareExchange_sequentiallyConsistent_acquiring),
    ("test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent", test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent),
  ]
#endif
}

