//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift Atomics open source project
//
// Copyright (c) 2020 - 2023 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See https://swift.org/LICENSE.txt for license information
// See https://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//


// #############################################################################
// #                                                                           #
// #            DO NOT EDIT THIS FILE; IT IS AUTOGENERATED.                    #
// #                                                                           #
// #############################################################################


import XCTest
import Atomics

/// Exercises all operations in a single-threaded context, verifying
/// they provide the expected results.
class BasicAtomicMutablePointerTests: XCTestCase {
  private let _mfoo1: UnsafeMutablePointer<Foo> = {
    let p = UnsafeMutablePointer<Foo>.allocate(capacity: 1)
    p.initialize(to: Foo(1))
    return p
  }()
  private let _mfoo2: UnsafeMutablePointer<Foo> = {
    let p = UnsafeMutablePointer<Foo>.allocate(capacity: 1)
    p.initialize(to: Foo(2))
    return p
  }()

  private var _foo1: UnsafePointer<Foo> { UnsafePointer(_mfoo1) }
  private var _foo2: UnsafePointer<Foo> { UnsafePointer(_mfoo2) }

  deinit {
    _mfoo1.deinitialize(count: 1)
    _mfoo1.deallocate()

    _mfoo2.deinitialize(count: 1)
    _mfoo2.deallocate()
  }

  func test_create_destroy() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .relaxed), _mfoo2)
  }

  func test_load_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .relaxed), _mfoo2)
  }

  func test_load_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .acquiring), _mfoo1)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .acquiring), _mfoo2)
  }

  func test_load_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    XCTAssertEqual(v.load(ordering: .sequentiallyConsistent), _mfoo1)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    XCTAssertEqual(w.load(ordering: .sequentiallyConsistent), _mfoo2)
  }


  func test_store_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    v.store(_mfoo2, ordering: .relaxed)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    w.store(_mfoo1, ordering: .relaxed)
    XCTAssertEqual(w.load(ordering: .relaxed), _mfoo1)
  }

  func test_store_releasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    v.store(_mfoo2, ordering: .releasing)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    w.store(_mfoo1, ordering: .releasing)
    XCTAssertEqual(w.load(ordering: .relaxed), _mfoo1)
  }

  func test_store_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }
    v.store(_mfoo2, ordering: .sequentiallyConsistent)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    let w: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo2)
    defer { w.destroy() }
    w.store(_mfoo1, ordering: .sequentiallyConsistent)
    XCTAssertEqual(w.load(ordering: .relaxed), _mfoo1)
  }


  func test_exchange_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_mfoo1, ordering: .relaxed), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .relaxed), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .relaxed), _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)
  }

  func test_exchange_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_mfoo1, ordering: .acquiring), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .acquiring), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .acquiring), _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)
  }

  func test_exchange_releasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_mfoo1, ordering: .releasing), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .releasing), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .releasing), _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)
  }

  func test_exchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_mfoo1, ordering: .acquiringAndReleasing), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .acquiringAndReleasing), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .acquiringAndReleasing), _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)
  }

  func test_exchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    XCTAssertEqual(v.exchange(_mfoo1, ordering: .sequentiallyConsistent), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .sequentiallyConsistent), _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    XCTAssertEqual(v.exchange(_mfoo2, ordering: .sequentiallyConsistent), _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)
  }


  func test_compareExchange_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_releasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .releasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .releasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .acquiringAndReleasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .acquiringAndReleasing)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }


  func test_weakCompareExchange_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        ordering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        ordering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        ordering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        ordering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_releasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        ordering: .releasing)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        ordering: .releasing)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .releasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiringAndReleasing() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        ordering: .acquiringAndReleasing)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        ordering: .acquiringAndReleasing)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .acquiringAndReleasing)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        ordering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        ordering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      ordering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }


  func test_compareExchange_relaxed_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_relaxed_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_relaxed_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiring_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiring_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiring_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_releasing_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_releasing_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_releasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiringAndReleasing_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiringAndReleasing_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_acquiringAndReleasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_sequentiallyConsistent_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_sequentiallyConsistent_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_compareExchange_sequentiallyConsistent_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertTrue(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.compareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }


  func test_weakCompareExchange_relaxed_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .relaxed,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .relaxed,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_relaxed_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .relaxed,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .relaxed,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_relaxed_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .relaxed,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .relaxed,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .relaxed,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiring_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .acquiring,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .acquiring,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiring_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .acquiring,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .acquiring,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiring_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .acquiring,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .acquiring,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiring,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_releasing_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .releasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .releasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_releasing_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .releasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .releasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_releasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .releasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .releasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .releasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiringAndReleasing_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiringAndReleasing_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .acquiringAndReleasing,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .acquiringAndReleasing,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_sequentiallyConsistent_relaxed() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .relaxed)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .relaxed)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_sequentiallyConsistent_acquiring() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .acquiring)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .acquiring)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }

  func test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent() {
    let v: UnsafeAtomic<UnsafeMutablePointer<Foo>> = .create(_mfoo1)
    defer { v.destroy() }

    var (exchanged, original): (Bool, UnsafeMutablePointer<Foo>)
    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo1,
        desired: _mfoo2,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo1,
      desired: _mfoo2,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo2)

    repeat {
      (exchanged, original) = v.weakCompareExchange(
        expected: _mfoo2,
        desired: _mfoo1,
        successOrdering: .sequentiallyConsistent,
        failureOrdering: .sequentiallyConsistent)
    } while !exchanged
    XCTAssertEqual(original, _mfoo2)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)

    (exchanged, original) = v.weakCompareExchange(
      expected: _mfoo2,
      desired: _mfoo1,
      successOrdering: .sequentiallyConsistent,
      failureOrdering: .sequentiallyConsistent)
    XCTAssertFalse(exchanged)
    XCTAssertEqual(original, _mfoo1)
    XCTAssertEqual(v.load(ordering: .relaxed), _mfoo1)
  }





#if MANUAL_TEST_DISCOVERY
  public static var allTests: [(String, (BasicAtomicMutablePointerTests) -> () -> ())] = [
    ("test_create_destroy", test_create_destroy),
    ("test_load_relaxed", test_load_relaxed),
    ("test_load_acquiring", test_load_acquiring),
    ("test_load_sequentiallyConsistent", test_load_sequentiallyConsistent),
    ("test_store_relaxed", test_store_relaxed),
    ("test_store_releasing", test_store_releasing),
    ("test_store_sequentiallyConsistent", test_store_sequentiallyConsistent),
    ("test_exchange_relaxed", test_exchange_relaxed),
    ("test_exchange_acquiring", test_exchange_acquiring),
    ("test_exchange_releasing", test_exchange_releasing),
    ("test_exchange_acquiringAndReleasing", test_exchange_acquiringAndReleasing),
    ("test_exchange_sequentiallyConsistent", test_exchange_sequentiallyConsistent),
    ("test_compareExchange_relaxed", test_compareExchange_relaxed),
    ("test_compareExchange_acquiring", test_compareExchange_acquiring),
    ("test_compareExchange_releasing", test_compareExchange_releasing),
    ("test_compareExchange_acquiringAndReleasing", test_compareExchange_acquiringAndReleasing),
    ("test_compareExchange_sequentiallyConsistent", test_compareExchange_sequentiallyConsistent),
    ("test_compareExchange_relaxed_relaxed", test_compareExchange_relaxed_relaxed),
    ("test_compareExchange_relaxed_acquiring", test_compareExchange_relaxed_acquiring),
    ("test_compareExchange_relaxed_sequentiallyConsistent", test_compareExchange_relaxed_sequentiallyConsistent),
    ("test_compareExchange_acquiring_relaxed", test_compareExchange_acquiring_relaxed),
    ("test_compareExchange_acquiring_acquiring", test_compareExchange_acquiring_acquiring),
    ("test_compareExchange_acquiring_sequentiallyConsistent", test_compareExchange_acquiring_sequentiallyConsistent),
    ("test_compareExchange_releasing_relaxed", test_compareExchange_releasing_relaxed),
    ("test_compareExchange_releasing_acquiring", test_compareExchange_releasing_acquiring),
    ("test_compareExchange_releasing_sequentiallyConsistent", test_compareExchange_releasing_sequentiallyConsistent),
    ("test_compareExchange_acquiringAndReleasing_relaxed", test_compareExchange_acquiringAndReleasing_relaxed),
    ("test_compareExchange_acquiringAndReleasing_acquiring", test_compareExchange_acquiringAndReleasing_acquiring),
    ("test_compareExchange_acquiringAndReleasing_sequentiallyConsistent", test_compareExchange_acquiringAndReleasing_sequentiallyConsistent),
    ("test_compareExchange_sequentiallyConsistent_relaxed", test_compareExchange_sequentiallyConsistent_relaxed),
    ("test_compareExchange_sequentiallyConsistent_acquiring", test_compareExchange_sequentiallyConsistent_acquiring),
    ("test_compareExchange_sequentiallyConsistent_sequentiallyConsistent", test_compareExchange_sequentiallyConsistent_sequentiallyConsistent),
    ("test_weakCompareExchange_relaxed_relaxed", test_weakCompareExchange_relaxed_relaxed),
    ("test_weakCompareExchange_relaxed_acquiring", test_weakCompareExchange_relaxed_acquiring),
    ("test_weakCompareExchange_relaxed_sequentiallyConsistent", test_weakCompareExchange_relaxed_sequentiallyConsistent),
    ("test_weakCompareExchange_acquiring_relaxed", test_weakCompareExchange_acquiring_relaxed),
    ("test_weakCompareExchange_acquiring_acquiring", test_weakCompareExchange_acquiring_acquiring),
    ("test_weakCompareExchange_acquiring_sequentiallyConsistent", test_weakCompareExchange_acquiring_sequentiallyConsistent),
    ("test_weakCompareExchange_releasing_relaxed", test_weakCompareExchange_releasing_relaxed),
    ("test_weakCompareExchange_releasing_acquiring", test_weakCompareExchange_releasing_acquiring),
    ("test_weakCompareExchange_releasing_sequentiallyConsistent", test_weakCompareExchange_releasing_sequentiallyConsistent),
    ("test_weakCompareExchange_acquiringAndReleasing_relaxed", test_weakCompareExchange_acquiringAndReleasing_relaxed),
    ("test_weakCompareExchange_acquiringAndReleasing_acquiring", test_weakCompareExchange_acquiringAndReleasing_acquiring),
    ("test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent", test_weakCompareExchange_acquiringAndReleasing_sequentiallyConsistent),
    ("test_weakCompareExchange_sequentiallyConsistent_relaxed", test_weakCompareExchange_sequentiallyConsistent_relaxed),
    ("test_weakCompareExchange_sequentiallyConsistent_acquiring", test_weakCompareExchange_sequentiallyConsistent_acquiring),
    ("test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent", test_weakCompareExchange_sequentiallyConsistent_sequentiallyConsistent),
  ]
#endif
}

