/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */

import { expectTransformation } from '../test-helpers';

describe('Jasmine to Vitest Transformer', () => {
  describe('transformSpies', () => {
    const testCases = [
      {
        description: 'should transform spyOn(object, "method") to vi.spyOn(object, "method")',
        input: `spyOn(service, 'myMethod');`,
        expected: `vi.spyOn(service, 'myMethod');`,
      },
      {
        description: 'should transform .and.returnValue(...) to .mockReturnValue(...)',
        input: `spyOn(service, 'myMethod').and.returnValue(42);`,
        expected: `vi.spyOn(service, 'myMethod').mockReturnValue(42);`,
      },
      {
        description: 'should transform .and.returnValues() to chained .mockReturnValueOnce() calls',
        input: `spyOn(service, 'myMethod').and.returnValues('a', 'b', 'c');`,
        expected: `vi.spyOn(service, 'myMethod').mockReturnValueOnce('a').mockReturnValueOnce('b').mockReturnValueOnce('c');`,
      },
      {
        description: 'should transform .and.callFake(...) to .mockImplementation(...)',
        input: `spyOn(service, 'myMethod').and.callFake(() => 'fake');`,
        expected: `vi.spyOn(service, 'myMethod').mockImplementation(() => 'fake');`,
      },
      {
        description: 'should remove .and.callThrough()',
        input: `spyOn(service, 'myMethod').and.callThrough();`,
        expected: `vi.spyOn(service, 'myMethod');`,
      },
      {
        description: 'should transform jasmine.createSpy("name") to vi.fn()',
        input: `const mySpy = jasmine.createSpy('mySpy');`,
        expected: `const mySpy = vi.fn();`,
      },
      {
        description: 'should transform jasmine.createSpy("name", fn) to vi.fn(fn)',
        input: `const mySpy = jasmine.createSpy('mySpy', () => 'foo');`,
        expected: `const mySpy = vi.fn(() => 'foo');`,
      },
      {
        description: 'should transform spyOnProperty(object, "prop") to vi.spyOn(object, "prop")',
        input: `spyOnProperty(service, 'myProp');`,
        expected: `vi.spyOn(service, 'myProp');`,
      },
      {
        description: 'should transform .and.stub() to .mockImplementation(() => {})',
        input: `spyOn(service, 'myMethod').and.stub();`,
        expected: `vi.spyOn(service, 'myMethod').mockImplementation(() => {});`,
      },
      {
        description: 'should add a TODO for jasmine.spyOnAllFunctions(object)',
        input: `jasmine.spyOnAllFunctions(myObject);`,
        expected: `// TODO: vitest-migration: Vitest does not have a direct equivalent for jasmine.spyOnAllFunctions(). Please spy on individual methods manually using vi.spyOn(). See: https://vitest.dev/api/vi.html#vi-spyon
          jasmine.spyOnAllFunctions(myObject);
        `,
      },
      {
        description: 'should handle chained calls on jasmine.createSpy()',
        input: `const mySpy = jasmine.createSpy('mySpy').and.returnValue(true);`,
        expected: `const mySpy = vi.fn().mockReturnValue(true);`,
      },
      {
        description: 'should handle .and.returnValues() with no arguments',
        input: `spyOn(service, 'myMethod').and.returnValues();`,
        expected: `vi.spyOn(service, 'myMethod');`,
      },
      {
        description:
          'should transform .and.throwError("message") to .mockImplementation(() => { throw new Error("message") })',
        input: `spyOn(service, 'myMethod').and.throwError('Something went wrong');`,
        expected: `vi.spyOn(service, 'myMethod').mockImplementation(() => { throw new Error('Something went wrong') });`,
      },
      {
        description:
          'should transform .and.throwError(new Error("message")) to .mockImplementation(() => { throw new Error("message") })',
        input: `spyOn(service, 'myMethod').and.throwError(new Error('Custom Error'));`,
        expected: `vi.spyOn(service, 'myMethod').mockImplementation(() => { throw new Error('Custom Error') });`,
      },
      {
        description: 'should transform .and.resolveTo(value) to .mockResolvedValue(value)',
        input: `spyOn(service, 'myMethod').and.resolveTo('some value');`,
        expected: `vi.spyOn(service, 'myMethod').mockResolvedValue('some value');`,
      },
      {
        description: 'should transform .and.rejectWith(error) to .mockRejectedValue(error)',
        input: `spyOn(service, 'myMethod').and.rejectWith('some error');`,
        expected: `vi.spyOn(service, 'myMethod').mockRejectedValue('some error');`,
      },
      {
        description: 'should add a TODO for an unsupported spy strategy',
        input: `spyOn(service, 'myMethod').and.unknownStrategy();`,
        expected: `// TODO: vitest-migration: Unsupported spy strategy ".and.unknownStrategy()" found. Please migrate this manually. See: https://vitest.dev/api/mocked.html#mock
vi.spyOn(service, 'myMethod').and.unknownStrategy();`,
      },
    ];

    testCases.forEach(({ description, input, expected }) => {
      it(description, async () => {
        await expectTransformation(input, expected);
      });
    });
  });

  describe('transformCreateSpyObj', () => {
    const testCases = [
      {
        description: 'should transform jasmine.createSpyObj with an array of methods',
        input: `const myService = jasmine.createSpyObj('MyService', ['methodA', 'methodB']);`,
        expected: `const myService = {
          methodA: vi.fn().mockName("MyService.methodA"),
          methodB: vi.fn().mockName("MyService.methodB"),
        };`,
      },
      {
        description:
          'should transform jasmine.createSpyObj with an array of methods without base name',
        input: `const myService = jasmine.createSpyObj(['methodA', 'methodB']);`,
        expected: `const myService = {
          methodA: vi.fn(),
          methodB: vi.fn(),
        };`,
      },
      {
        description: 'should add a TODO if the second argument is not a literal',
        input: `const myService = jasmine.createSpyObj('MyService', methodNames);`,
        expected: `
          // TODO: vitest-migration: Cannot transform jasmine.createSpyObj with a dynamic variable. Please migrate this manually. See: https://vitest.dev/api/vi.html#vi-fn
          const myService = jasmine.createSpyObj('MyService', methodNames);
        `,
      },
      {
        description: 'should transform jasmine.createSpyObj with an object of return values',
        input: `const myService = jasmine.createSpyObj('MyService', { methodA: 'foo', methodB: 42 });`,
        expected: `const myService = {
          methodA: vi.fn().mockName("MyService.methodA").mockReturnValue('foo'),
          methodB: vi.fn().mockName("MyService.methodB").mockReturnValue(42),
        };`,
      },
      {
        description:
          'should transform jasmine.createSpyObj with an object of return values without base name',
        input: `const myService = jasmine.createSpyObj({ methodA: 'foo', methodB: 42 });`,
        expected: `const myService = {
          methodA: vi.fn().mockReturnValue('foo'),
          methodB: vi.fn().mockReturnValue(42),
        };`,
      },
      {
        description:
          'should transform jasmine.createSpyObj with an object of return values containing an asymmetric matcher',
        input: `const myService = jasmine.createSpyObj('MyService', { methodA: jasmine.any(String) });`,
        expected: `const myService = {
          methodA: vi.fn().mockName("MyService.methodA").mockReturnValue(expect.any(String)),
        };`,
      },
      {
        description: 'should add a TODO for jasmine.createSpyObj with only base name argument',
        input: `const myService = jasmine.createSpyObj('MyService');`,
        expected: `
          // TODO: vitest-migration: jasmine.createSpyObj called with a single argument is not supported for transformation. See: https://vitest.dev/api/vi.html#vi-fn
          const myService = jasmine.createSpyObj('MyService');
        `,
      },
      {
        description: 'should transform jasmine.createSpyObj with a property map',
        input: `const myService = jasmine.createSpyObj('MyService', ['methodA'], { propA: 'valueA' });`,
        expected: `const myService = {
          methodA: vi.fn().mockName("MyService.methodA"),
          propA: 'valueA',
        };`,
      },
      {
        description: 'should transform jasmine.createSpyObj with a method map and a property map',
        input: `const myService = jasmine.createSpyObj('MyService', { methodA: 'foo' }, { propA: 'valueA' });`,
        expected: `const myService = {
          methodA: vi.fn().mockName("MyService.methodA").mockReturnValue('foo'),
          propA: 'valueA',
        };`,
      },
      {
        description:
          'should transform jasmine.createSpyObj with a method map and a property map without base name',
        input: `const myService = jasmine.createSpyObj({ methodA: 'foo' }, { propA: 'valueA' });`,
        expected: `const myService = {
          methodA: vi.fn().mockReturnValue('foo'),
          propA: 'valueA',
        };`,
      },
      {
        description: 'should ignore non-string literals in the method array',
        input: `const myService = jasmine.createSpyObj('MyService', ['methodA', 123, someVar]);`,
        expected: `const myService = {
          methodA: vi.fn().mockName("MyService.methodA"),
        };`,
      },
    ];

    testCases.forEach(({ description, input, expected }) => {
      it(description, async () => {
        await expectTransformation(input, expected);
      });
    });
  });

  describe('transformSpyReset', () => {
    const testCases = [
      {
        description: 'should transform spy.calls.reset() to spy.mockClear()',
        input: `mySpy.calls.reset();`,
        expected: `mySpy.mockClear();`,
      },
    ];

    testCases.forEach(({ description, input, expected }) => {
      it(description, async () => {
        await expectTransformation(input, expected);
      });
    });
  });

  describe('transformSpyCallInspection', () => {
    const testCases = [
      {
        description: 'should transform spy.calls.any()',
        input: `expect(mySpy.calls.any()).toBe(true);`,
        expected: `expect(vi.mocked(mySpy).mock.calls.length > 0).toBe(true);`,
      },
      {
        description: 'should transform spy.calls.count()',
        input: `expect(mySpy.calls.count()).toBe(1);`,
        expected: `expect(vi.mocked(mySpy).mock.calls.length).toBe(1);`,
      },
      {
        description: 'should transform spy.calls.argsFor(0)',
        input: `const args = mySpy.calls.argsFor(0);`,
        expected: `const args = vi.mocked(mySpy).mock.calls[0];`,
      },
      {
        description: 'should transform spy.calls.allArgs()',
        input: `const allArgs = mySpy.calls.allArgs();`,
        expected: `const allArgs = vi.mocked(mySpy).mock.calls;`,
      },
      {
        description: 'should transform spy.calls.all()',
        input: `const allCalls = mySpy.calls.all();`,
        expected: `const allCalls = vi.mocked(mySpy).mock.calls;`,
      },
      {
        description: 'should transform spy.calls.mostRecent().args',
        input: `const recentArgs = mySpy.calls.mostRecent().args;`,
        expected: `const recentArgs = vi.mocked(mySpy).mock.lastCall;`,
      },
      {
        description: 'should transform spy.calls.first()',
        input: `const firstCall = mySpy.calls.first();`,
        expected: `const firstCall = vi.mocked(mySpy).mock.calls[0];`,
      },
      {
        description: 'should add a TODO for spy.calls.mostRecent() without .args',
        input: `const mostRecent = mySpy.calls.mostRecent();`,
        expected: `// TODO: vitest-migration: Direct usage of mostRecent() is not supported. Please refactor to access .args directly or use vi.mocked(spy).mock.lastCall. See: https://vitest.dev/api/mocked.html#mock-lastcall
const mostRecent = mySpy.calls.mostRecent();`,
      },
    ];

    testCases.forEach(({ description, input, expected }) => {
      it(description, async () => {
        await expectTransformation(input, expected);
      });
    });
  });
});
