/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */

import type { BuilderContext, BuilderOutput } from '@angular-devkit/architect';
import assert from 'node:assert';
import path from 'node:path';
import type { Vitest } from 'vitest/node';
import {
  DevServerExternalResultMetadata,
  updateExternalMetadata,
} from '../../../../tools/vite/utils';
import { assertIsError } from '../../../../utils/error';
import {
  type FullResult,
  type IncrementalResult,
  type ResultFile,
  ResultKind,
} from '../../../application/results';
import { NormalizedUnitTestBuilderOptions } from '../../options';
import type { TestExecutor } from '../api';
import { setupBrowserConfiguration } from './browser-provider';
import { findVitestBaseConfig } from './configuration';
import { createVitestConfigPlugin, createVitestPlugins } from './plugins';

export class VitestExecutor implements TestExecutor {
  private vitest: Vitest | undefined;
  private normalizePath: ((id: string) => string) | undefined;
  private readonly projectName: string;
  private readonly options: NormalizedUnitTestBuilderOptions;
  private readonly logger: BuilderContext['logger'];
  private readonly buildResultFiles = new Map<string, ResultFile>();
  private readonly externalMetadata: DevServerExternalResultMetadata = {
    implicitBrowser: [],
    implicitServer: [],
    explicitBrowser: [],
    explicitServer: [],
  };

  // This is a reverse map of the entry points created in `build-options.ts`.
  // It is used by the in-memory provider plugin to map the requested test file
  // path back to its bundled output path.
  // Example: `Map<'/path/to/src/app.spec.ts', 'spec-src-app-spec'>`
  private readonly testFileToEntryPoint = new Map<string, string>();
  private readonly entryPointToTestFile = new Map<string, string>();

  constructor(
    projectName: string,
    options: NormalizedUnitTestBuilderOptions,
    testEntryPointMappings: Map<string, string> | undefined,
    logger: BuilderContext['logger'],
  ) {
    this.projectName = projectName;
    this.options = options;
    this.logger = logger;

    if (testEntryPointMappings) {
      for (const [entryPoint, testFile] of testEntryPointMappings) {
        this.testFileToEntryPoint.set(testFile, entryPoint);
        this.entryPointToTestFile.set(entryPoint + '.js', testFile);
      }
    }
  }

  async *execute(buildResult: FullResult | IncrementalResult): AsyncIterable<BuilderOutput> {
    this.normalizePath ??= (await import('vite')).normalizePath;

    if (buildResult.kind === ResultKind.Full) {
      this.buildResultFiles.clear();
      for (const [path, file] of Object.entries(buildResult.files)) {
        this.buildResultFiles.set(this.normalizePath(path), file);
      }
    } else {
      for (const file of buildResult.removed) {
        this.buildResultFiles.delete(this.normalizePath(file.path));
      }
      for (const [path, file] of Object.entries(buildResult.files)) {
        this.buildResultFiles.set(this.normalizePath(path), file);
      }
    }

    updateExternalMetadata(buildResult, this.externalMetadata, undefined, true);

    // Initialize Vitest if not already present.
    this.vitest ??= await this.initializeVitest();
    const vitest = this.vitest;

    let testResults;
    if (buildResult.kind === ResultKind.Incremental) {
      // To rerun tests, Vitest needs the original test file paths, not the output paths.
      const modifiedSourceFiles = new Set<string>();
      for (const modifiedFile of [...buildResult.modified, ...buildResult.added]) {
        // The `modified` files in the build result are the output paths.
        // We need to find the original source file path to pass to Vitest.
        const source = this.entryPointToTestFile.get(modifiedFile);
        if (source) {
          modifiedSourceFiles.add(source);
        }
        vitest.invalidateFile(
          this.normalizePath(path.join(this.options.workspaceRoot, modifiedFile)),
        );
      }

      const specsToRerun = [];
      for (const file of modifiedSourceFiles) {
        vitest.invalidateFile(file);
        const specs = vitest.getModuleSpecifications(file);
        if (specs) {
          specsToRerun.push(...specs);
        }
      }

      if (specsToRerun.length > 0) {
        testResults = await vitest.rerunTestSpecifications(specsToRerun);
      }
    }

    // Check if all the tests pass to calculate the result
    const testModules = testResults?.testModules ?? this.vitest.state.getTestModules();

    yield { success: testModules.every((testModule) => testModule.ok()) };
  }

  async [Symbol.asyncDispose](): Promise<void> {
    await this.vitest?.close();
  }

  private prepareSetupFiles(): string[] {
    const { setupFiles } = this.options;
    // Add setup file entries for TestBed initialization and project polyfills
    const testSetupFiles = ['init-testbed.js', ...setupFiles];

    // TODO: Provide additional result metadata to avoid needing to extract based on filename
    if (this.buildResultFiles.has('polyfills.js')) {
      testSetupFiles.unshift('polyfills.js');
    }

    return testSetupFiles;
  }

  private async initializeVitest(): Promise<Vitest> {
    const {
      coverage,
      reporters,
      outputFile,
      workspaceRoot,
      browsers,
      debug,
      watch,
      browserViewport,
      ui,
      projectRoot,
      runnerConfig,
      projectSourceRoot,
      cacheOptions,
    } = this.options;
    const projectName = this.projectName;

    let vitestNodeModule;
    try {
      vitestNodeModule = await import('vitest/node');
    } catch (error: unknown) {
      assertIsError(error);
      if (error.code !== 'ERR_MODULE_NOT_FOUND') {
        throw error;
      }
      throw new Error(
        'The `vitest` package was not found. Please install the package and rerun the test command.',
      );
    }
    const { startVitest } = vitestNodeModule;

    // Setup vitest browser options if configured
    const browserOptions = await setupBrowserConfiguration(
      browsers,
      debug,
      projectSourceRoot,
      browserViewport,
    );
    if (browserOptions.errors?.length) {
      throw new Error(browserOptions.errors.join('\n'));
    }

    assert(
      this.buildResultFiles.size > 0,
      'buildResult must be available before initializing vitest',
    );

    const testSetupFiles = this.prepareSetupFiles();
    const projectPlugins = createVitestPlugins({
      workspaceRoot,
      projectSourceRoot,
      projectName,
      buildResultFiles: this.buildResultFiles,
      testFileToEntryPoint: this.testFileToEntryPoint,
    });

    const debugOptions = debug
      ? {
          inspectBrk: true,
          isolate: false,
          fileParallelism: false,
        }
      : {};

    const externalConfigPath =
      runnerConfig === true
        ? await findVitestBaseConfig([projectRoot, workspaceRoot])
        : runnerConfig;

    let project = projectName;
    if (debug && browserOptions.browser?.instances) {
      if (browserOptions.browser.instances.length > 1) {
        this.logger.warn(
          'Multiple browsers are configured, but only the first browser will be used for debugging.',
        );
      }

      // When running browser tests, Vitest appends the browser name to the project identifier.
      // The project name must match this augmented name to ensure the correct project is targeted.
      project = `${projectName} (${browserOptions.browser.instances[0].browser})`;
    }

    return startVitest(
      'test',
      undefined,
      {
        config: externalConfigPath,
        root: workspaceRoot,
        project,
        outputFile,
        cache: cacheOptions.enabled ? undefined : false,
        testNamePattern: this.options.filter,
        watch,
        ...(typeof ui === 'boolean' ? { ui } : {}),
        ...debugOptions,
      },
      {
        // Note `.vitest` is auto appended to the path.
        cacheDir: cacheOptions.path,
        server: {
          // Disable the actual file watcher. The boolean watch option above should still
          // be enabled as it controls other internal behavior related to rerunning tests.
          watch: null,
        },
        plugins: [
          await createVitestConfigPlugin({
            browser: browserOptions.browser,
            coverage,
            projectName,
            projectSourceRoot,
            optimizeDepsInclude: this.externalMetadata.implicitBrowser,
            reporters,
            setupFiles: testSetupFiles,
            projectPlugins,
            include: [...this.testFileToEntryPoint.keys()].filter(
              // Filter internal entries
              (entry) => !entry.startsWith('angular:'),
            ),
          }),
        ],
      },
    );
  }
}
