/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.dev/license
 */

import { SchematicTestRunner, UnitTestTree } from '@angular-devkit/schematics/testing';
import { Schema as ApplicationOptions } from '../application/schema';
import { Schema as WorkspaceOptions } from '../workspace/schema';
import { Schema as ServiceOptions } from './schema';

describe('Interceptor Schematic', () => {
  const schematicRunner = new SchematicTestRunner(
    '@schematics/angular',
    require.resolve('../collection.json'),
  );
  const defaultOptions: ServiceOptions = {
    name: 'foo',
    flat: false,
    project: 'bar',
  };

  const workspaceOptions: WorkspaceOptions = {
    name: 'workspace',
    newProjectRoot: 'projects',
    version: '6.0.0',
  };

  const appOptions: ApplicationOptions = {
    name: 'bar',
    inlineStyle: false,
    inlineTemplate: false,
    routing: false,
    skipPackageJson: false,
  };
  let appTree: UnitTestTree;
  beforeEach(async () => {
    appTree = await schematicRunner.runSchematic('workspace', workspaceOptions);
    appTree = await schematicRunner.runSchematic('application', appOptions, appTree);
  });

  it('should create an interceptor', async () => {
    const options = { ...defaultOptions };

    const tree = await schematicRunner.runSchematic('interceptor', options, appTree);

    const files = tree.files;
    expect(files).toContain('/projects/bar/src/app/foo/foo-interceptor.spec.ts');
    expect(files).toContain('/projects/bar/src/app/foo/foo-interceptor.ts');
  });

  it('should use a `.` type separator when specified', async () => {
    const options = { ...defaultOptions, typeSeparator: '.' };

    const tree = await schematicRunner.runSchematic('interceptor', options, appTree);
    const files = tree.files;
    expect(files).toContain('/projects/bar/src/app/foo/foo.interceptor.spec.ts');
    expect(files).toContain('/projects/bar/src/app/foo/foo.interceptor.ts');
    const specContent = tree.readContent('/projects/bar/src/app/foo/foo.interceptor.spec.ts');
    expect(specContent).toContain(`'./foo.interceptor'`);
  });

  it('should use a `-` type separator when specified', async () => {
    const options = { ...defaultOptions, typeSeparator: '-' };

    const tree = await schematicRunner.runSchematic('interceptor', options, appTree);
    const files = tree.files;
    expect(files).toContain('/projects/bar/src/app/foo/foo-interceptor.spec.ts');
    expect(files).toContain('/projects/bar/src/app/foo/foo-interceptor.ts');
    const specContent = tree.readContent('/projects/bar/src/app/foo/foo-interceptor.spec.ts');
    expect(specContent).toContain(`'./foo-interceptor'`);
  });

  it('should respect the skipTests flag', async () => {
    const options = { ...defaultOptions, skipTests: true };

    const tree = await schematicRunner.runSchematic('interceptor', options, appTree);

    const files = tree.files;
    expect(files).toContain('/projects/bar/src/app/foo/foo-interceptor.ts');
    expect(files).not.toContain('/projects/bar/src/app/foo/foo-interceptor.spec.ts');
  });

  it('should respect the sourceRoot value', async () => {
    const config = JSON.parse(appTree.readContent('/angular.json'));
    config.projects.bar.sourceRoot = 'projects/bar/custom';
    appTree.overwrite('/angular.json', JSON.stringify(config, null, 2));
    appTree = await schematicRunner.runSchematic('interceptor', defaultOptions, appTree);

    expect(appTree.files).toContain('/projects/bar/custom/app/foo/foo-interceptor.ts');
  });

  it('should create a functional interceptor', async () => {
    const tree = await schematicRunner.runSchematic(
      'interceptor',
      { ...defaultOptions, functional: true },
      appTree,
    );

    const fileString = tree.readContent('/projects/bar/src/app/foo/foo-interceptor.ts');
    expect(fileString).toContain(
      'export const fooInterceptor: HttpInterceptorFn = (req, next) => {',
    );
  });

  it('should create a helper function to run a functional interceptor in a test', async () => {
    const tree = await schematicRunner.runSchematic(
      'interceptor',
      { ...defaultOptions, functional: true },
      appTree,
    );

    const fileString = tree.readContent('/projects/bar/src/app/foo/foo-interceptor.spec.ts');
    expect(fileString).toContain('const interceptor: HttpInterceptorFn = (req, next) => ');
    expect(fileString).toContain('TestBed.runInInjectionContext(() => fooInterceptor(req, next));');
  });
});
