export type useDialogOptions<D, DE extends D | undefined> = {
  /**
   * Default data to pass to the dialog when .open() is called
   */
  defaultData?: DE;
  /**
   * If specified, the dialog will remain mounted for this many milliseconds.
   * Useful for allowing a close animation to play before unmounting the dialog.
   */
  unmountDelayInMs?: number;
  /**
   * By default, the dialog will be hidden if the hook is unmounted. Set this to
   * true if you want the dialog to persist even when the hook is unmounted.
   */
  persistOnUnmount?: boolean;
};

export type useDialogReturn<D, R, DE extends D | undefined> = {
  /**
   * Opens the dialog, and passes the provided data as props to the dialog
   * component. Returns a promise that resolves when the dialog is closed.
   *
   * The promise will resolve to the value that is passed to `handleClose`
   * within the dialog component.
   */
  open: DE extends undefined
    ? (data: D) => Promise<R | undefined>
    : (data?: D) => Promise<R | undefined>;
  /**
   * Force closes the dialog. Generally avoid calling this method as it can
   * lead to poor user experience
   */
  close: () => void;
  /**
   * Updates the data being passed to the dialog. This should only be called
   * when the dialog is already open, and you need to change the data on the fly
   */
  updateData: (data: D) => void;
};
