//
//  EnvironmentStorage.swift
//
//
//  Created by Wouter Hennen on 14/06/2023.
//

import SwiftSyntax
import SwiftSyntaxMacros
import SwiftDiagnostics

public struct EnvironmentStorage: MemberAttributeMacro {

    enum Kind: String {
        case environmentValues = "EnvironmentValues"
        case focusedValues = "FocusedValues"
        case environmentStorage = "EnvironmentStorage"

        init?(node: TokenKind) {
            if case .identifier(let name) = node {
                self.init(rawValue: name)
            } else {
                return nil
            }
        }

        var matchedType: String {
            if self == .environmentStorage {
                return Self.environmentValues.rawValue
            }

            return rawValue
        }

        var childMacro: String {
            switch self {
            case .environmentValues, .environmentStorage:
                "EnvironmentKey"
            case .focusedValues:
                "FocusedValueKey"
            }
        }
    }

    public static func expansion(
        of node: AttributeSyntax,
        attachedTo declaration: some DeclGroupSyntax,
        providingAttributesFor member: some DeclSyntaxProtocol,
        in context: some MacroExpansionContext
    ) throws -> [AttributeSyntax] {

        let macroName = node.attributeName.as(IdentifierTypeSyntax.self)?.name.tokenKind
        guard let macroName, let kind = Kind(node: macroName) else { return [] }

        let attachedName = declaration.as(ExtensionDeclSyntax.self)?.extendedType.as(IdentifierTypeSyntax.self)?.name.tokenKind

        // Check if macro is attached to extension type
        guard case .identifier(let attachedName) = attachedName else {
            let feedback = Feedback.notAttachedToExtension
            context.diagnose(Diagnostic(node: Syntax(node), message: feedback))
            return []
        }

        // Check if extension type is correct for the applied macro
        guard attachedName == kind.matchedType else {
            let feedback = Feedback.attachedToWrongType(macro: kind.rawValue, wrongType: attachedName, correctType: kind.matchedType)
            context.diagnose(Diagnostic(node: Syntax(node), message: feedback))
            return []
        }

        // Only attach macro if member is a variable.
        // Otherwise, it will also get attached to the structs generated by @EnvironmentValue
        guard member.is(VariableDeclSyntax.self) else {
            return []
        }

        return [
            AttributeSyntax(
                atSign: .atSignToken(),
                attributeName: IdentifierTypeSyntax(name: .identifier(kind.childMacro))
            )
        ]
    }
}
