// Unity C# reference source
// Copyright (c) Unity Technologies. For terms of use, see
// https://unity3d.com/legal/licenses/Unity_Reference_Only_License

using System;
using System.IO;
using UnityEditor.Build;
using UnityEditor.Build.Profile;
using UnityEditor.UIElements;
using UnityEngine;
using UnityEngine.UIElements;

namespace UnityEditor.Modules
{
    internal abstract class DefaultBuildProfileExtension : IBuildProfileExtension
    {
        static readonly GUIContent k_DevelopmentBuild = EditorGUIUtility.TrTextContent("Development Build");
        static readonly GUIContent k_AutoconnectProfiler = EditorGUIUtility.TrTextContent("Autoconnect Profiler", "When the build is started, an open Profiler Window will automatically connect to the Player and start profiling. The \"Build And Run\" option will also automatically open the Profiler Window.");
        static readonly GUIContent k_AutoconnectProfilerDisabled = EditorGUIUtility.TrTextContent("Autoconnect Profiler", "Profiling is only enabled in a Development Player.");
        static readonly GUIContent k_BuildWithDeepProfiler = EditorGUIUtility.TrTextContent("Deep Profiling Support", "Build Player with Deep Profiling Support. This might affect Player performance.");
        static readonly GUIContent k_BuildWithDeepProfilerDisabled = EditorGUIUtility.TrTextContent("Deep Profiling Support", "Profiling is only enabled in a Development Player.");
        static readonly GUIContent k_AllowDebugging = EditorGUIUtility.TrTextContent("Script Debugging", "Enable this setting to allow your script code to be debugged.");
        static readonly GUIContent k_WaitForManagedDebugger = EditorGUIUtility.TrTextContent("Wait For Managed Debugger", "Show a dialog where you can attach a managed debugger before any script execution. Can also use volume Up or Down button to confirm on Android.");
        static readonly GUIContent k_ManagedDebuggerFixedPort = EditorGUIUtility.TrTextContent("Managed Debugger Fixed Port", "Use the specified port to attach to the managed debugger. If 0, the port will be automatically selected.");
        static readonly GUIContent k_CompressionMethod = EditorGUIUtility.TrTextContent("Compression Method", "Compression applied to Player data (scenes and resources).\nDefault - none or default platform compression.\nLZ4 - fast compression suitable for Development Builds.\nLZ4HC - higher compression rate variance of LZ4, causes longer build times. Works best for Release Builds.");
        static readonly GUIContent k_ExplicitNullChecks = EditorGUIUtility.TrTextContent("Explicit Null Checks");
        static readonly GUIContent k_ExplicitDivideByZeroChecks = EditorGUIUtility.TrTextContent("Divide By Zero Checks");
        static readonly GUIContent k_ExplicitArrayBoundsChecks = EditorGUIUtility.TrTextContent("Array Bounds Checks");
        static readonly Compression[] k_CompressionTypes =
        {
            Compression.None,
            Compression.Lz4,
            Compression.Lz4HC
        };
        static readonly GUIContent[] k_CompressionStrings =
        {
            EditorGUIUtility.TrTextContent("Default"),
            EditorGUIUtility.TrTextContent("LZ4"),
            EditorGUIUtility.TrTextContent("LZ4HC"),
        };
        static readonly GUIContent k_InstallInBuildFolder = EditorGUIUtility.TrTextContent("Install into source code 'build' folder", "Install into source checkout 'build' folder, for debugging with source code");
        static readonly GUIContent k_InstallInBuildFolderHelp = EditorGUIUtility.TrIconContent("_Help", "Open documentation about source code building and debugging");

        SerializedProperty m_Development;
        SerializedProperty m_ConnectProfiler;
        SerializedProperty m_BuildWithDeepProfilingSupport;
        SerializedProperty m_AllowDebugging;
        SerializedProperty m_WaitForManagedDebugger;
        SerializedProperty m_ManagedDebuggerFixedPort;
        SerializedProperty m_ExplicitNullChecks;
        SerializedProperty m_ExplicitDivideByZeroChecks;
        SerializedProperty m_ExplicitArrayBoundsChecks;
        SerializedProperty m_CompressionType;
        SerializedProperty m_InstallInBuildFolder;

        BuildTarget m_BuildTarget;
        BuildTargetGroup m_BuildTargetGroup;
        NamedBuildTarget m_NamedBuildTarget;
        protected bool m_IsClassicProfile = false;
        protected SharedPlatformSettings m_SharedSettings;
        BuildProfile m_BuildProfile;

        // The properties can be unresponsive on multiple clicks due to the
        // complex layout calculations which can slow down GUI rendering.
        // So we set the GUI label elements' width to make it more responsive.
        protected virtual float labelWidth => 230;

        public abstract BuildProfilePlatformSettingsBase CreateBuildProfilePlatformSettings();

        public virtual void CopyPlatformSettingsToBuildProfile(BuildProfilePlatformSettingsBase platformSettingsBase)
        {
        }

        public virtual void CopyPlatformSettingsFromBuildProfile(BuildProfilePlatformSettingsBase platformSettings)
        {
        }

        public virtual PreconfiguredSettingsVariant[] GetPreconfiguredSettingsVariants()
        {
            return Array.Empty<PreconfiguredSettingsVariant>();
        }

        public virtual void OnBuildProfileCreated(BuildProfile buildProfile, int preconfiguredSettingsVariant)
        {
        }

        public virtual void OnDisable()
        {
        }

        public virtual bool ShouldDrawDevelopmentPlayerCheckbox() => true;

        public virtual bool ShouldDrawLinkTimeOptimization()
        {
            return BuildTargetDiscovery.TryGetBuildTarget(m_BuildTarget, out var buildTarget) && buildTarget.BuildPlatformProperties?.SupportLinkTimeOptimization == true;
        }
        public virtual bool ShouldDrawProfilerCheckbox() => true;
        public virtual bool ShouldDrawScriptDebuggingCheckbox() => true;
        // Enables a dialog "Wait For Managed debugger", which halts program execution until managed debugger is connected
        public virtual bool ShouldDrawWaitForManagedDebugger() => false;
        public virtual bool ShouldDrawManagedDebuggerFixedPort() => false;
        public virtual bool ShouldDrawExplicitNullCheckbox() => false;
        public virtual bool ShouldDrawExplicitDivideByZeroCheckbox() => false;
        public virtual bool ShouldDrawExplicitArrayBoundsCheckbox() => false;
        public virtual bool ShouldDrawInstallInBuildFolderCheckbox()
        {
            return Unsupported.IsSourceBuild() && PostprocessBuildPlayer.SupportsInstallInBuildFolder(m_BuildTarget);
        }

        protected virtual string GetPlatformProfileInfoMessage()
        {
            return null;
        }

        public string GetProfileInfoMessage()
        {
            return GetPlatformProfileInfoMessage();
        }

        public VisualElement CreateSettingsGUI(
            SerializedObject serializedObject, SerializedProperty rootProperty, BuildProfileWorkflowState workflowState)
        {
            var commonSettingsGUI = CreateCommonSettingsGUI(serializedObject, rootProperty, workflowState);
            var platformSettingsGUI = CreatePlatformSettingsGUI(serializedObject, rootProperty, workflowState);

            var settingsGUI = new VisualElement();
            settingsGUI.Add(platformSettingsGUI);
            if (BuildPlayerWindow.WillDrawMultiplayerBuildOptions())
                settingsGUI.Add(CreateMultiplayerSettingsGUI(serializedObject.targetObject as BuildProfile));
            settingsGUI.Add(commonSettingsGUI);
            return settingsGUI;
        }

        public virtual VisualElement CreatePlatformSettingsGUI(
            SerializedObject serializedObject, SerializedProperty rootProperty, BuildProfileWorkflowState workflowState)
        {
            // Default implementation will render all platform settings defined in
            // BuildProfilePlatformSettingsBase as a PropertyField. Enumerators are
            // shown as-is.
            var field = new PropertyField(rootProperty);
            field.BindProperty(rootProperty);
            return field;
        }

        public virtual VisualElement CreatePlatformBuildWarningsGUI(SerializedObject serializedObject, SerializedProperty rootProperty)
        {
            return null;
        }

        public SerializedProperty FindPlatformSettingsPropertyAssert(SerializedProperty rootProperty, string name)
        {
            SerializedProperty property = rootProperty.FindPropertyRelative(name);
            Debug.Assert(property != null);
            return property;
        }

        public SerializedProperty FindSerializedObjectPropertyAssert(SerializedObject serializedObject, string name)
        {
            SerializedProperty property = serializedObject.FindProperty(name);
            Debug.Assert(property != null);
            return property;
        }

        public VisualElement CreateCommonSettingsGUI(SerializedObject serializedObject, SerializedProperty rootProperty, BuildProfileWorkflowState workflowState)
        {
            m_Development = FindPlatformSettingsPropertyAssert(rootProperty, "m_Development");
            m_ConnectProfiler = FindPlatformSettingsPropertyAssert(rootProperty, "m_ConnectProfiler");
            m_BuildWithDeepProfilingSupport = FindPlatformSettingsPropertyAssert(rootProperty, "m_BuildWithDeepProfilingSupport");
            m_AllowDebugging = FindPlatformSettingsPropertyAssert(rootProperty, "m_AllowDebugging");
            m_WaitForManagedDebugger = FindPlatformSettingsPropertyAssert(rootProperty, "m_WaitForManagedDebugger");
            m_ManagedDebuggerFixedPort = FindPlatformSettingsPropertyAssert(rootProperty, "m_ManagedDebuggerFixedPort");
            m_ExplicitNullChecks = FindPlatformSettingsPropertyAssert(rootProperty, "m_ExplicitNullChecks");
            m_ExplicitDivideByZeroChecks = FindPlatformSettingsPropertyAssert(rootProperty, "m_ExplicitDivideByZeroChecks");
            m_ExplicitArrayBoundsChecks = FindPlatformSettingsPropertyAssert(rootProperty, "m_ExplicitArrayBoundsChecks");
            m_CompressionType = FindPlatformSettingsPropertyAssert(rootProperty, "m_CompressionType");
            m_InstallInBuildFolder = FindPlatformSettingsPropertyAssert(rootProperty, "m_InstallInBuildFolder");

            m_BuildProfile = serializedObject.targetObject as BuildProfile;
            Debug.Assert(m_BuildProfile != null, "Build profile cannot be null");
            m_BuildTarget = m_BuildProfile.buildTarget;
            var subtarget = m_BuildProfile.subtarget;
            m_BuildTargetGroup = BuildPipeline.GetBuildTargetGroup(m_BuildTarget);
            m_NamedBuildTarget = (subtarget == StandaloneBuildSubtarget.Server) ? NamedBuildTarget.Server : NamedBuildTarget.FromBuildTargetGroup(m_BuildTargetGroup);
            m_IsClassicProfile = BuildProfileContext.IsClassicPlatformProfile(m_BuildProfile);
            m_SharedSettings = BuildProfileContext.instance.sharedProfile.platformBuildProfile as SharedPlatformSettings;

            return new IMGUIContainer(
                () =>
                {
                    if (serializedObject == null || !serializedObject.isValid)
                    {
                        return;
                    }

                    var oldLabelWidth = EditorGUIUtility.labelWidth;
                    EditorGUIUtility.labelWidth = labelWidth;
                    serializedObject.UpdateIfRequiredOrScript();
                    ShowCommonBuildOptions(workflowState);
                    serializedObject.ApplyModifiedProperties();
                    EditorGUIUtility.labelWidth = oldLabelWidth;
                });
        }

        VisualElement CreateMultiplayerSettingsGUI(BuildProfile profile)
        {
            return new IMGUIContainer(
                () =>
                {
                    var oldLabelWidth = EditorGUIUtility.labelWidth;
                    EditorGUIUtility.labelWidth = labelWidth;
                    BuildPlayerWindow.DrawMultiplayerBuildOption(profile);
                    EditorGUIUtility.labelWidth = oldLabelWidth;
                });
        }

        public void ShowCommonBuildOptions(BuildProfileWorkflowState workflowState)
        {
            if (ShouldDrawDevelopmentPlayerCheckbox())
            {
                ShowDevelopmentPlayerCheckbox();
            }

            if (ShouldDrawLinkTimeOptimization())
            {
                ShowLinkTimeOptimization();
            }

            using (new EditorGUI.DisabledScope(!m_Development.boolValue))
            {
                if (ShouldDrawProfilerCheckbox())
                {
                    ShowProfilerCheckbox();
                }

                if (ShouldDrawScriptDebuggingCheckbox())
                {
                    ShowScriptDebuggingCheckbox();
                }
            }

            if (ShouldDrawExplicitNullCheckbox())
            {
                ShowExplicitNullChecksToggle();
            }

            if (ShouldDrawExplicitDivideByZeroCheckbox())
            {
                ShowDivideByZeroChecksToggle();
            }

            if (ShouldDrawExplicitArrayBoundsCheckbox())
            {
                ShowArrayBoundsChecksToggle();
            }

            var postprocessor = ModuleManager.GetBuildPostProcessor(m_BuildTarget);

            if (postprocessor != null && postprocessor.SupportsLz4Compression())
            {
                using (var vertical = new EditorGUILayout.VerticalScope())
                using (var propertyScope = new EditorGUI.PropertyScope(vertical.rect, GUIContent.none, m_CompressionType))
                {
                    var cmpIdx = Array.IndexOf(k_CompressionTypes, (Compression)m_CompressionType.intValue);
                    if (cmpIdx == BuildProfilePlatformSettingsBase.k_InvalidCompressionIdx)
                        cmpIdx = Array.IndexOf(k_CompressionTypes, postprocessor.GetDefaultCompression());
                    if (cmpIdx == BuildProfilePlatformSettingsBase.k_InvalidCompressionIdx)
                        cmpIdx = (int)CompressionType.Lz4;  // Lz4 by default.
                    cmpIdx = EditorGUILayout.Popup(k_CompressionMethod, cmpIdx, k_CompressionStrings);
                    m_CompressionType.intValue = (int)k_CompressionTypes[cmpIdx];

                    if (m_BuildTargetGroup == BuildTargetGroup.Standalone && m_IsClassicProfile)
                    {
                        m_SharedSettings.compressionType = (Compression)m_CompressionType.intValue;
                    }
                }
            }

            if (ShouldDrawInstallInBuildFolderCheckbox())
            {
                using (new EditorGUILayout.HorizontalScope())
                {
                    EditorGUILayout.PropertyField(m_InstallInBuildFolder, k_InstallInBuildFolder, GUILayout.ExpandWidth(false));
                    EditorGUILayout.BeginVertical();
                    GUILayout.Space(3);
                    if (GUILayout.Button(k_InstallInBuildFolderHelp, EditorStyles.iconButton))
                    {
                        const string k_ViewScriptPath = "Documentation/InternalDocs/view";
                        const string k_WindowsEditorScriptExtension = ".cmd";
                        const string k_MacAndLinuxEditorScriptExtension = ".sh";

                        var path = Path.Combine(Unsupported.GetBaseUnityDeveloperFolder(), k_ViewScriptPath);
                        if (Application.platform == RuntimePlatform.WindowsEditor)
                            System.Diagnostics.Process.Start(path + k_WindowsEditorScriptExtension);
                        else
                            System.Diagnostics.Process.Start(path + k_MacAndLinuxEditorScriptExtension);
                    }

                    EditorGUILayout.EndVertical();
                }
            }
            else
                m_InstallInBuildFolder.boolValue = false;

            if (m_IsClassicProfile)
            {
                m_SharedSettings.installInBuildFolder = m_InstallInBuildFolder.boolValue;
            }

            ActionState buildAndRunState = (m_InstallInBuildFolder != null && m_InstallInBuildFolder.boolValue) ? ActionState.Disabled : workflowState.buildAndRunAction;
            workflowState.UpdateBuildActionStates(workflowState.buildAction, buildAndRunState);

            if (Unsupported.IsSourceBuild())
                ShowInternalPlatformBuildOptions();
        }

        /// <summary>
        /// Show the Development checkbox. Platforms can override this method to hide/customize the UI.
        /// </summary>
        public virtual void ShowDevelopmentPlayerCheckbox()
        {
            EditorGUI.BeginChangeCheck();
            EditorGUILayout.PropertyField(m_Development, k_DevelopmentBuild);
            if (EditorGUI.EndChangeCheck() && m_IsClassicProfile)
            {
                m_SharedSettings.development = m_Development.boolValue;
            }
        }

        /// <summary>
        /// Show Link Time Optimization. Platforms can override this method to show/customize the UI.
        /// </summary>
        public virtual void ShowLinkTimeOptimization()
        {
        }

        /// <summary>
        /// Show profiler checkboxes, including Autoconnect Profiler and Deep Profiling Support checkboxes.
        /// Platforms can override this method to hide/customize the UI.
        /// </summary>
        public virtual void ShowProfilerCheckbox()
        {
            var autoConnectLabel = m_Development.boolValue ? k_AutoconnectProfiler : k_AutoconnectProfilerDisabled;
            EditorGUI.BeginChangeCheck();
            EditorGUILayout.PropertyField(m_ConnectProfiler, autoConnectLabel);
            if (EditorGUI.EndChangeCheck() && m_IsClassicProfile)
            {
                m_SharedSettings.connectProfiler = m_ConnectProfiler.boolValue;
            }

            var buildWithDeepProfilerLabel = m_Development.boolValue ? k_BuildWithDeepProfiler : k_BuildWithDeepProfilerDisabled;
            EditorGUI.BeginChangeCheck();
            EditorGUILayout.PropertyField(m_BuildWithDeepProfilingSupport, buildWithDeepProfilerLabel);
            if (EditorGUI.EndChangeCheck() && m_IsClassicProfile)
            {
                m_SharedSettings.buildWithDeepProfilingSupport = m_BuildWithDeepProfilingSupport.boolValue;
            }
        }

        /// <summary>
        /// Show script debugging options. Platforms can override this method to hide/customize the UI.
        /// </summary>
        public virtual void ShowScriptDebuggingCheckbox()
        {
            ShowManagedDebuggerCheckboxes();
            var currentPlayerSettings = BuildProfileModuleUtil.GetBuildProfileOrGlobalPlayerSettings(m_BuildProfile);
            if (m_AllowDebugging.boolValue && PlayerSettings.GetScriptingBackend_Internal(currentPlayerSettings, m_NamedBuildTarget.TargetName) == ScriptingImplementation.IL2CPP)
            {
                var apiCompatibilityLevel = PlayerSettings.GetApiCompatibilityLevel(m_NamedBuildTarget);
                bool isDebuggerUsable = apiCompatibilityLevel == ApiCompatibilityLevel.NET_4_6 || apiCompatibilityLevel == ApiCompatibilityLevel.NET_Standard_2_0 ||
                    apiCompatibilityLevel == ApiCompatibilityLevel.NET_Unity_4_8 || apiCompatibilityLevel == ApiCompatibilityLevel.NET_Standard;

                if (!isDebuggerUsable)
                    EditorGUILayout.HelpBox("Script debugging is only supported with IL2CPP on .NET 4.x and .NET Standard 2.0 API Compatibility Levels.", MessageType.Warning);
            }
        }

        /// <summary>
        /// Show managed debugger options. Platforms can override this method to
        /// hide/customize the UI.
        /// </summary>
        public virtual void ShowManagedDebuggerCheckboxes()
        {
            EditorGUI.BeginChangeCheck();
            EditorGUILayout.PropertyField(m_AllowDebugging, k_AllowDebugging);
            if (EditorGUI.EndChangeCheck() && m_IsClassicProfile)
            {
                m_SharedSettings.allowDebugging = m_AllowDebugging.boolValue;
            }

            // Not all platforms have native dialog implemented in Runtime\Misc\GiveDebuggerChanceToAttachIfRequired.cpp
            // Display this option only for developer builds
            if (ShouldDrawWaitForManagedDebugger())
            {
                ShowWaitForManagedDebuggerCheckbox();
            }

            if (ShouldDrawManagedDebuggerFixedPort())
            {
                ShowManagedDebuggerFixedPort();
            }
        }

        public void ShowWaitForManagedDebuggerCheckbox()
        {
            if (m_AllowDebugging.boolValue)
            {
                EditorGUI.BeginChangeCheck();
                EditorGUILayout.PropertyField(m_WaitForManagedDebugger, k_WaitForManagedDebugger);
                if (EditorGUI.EndChangeCheck() && m_IsClassicProfile)
                {
                    m_SharedSettings.waitForManagedDebugger = m_WaitForManagedDebugger.boolValue;
                }
            }
        }

        public void ShowManagedDebuggerFixedPort()
        {
            if (m_AllowDebugging.boolValue)
            {
                EditorGUI.BeginChangeCheck();
                EditorGUILayout.PropertyField(m_ManagedDebuggerFixedPort, k_ManagedDebuggerFixedPort);
                if (m_ManagedDebuggerFixedPort.intValue < 0 || m_ManagedDebuggerFixedPort.intValue > BuildProfilePlatformSettingsBase.k_MaxPortNumber)
                {
                    m_ManagedDebuggerFixedPort.intValue = 0;
                }

                if (EditorGUI.EndChangeCheck() && m_IsClassicProfile)
                {
                    m_SharedSettings.managedDebuggerFixedPort = m_ManagedDebuggerFixedPort.intValue;
                }
            }
        }

        public void ShowExplicitNullChecksToggle()
        {
            using (new EditorGUI.DisabledScope(m_Development.boolValue))
            {
                EditorGUILayout.PropertyField(m_ExplicitNullChecks, k_ExplicitNullChecks);
            }

            // Force 'ExplicitNullChecks' to true if it's a development build.
            if (m_Development.boolValue)
                m_ExplicitNullChecks.boolValue = true;

            if (m_IsClassicProfile)
            {
                m_SharedSettings.explicitNullChecks = m_ExplicitNullChecks.boolValue;
            }
        }

        public void ShowDivideByZeroChecksToggle()
        {
            using (new EditorGUI.DisabledScope(m_Development.boolValue))
            {
                EditorGUILayout.PropertyField(m_ExplicitDivideByZeroChecks, k_ExplicitDivideByZeroChecks);
            }

            // Force 'explicitDivideByZeroChecks' to true if it's a development build.
            if (m_Development.boolValue)
                m_ExplicitDivideByZeroChecks.boolValue = true;

            if (m_IsClassicProfile)
            {
                m_SharedSettings.explicitDivideByZeroChecks = m_ExplicitDivideByZeroChecks.boolValue;
            }
        }

        public void ShowArrayBoundsChecksToggle()
        {
            using (new EditorGUI.DisabledScope(m_Development.boolValue))
            {
                EditorGUILayout.PropertyField(m_ExplicitArrayBoundsChecks, k_ExplicitArrayBoundsChecks);
            }

            // Force 'explicitArrayBoundsChecks' to true if it's a development build.
            if (m_Development.boolValue)
                m_ExplicitArrayBoundsChecks.boolValue = true;

            if (m_IsClassicProfile)
            {
                m_SharedSettings.explicitArrayBoundsChecks = m_ExplicitArrayBoundsChecks.boolValue;
            }
        }

        /// <summary>
        /// Show internal platform build options for source-built editor.
        /// </summary>
        public virtual void ShowInternalPlatformBuildOptions()
        {
        }

        /// Helper method for rendering an IMGUI popup over an enum
        /// serialized property.
        /// </summary>
        protected void ShowIMGUIPopupOption<T>
        (
            GUIContent label,
            (T SettingValue, GUIContent GUIString)[] options,
            SerializedProperty currentSetting
        ) where T : Enum
        {
            using var vertical = new EditorGUILayout.VerticalScope();
            using var prop = new EditorGUI.PropertyScope(vertical.rect, GUIContent.none, currentSetting);

            // Find the index of the currently set value relative to the GUI layout
            var selectedIndex = Array.FindIndex(options,
                match => match.SettingValue.Equals((T)(object)currentSetting.intValue));
            selectedIndex = selectedIndex < 0 ? 0 : selectedIndex;

            var GUIStrings = new GUIContent[options.Length];
            for (var i = 0; i < options.Length; i++)
                GUIStrings[i] = options[i].GUIString;

            EditorGUI.BeginChangeCheck();
            var newIndex = EditorGUILayout.Popup(label, selectedIndex, GUIStrings);
            if (EditorGUI.EndChangeCheck())
                currentSetting.intValue = (int)(object)options[newIndex].SettingValue;
        }

        /// <summary>
        /// Helper method for rendering an IMGUI popup over an enum
        /// serialized property.
        /// </summary>
        protected bool ShowIMGUIPopupOption<T>
        (
            GUIContent label,
            T[] options,
            GUIContent[] optionString,
            SerializedProperty currentSetting
        ) where T : Enum
        {
            using var vertical = new EditorGUILayout.VerticalScope();
            using var prop = new EditorGUI.PropertyScope(vertical.rect, GUIContent.none, currentSetting);

            // Find the index of the currently set value relative to the GUI layout
            var selectedIndex = Array.FindIndex(options,
                match => match.Equals((T)(object)currentSetting.intValue));
            selectedIndex = selectedIndex < 0 ? 0 : selectedIndex;

            EditorGUI.BeginChangeCheck();
            var newIndex = EditorGUILayout.Popup(label, selectedIndex, optionString);
            if (EditorGUI.EndChangeCheck())
            {
                currentSetting.intValue = (int)(object)options[newIndex];
                return true;
            }

            return false;
        }

        /// <summary>
        /// Helper method for rendering an IMGUI popup over an enum.
        /// </summary>
        protected bool ShowIMGUIPopupOption<T>
        (
            GUIContent label,
            T[] options,
            GUIContent[] optionString,
            T currentSetting,
            out T returnSettings
        ) where T : Enum
        {
            using var vertical = new EditorGUILayout.VerticalScope();
            returnSettings = currentSetting;

            // Find the index of the currently set value relative to the GUI layout
            var selectedIndex = Array.FindIndex(options,
                match => match.Equals(currentSetting));
            selectedIndex = selectedIndex < 0 ? 0 : selectedIndex;

            EditorGUI.BeginChangeCheck();
            var newIndex = EditorGUILayout.Popup(label, selectedIndex, optionString);
            if (EditorGUI.EndChangeCheck())
            {
                returnSettings = options[newIndex];
                return true;
            }

            return false;
        }

        /// <summary>
        /// Helper method for rendering an IMGUI popup for GUIContent
        /// option values
        /// </summary>
        protected int ShowIMGUIPopupOptionForGUIContents
        (
            GUIContent label,
            GUIContent[] optionValues,
            GUIContent[] displayNames,
            SerializedProperty property
        )
        {
            EditorGUI.BeginChangeCheck();
            using var verticalScope = new EditorGUILayout.VerticalScope();
            using var propertyScope = new EditorGUI.PropertyScope(verticalScope.rect, GUIContent.none, property);
            int selectedIndex = Math.Max(0, Array.FindIndex(optionValues, item => item.text == property.stringValue));
            selectedIndex = EditorGUILayout.Popup(label, selectedIndex, displayNames);
            if (EditorGUI.EndChangeCheck())
            {
                property.stringValue = optionValues[selectedIndex].text;
            }
            return selectedIndex;
        }
    }
}
