//
// Copyright (c) Vatsal Manot
//

import Combine
import Swift
import SwiftUI

public func withInlineState<Value, Content: View>(
    initialValue: Value,
    @ViewBuilder content: @escaping (Binding<Value>) -> Content
) -> some View {
    WithInlineState(initialValue: initialValue, content: content)
}

public func withInlineObservedObject<Object: ObservableObject, Content: View>(
    _ object: Object,
    @ViewBuilder content: (Object) -> Content
) -> some View {
    WithInlineObservedObject(object: object, content: content(object))
}

public func withInlineObservedObject<Object: ObservableObject, Content: View>(
    _ object: Object?,
    @ViewBuilder content: (Object?) -> Content
) -> some View {
    WithOptionalInlineObservedObject(object: object, content: content(object))
}

@available(iOS 14.0, macOS 11.0, tvOS 14.0, watchOS 7.0,  *)
public func withInlineStateObject<Object: ObservableObject, Content: View>(
    _ object: @autoclosure @escaping () -> Object,
    @ViewBuilder content: @escaping (Object) -> Content
) -> some View {
    WithInlineStateObject(object(), content: { content($0.wrappedValue) })
}

@available(iOS 14.0, macOS 11.0, tvOS 14.0, watchOS 7.0,  *)
public func withInlineStateObject<Object: ObservableObject, Content: View>(
    _ object: @autoclosure @escaping () -> Object,
    @ViewBuilder content: @escaping (ObservedObject<Object>.Wrapper) -> Content
) -> some View {
    WithInlineStateObject(object(), content: { content($0.projectedValue) })
}

private struct WithInlineState<Value, Content: View>: View {
    @State var value: Value
    
    let content: (Binding<Value>) -> Content
    
    init(
        initialValue: Value,
        @ViewBuilder content: @escaping (Binding<Value>) -> Content
    ) {
        self._value = .init(initialValue: initialValue)
        self.content = content
    }
    
    var body: some View {
        content($value)
    }
}

private struct WithInlineObservedObject<Object: ObservableObject, Content: View>: View {
    @ObservedObject var object: Object
    
    let content: Content
    
    var body: some View {
        content
    }
}

private struct WithOptionalInlineObservedObject<Object: ObservableObject, Content: View>: View {
    @ObservedObject.Optional var object: Object?
    
    let content: Content
    
    var body: some View {
        content
    }
}

@available(iOS 14.0, macOS 11.0, tvOS 14.0, watchOS 7.0,  *)
private struct WithInlineStateObject<Object: ObservableObject, Content: View>: View {
    @StateObject var object: Object
    
    let content: (StateObject<Object>) -> Content
    
    init(
        _ object: @autoclosure @escaping () -> Object,
        @ViewBuilder content: @escaping (StateObject<Object>) -> Content
    ) {
        self._object = .init(wrappedValue: object())
        self.content = content
    }
    
    var body: some View {
        content(_object)
    }
}
