(function() {
  "use strict";

  CodeMirror.showHint = function(cm, getHints, options) {
    // We want a single cursor position.
    if (cm.somethingSelected()) return;
    if (getHints == null) getHints = cm.getHelper(cm.getCursor(), "hint");
    if (getHints == null) return;

    if (cm.state.completionActive) cm.state.completionActive.close();

    var completion = cm.state.completionActive = new Completion(cm, getHints, options || {});
    CodeMirror.signal(cm, "startCompletion", cm);
    if (completion.options.async)
      getHints(cm, function(hints) { completion.showHints(hints); }, completion.options);
    else
      return completion.showHints(getHints(cm, completion.options));
  };

  function Completion(cm, getHints, options) {
    this.cm = cm;
    this.getHints = getHints;
    this.options = options;
    this.widget = this.onClose = null;
  }

  Completion.prototype = {
    close: function() {
      if (!this.active()) return;

      if (this.widget) this.widget.close();
      if (this.onClose) this.onClose();
      this.cm.state.completionActive = null;
      CodeMirror.signal(this.cm, "endCompletion", this.cm);
    },

    active: function() {
      return this.cm.state.completionActive == this;
    },

    pick: function(data, i) {
      var completion = data.list[i];
      if (completion.hint) completion.hint(this.cm, data, completion);
      else this.cm.replaceRange(getText(completion), data.from, data.to);
      this.close();
    },

    showHints: function(data) {
      if (!data || !data.list.length || !this.active()) return this.close();

      if (!this.options.ghosting && this.options.completeSingle != false && data.list.length == 1)
        this.pick(data, 0);
      else
        this.showWidget(data);
    },

    showWidget: function(data) {
      this.widget = new Widget(this, data);
      CodeMirror.signal(data, "shown");

      var debounce = null, completion = this, finished;
      var closeOn = this.options.closeCharacters || /[\s()\[\]{};:>,]/;
      var startPos = this.cm.getCursor(), startLen = this.cm.getLine(startPos.line).length;

      function done() {
        if (finished) return;
        finished = true;
        completion.close();
        completion.cm.off("cursorActivity", activity);
        CodeMirror.signal(data, "close");
      }
      function isDone() {
        if (finished) return true;
        if (!completion.widget) { done(); return true; }
      }

      function update() {
        if (isDone()) return;
        if (completion.options.async)
          completion.getHints(completion.cm, finishUpdate, completion.options);
        else
          finishUpdate(completion.getHints(completion.cm, completion.options));
      }
      function finishUpdate(data) {
        if (isDone()) return;
        if (!data || !data.list.length) return done();
        completion.widget.close();
        completion.widget = new Widget(completion, data);
      }

      function activity() {
        clearTimeout(debounce);
        var pos = completion.cm.getCursor(), line = completion.cm.getLine(pos.line);
        if (pos.line != startPos.line || line.length - pos.ch != startLen - startPos.ch ||
            pos.ch < startPos.ch || completion.cm.somethingSelected() ||
            (pos.ch && closeOn.test(line.charAt(pos.ch - 1))))
          completion.close();
        else
          debounce = setTimeout(update, 170);
      }
      this.cm.on("cursorActivity", activity);
      this.onClose = done;
    }
  };

  function getText(completion) {
    if (typeof completion == "string") return completion;
    else return completion.text;
  }

  function buildKeyMap(options, handle) {
    var baseMap = {
      Up: function() {handle.moveFocus(-1);},
      Down: function() {handle.moveFocus(1);},
      PageUp: function() {handle.moveFocus(-handle.menuSize());},
      PageDown: function() {handle.moveFocus(handle.menuSize());},
      Home: function() {handle.setFocus(0);},
      End: function() {handle.setFocus(handle.length);},
      Enter: handle.pick,
      Tab: handle.pick,
      Esc: handle.close
    };
    var ourMap = options.customKeys ? {} : baseMap;
    function addBinding(key, val) {
      var bound;
      if (typeof val != "string")
        bound = function(cm) { return val(cm, handle); };
      // This mechanism is deprecated
      else if (baseMap.hasOwnProperty(val))
        bound = baseMap[val];
      else
        bound = val;
      ourMap[key] = bound;
    }
    if (options.customKeys)
      for (var key in options.customKeys) if (options.customKeys.hasOwnProperty(key))
        addBinding(key, options.customKeys[key]);
    if (options.extraKeys)
      for (var key in options.extraKeys) if (options.extraKeys.hasOwnProperty(key))
        addBinding(key, options.extraKeys[key]);
    return ourMap;
  }

  function Widget(completion, data) {
    this.completion = completion;
    this.data = data;
    this.options = completion.options || {};

    var widget = this, cm = completion.cm, options = completion.options;

    var hints = this.hints = document.createElement("ul");
    hints.className = "CodeMirror-hints";
    this.selectedHint = 0;

    var completions = data.list;
    for (var i = 0; i < completions.length; ++i) {
      var elt = hints.appendChild(document.createElement("li")), cur = completions[i];
      var className = "CodeMirror-hint" + (i ? "" : " CodeMirror-hint-active");
      if (cur.className != null) className = cur.className + " " + className;
      elt.className = className;
      if (cur.render) cur.render(elt, data, cur);
      else elt.appendChild(document.createTextNode(cur.displayText || getText(cur)));
      elt.hintId = i;
    }

    var pos = cm.cursorCoords(options.alignWithWord !== false ? data.from : null);
    var left = pos.left, top = pos.bottom, below = true;
    hints.style.left = left + "px";
    hints.style.top = top + "px";
    if (this.options.ghosting) {
      hints.style.display = completions.length > 1 ? "block" : "none";
    }
    // If we're at the edge of the screen, then we want the menu to appear on the left of the cursor.
    var winW = window.innerWidth || Math.max(document.body.offsetWidth, document.documentElement.offsetWidth);
    var winH = window.innerHeight || Math.max(document.body.offsetHeight, document.documentElement.offsetHeight);
    var box = hints.getBoundingClientRect();
    var overlapX = box.right - winW, overlapY = box.bottom - winH;
    if (overlapX > 0) {
      if (box.right - box.left > winW) {
        hints.style.width = (winW - 5) + "px";
        overlapX -= (box.right - box.left) - winW;
      }
      hints.style.left = (left = pos.left - overlapX) + "px";
    }
    if (overlapY > 0) {
      var height = box.bottom - box.top;
      if (box.top - (pos.bottom - pos.top) - height > 0) {
        overlapY = height + (pos.bottom - pos.top);
        below = false;
      } else if (height > winH) {
        hints.style.height = (winH - 5) + "px";
        overlapY -= height - winH;
      }
      hints.style.top = (top = pos.bottom - overlapY) + "px";
    }
    (options.container || document.body).appendChild(hints);

    cm.addKeyMap(this.keyMap = buildKeyMap(options, {
      moveFocus: function(n) { widget.changeActive(widget.selectedHint + n); },
      setFocus: function(n) { widget.changeActive(n); },
      menuSize: function() { return widget.screenAmount(); },
      length: completions.length,
      close: function() { completion.close(); },
      pick: function() { widget.pick(); }
    }));

    if (options.closeOnUnfocus !== false) {
      var closingOnBlur;
      cm.on("blur", this.onBlur = function() { closingOnBlur = setTimeout(function() { completion.close(); }, 100); });
      cm.on("focus", this.onFocus = function() { clearTimeout(closingOnBlur); });
    }

    var startScroll = cm.getScrollInfo();
    cm.on("scroll", this.onScroll = function() {
      var curScroll = cm.getScrollInfo(), editor = cm.getWrapperElement().getBoundingClientRect();
      var newTop = top + startScroll.top - curScroll.top;
      var point = newTop - (window.pageYOffset || (document.documentElement || document.body).scrollTop);
      if (!below) point += hints.offsetHeight;
      if (point <= editor.top || point >= editor.bottom) return completion.close();
      hints.style.top = newTop + "px";
      hints.style.left = (left + startScroll.left - curScroll.left) + "px";
    });

    CodeMirror.on(hints, "dblclick", function(e) {
      var t = widget.getHintElement(hints, e.target || e.srcElement);
      if (t && t.hintId != null) {widget.changeActive(t.hintId); widget.pick();}
    });
    CodeMirror.on(hints, "click", function(e) {
      var t = widget.getHintElement(hints, e.target || e.srcElement);
      if (t && t.hintId != null) widget.changeActive(t.hintId);
    });
    CodeMirror.on(hints, "mousedown", function() {
      setTimeout(function(){cm.focus();}, 20);
    });

    CodeMirror.signal(data, "select", completions[0], hints.firstChild);

    if (this.options.ghosting && this.data.list[0]) {
      this.removeGhost();
      this.ghost = new Ghost(this, this.data, this.data.list[0].displayText, this.pick.bind(this));
    }

    return true;
  }

  Widget.prototype = {
    close: function() {
      if (this.completion.widget != this) return;
      this.completion.widget = null;
      this.hints.parentNode.removeChild(this.hints);
      this.completion.cm.removeKeyMap(this.keyMap);
      this.removeGhost();

      var cm = this.completion.cm;
      if (this.options.closeOnUnfocus !== false) {
        cm.off("blur", this.onBlur);
        cm.off("focus", this.onFocus);
      }
      cm.off("scroll", this.onScroll);
    },

    pick: function() {
      this.completion.pick(this.data, this.selectedHint);
    },

    changeActive: function(i) {
      i = Math.max(0, Math.min(i, this.data.list.length - 1));
      if (this.selectedHint == i) return;
      var node = this.hints.childNodes[this.selectedHint];
      node.className = node.className.replace(" CodeMirror-hint-active", "");
      node = this.hints.childNodes[this.selectedHint = i];
      node.className += " CodeMirror-hint-active";

      if (this.options.ghosting) {
        this.removeGhost();
        this.ghost = new Ghost(this, this.data, this.data.list[i].displayText, this.pick.bind(this));
      }

      if (node.offsetTop < this.hints.scrollTop)
        this.hints.scrollTop = node.offsetTop - 3;
      else if (node.offsetTop + node.offsetHeight > this.hints.scrollTop + this.hints.clientHeight)
        this.hints.scrollTop = node.offsetTop + node.offsetHeight - this.hints.clientHeight + 3;
      CodeMirror.signal(this.data, "select", this.data.list[this.selectedHint], node);
    },

    screenAmount: function() {
      return Math.floor(this.hints.clientHeight / this.hints.firstChild.offsetHeight) || 1;
    },

    removeGhost: function() {
      if (!this.ghost) { return; }
      this.ghost.remove();
      return this;
    },

    getHintElement: function (parent, el) {
      while (el && el !== parent && !this.isHintElement(el)) {
        el = el.parentNode;
      }

      return el === parent
        ? void(0)
        : el
      ;
    },

    isHintElement: function (el) {
      return el.nodeName &&
             el.nodeName.toUpperCase() === 'LI' &&
             el.className.split(/\s/).indexOf('CodeMirror-hint') !== -1
      ;
    }
  };

  function Ghost(widget, data, text, accept) {
    var that = this;

    this.cm         = widget.completion.cm;
    this.data       = data;
    this.widget     = widget;
    this.completion = widget.completion;

    this.cm.addKeyMap(this.keyMap = {
      'Tab':   accept || function () { that.accept(); },
      'Right': accept || function () { that.accept(); }
    });

    if (!text) { return this.remove(); }

    // At the moment, the ghost is going to assume the prefix text is accurate
    var suffix = this.suffix = text.substr(data.word.length);

    if (!suffix.length) { return this.remove(); }

    // Creates the ghost element to be styled.
    var ghostHint = document.createElement('span');
    ghostHint.className = 'CodeMirror-hint-ghost';
    ghostHint.appendChild(document.createTextNode(suffix));

    // Abuse the bookmark feature of CodeMirror to achieve the desired completion
    // effect without modifying source code.
    this._ghost = this.cm.setBookmark(this.data.to, {
      widget:     ghostHint,
      insertLeft: true
    });
  }

  Ghost.prototype = {
    accept: function () {
      if (this.suffix && this.data) {
        this.cm.replaceRange(this.suffix, this.data.to, this.data.to);
      }

      return this.remove();
    },

    remove: function () {
      if (this._ghost) { this._ghost.clear(); }

      this.cm.removeKeyMap(this.keyMap);
      delete this.ghost;
      delete this.suffix;
      delete this.widget.ghost;

      return this;
    }
  };

})();
