"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getUniqueConditionSets = void 0;
const get_conditional_values_list_js_1 = require("./get-conditional-values-list.js");
/**
 * Get the minimal set of conditions that can potentially produce different
 * resolution values for a given imports or exports object from a package
 * manifest.
 *
 * For example:
 *
 * ```json
 * {
 *   ".": [{"import":[{"types":"x.d.ts"},"x.mjs"], "require":"y.js"}]
 *   "./a": {"browser":{"require":"./a.js"}},
 *   "./b": {"browser":"./b.js"},
 *   "./c": {"require":{"browser":"./c.js"}}
 * }
 * ```
 *
 * would return:
 * ```js
 * [
 *   ['import','types'],
 *   ['import'],
 *   ['require'],
 *   ['browser'],
 *   ['browser', 'require'],
 * ]
 * ```
 *
 * With the `['require', 'browser']` condition set omitted, as it is already
 * covered by `['browser', 'require']`.
 *
 * Condition ordering is arbitrary and not guaranteed to be consistent.
 */
const getUniqueConditionSets = (importsExports) => {
    const list = (0, get_conditional_values_list_js_1.getConditionalValuesList)(importsExports);
    let results = [];
    for (const [_, conditions] of list) {
        if (!results.some(arr => arrayIsEquivalent(arr, conditions))) {
            results.push([...conditions]);
        }
    }
    return results;
};
exports.getUniqueConditionSets = getUniqueConditionSets;
const arrayIsEquivalent = (arr, sup) => {
    if (arr.length !== sup.size)
        return false;
    for (const c of arr) {
        if (!sup.has(c))
            return false;
    }
    return true;
};
//# sourceMappingURL=get-unique-condition-sets.js.map