"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.determineArtifactNameAndDirectoryOptions = determineArtifactNameAndDirectoryOptions;
exports.getRelativeCwd = getRelativeCwd;
exports.setCwd = setCwd;
const devkit_exports_1 = require("nx/src/devkit-exports");
const devkit_internals_1 = require("nx/src/devkit-internals");
const path_1 = require("path");
async function determineArtifactNameAndDirectoryOptions(tree, options) {
    const normalizedOptions = getNameAndDirectoryOptions(tree, options);
    validateResolvedProject(normalizedOptions.project, normalizedOptions.directory);
    return normalizedOptions;
}
function getNameAndDirectoryOptions(tree, options) {
    const path = options.path
        ? (0, devkit_exports_1.normalizePath)(options.path.replace(/^\.?\//, ''))
        : undefined;
    const fileExtension = options.fileExtension ?? 'ts';
    let { name: extractedName, directory } = extractNameAndDirectoryFromPath(path);
    const relativeCwd = getRelativeCwd();
    // append the directory to the current working directory if it doesn't start with it
    if (directory !== relativeCwd && !directory.startsWith(`${relativeCwd}/`)) {
        directory = (0, devkit_exports_1.joinPathFragments)(relativeCwd, directory);
    }
    const project = findProjectFromPath(tree, directory);
    const name = options.fileName ??
        (options.suffix ? `${extractedName}.${options.suffix}` : extractedName);
    const filePath = (0, devkit_exports_1.joinPathFragments)(directory, `${name}.${fileExtension}`);
    return {
        artifactName: options.name ?? extractedName,
        directory: directory,
        fileName: name,
        filePath: filePath,
        project: project,
    };
}
function validateResolvedProject(project, normalizedDirectory) {
    if (project) {
        return;
    }
    throw new Error(`The provided directory resolved relative to the current working directory "${normalizedDirectory}" does not exist under any project root. ` +
        `Please make sure to navigate to a location or provide a directory that exists under a project root.`);
}
function findProjectFromPath(tree, path) {
    const projectConfigurations = {};
    const projects = (0, devkit_exports_1.getProjects)(tree);
    for (const [projectName, project] of projects) {
        projectConfigurations[projectName] = project;
    }
    const projectRootMappings = (0, devkit_internals_1.createProjectRootMappingsFromProjectConfigurations)(projectConfigurations);
    return (0, devkit_internals_1.findProjectForPath)(path, projectRootMappings);
}
function getRelativeCwd() {
    return (0, devkit_exports_1.normalizePath)((0, path_1.relative)(devkit_exports_1.workspaceRoot, getCwd()));
}
/**
 * Function for setting cwd during testing
 */
function setCwd(path) {
    process.env.INIT_CWD = (0, path_1.join)(devkit_exports_1.workspaceRoot, path);
}
function getCwd() {
    return process.env.INIT_CWD?.startsWith(devkit_exports_1.workspaceRoot)
        ? process.env.INIT_CWD
        : process.cwd();
}
function extractNameAndDirectoryFromPath(path) {
    // Remove trailing slash
    path = path.replace(/\/$/, '');
    const parsedPath = (0, devkit_exports_1.normalizePath)(path).split('/');
    const name = parsedPath.pop();
    const directory = parsedPath.join('/');
    return { name, directory };
}
