"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolvePackageImport = void 0;
const path_1 = require("path");
const url_1 = require("url");
const walk_up_path_1 = require("walk-up-path");
const errors_js_1 = require("./errors.js");
const file_exists_js_1 = require("./file-exists.js");
const find_star_match_js_1 = require("./find-star-match.js");
const read_pkg_js_1 = require("./read-pkg.js");
const resolve_conditional_value_js_1 = require("./resolve-conditional-value.js");
const resolve_dependency_export_js_1 = require("./resolve-dependency-export.js");
const resolve_export_js_1 = require("./resolve-export.js");
const resolve_import_js_1 = require("./resolve-import.js");
/**
 * Resolve an import like '@package/name/sub/module', where
 * './sub/module' appears in the exports of the local package.
 */
const resolvePackageImport = async (url, parentPath, options) => {
    const { originalParent } = options;
    const parts = url.match(/^(@[^\/]+\/[^\/]+|[^\/]+)(?:\/(.*))?$/);
    // impossible
    /* c8 ignore start */
    if (!parts)
        throw (0, errors_js_1.invalidImportSpecifier)(url);
    /* c8 ignore stop */
    for (const dir of (0, walk_up_path_1.walkUp)((0, path_1.dirname)(parentPath))) {
        const pj = (0, path_1.resolve)(dir, 'package.json');
        const pkg = await (0, read_pkg_js_1.readPkg)(pj);
        if (!pkg)
            continue;
        if (pkg.name && pkg.exports) {
            // can import from this package name if exports is defined
            const [, pkgName, sub] = parts;
            if (pkgName === pkg.name) {
                // ok, see if sub is a valid export then
                const subPath = (0, resolve_export_js_1.resolveExport)(sub, pkg.exports, pj, originalParent, options);
                const resolved = (0, path_1.resolve)(dir, subPath);
                if (await (0, file_exists_js_1.fileExists)(resolved))
                    return (0, url_1.pathToFileURL)(resolved);
                else
                    throw (0, errors_js_1.moduleNotFound)(resolved, originalParent);
            }
        }
        if (url.startsWith('#')) {
            if (!pkg.imports) {
                throw (0, errors_js_1.packageImportNotDefined)(url, pj, originalParent);
            }
            const exact = pkg.imports[url];
            if (exact !== undefined) {
                const res = (0, resolve_conditional_value_js_1.resolveConditionalValue)(exact, options);
                if (!res) {
                    throw (0, errors_js_1.packageImportNotDefined)(url, pj, originalParent);
                }
                // kind of weird behavior, but it's what node does
                if (res.startsWith('#')) {
                    return (0, resolve_dependency_export_js_1.resolveDependencyExports)(null, parentPath, options);
                }
                return (0, resolve_import_js_1.resolveImport)(res, pj, options);
            }
            const sm = (0, find_star_match_js_1.findStarMatch)(url, pkg.imports);
            if (!sm) {
                throw (0, errors_js_1.packageImportNotDefined)(url, pj, originalParent);
            }
            const [key, mid] = sm;
            const match = pkg.imports[key];
            const res = (0, resolve_conditional_value_js_1.resolveConditionalValue)(match, options);
            if (!res) {
                throw (0, errors_js_1.packageImportNotDefined)(url, pj, originalParent);
            }
            if (res.startsWith('#')) {
                return (0, resolve_dependency_export_js_1.resolveDependencyExports)(null, parentPath, options);
            }
            const expand = res.replace(/\*/g, mid);
            // start over with the resolved import
            return (0, resolve_import_js_1.resolveImport)(expand, pj, options);
        }
        break;
    }
    return (0, resolve_dependency_export_js_1.resolveDependencyExports)(url, parentPath, options);
};
exports.resolvePackageImport = resolvePackageImport;
//# sourceMappingURL=resolve-package-import.js.map