"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveImport = void 0;
/**
 * Exported as `'resolve-import/resolve-import'`
 * @module
 */
const promises_1 = require("fs/promises");
const module_1 = __importDefault(require("module"));
const path_1 = require("path");
const url_1 = require("url");
const errors_js_1 = require("./errors.js");
const file_exists_js_1 = require("./file-exists.js");
const is_relative_require_js_1 = require("./is-relative-require.js");
const resolve_dependency_export_js_1 = require("./resolve-dependency-export.js");
const resolve_package_import_js_1 = require("./resolve-package-import.js");
const to_file_url_js_1 = require("./to-file-url.js");
const to_path_js_1 = require("./to-path.js");
// affordance for node 16 <16.17 and 18 <18.9
/* c8 ignore start */
if (typeof module_1.default.isBuiltin !== 'function') {
    module_1.default.isBuiltin = (moduleName) => {
        if (moduleName.startsWith('node:')) {
            moduleName = moduleName.substring('node:'.length);
        }
        return module_1.default.builtinModules.includes(moduleName);
    };
}
/* c8 ignore stop */
// It's pretty common to resolve against, eg, cwd + '/x', since we might not
// know the actual file that it's being loaded from, and want to resolve what
// a dep WOULD be from a given path. This allows us to realpath that directory,
// without requiring that the file exist.
const realpathParentDir = async (path) => {
    path = (0, to_path_js_1.toPath)(path);
    return (0, path_1.resolve)(await (0, promises_1.realpath)((0, path_1.dirname)(path)), (0, path_1.basename)(path));
};
/**
 * Resolve an import URL or string as if it were coming from the
 * module at parentURL.
 *
 * Returns a string for node builtin modules, and a file:// URL
 * object for anything resolved on disk.
 *
 * If the resolution is impossible, then an error will be raised, which
 * closely matches the errors raised by Node when failing for the same
 * reason.
 */
const resolveImport = async (
/** the thing being imported */
url, 
/**
 * the place the import() would be coming from. Required for relative
 * imports.
 */
parentURL = undefined, options = {}) => {
    // already resolved, just check that it exists
    if (typeof url === 'string' && url.startsWith('file://')) {
        url = new URL(url);
    }
    if (typeof url === 'object') {
        if (!(await (0, file_exists_js_1.fileExists)(url))) {
            throw (0, errors_js_1.moduleNotFound)(String(url), String(parentURL));
        }
        const rp = await (0, promises_1.realpath)((0, to_path_js_1.toPath)(url));
        return rp !== (0, url_1.fileURLToPath)(url) ? (0, url_1.pathToFileURL)(rp) : url;
    }
    const pu = parentURL ? (0, to_file_url_js_1.toFileURL)(await realpathParentDir(parentURL)) : undefined;
    if ((0, is_relative_require_js_1.isRelativeRequire)(url)) {
        if (!pu) {
            throw (0, errors_js_1.relativeImportWithoutParentURL)(url, parentURL);
        }
        const u = new URL(url, pu);
        if (!(await (0, file_exists_js_1.fileExists)(u))) {
            throw (0, errors_js_1.moduleNotFound)(url, String(parentURL));
        }
        return (0, url_1.pathToFileURL)(await (0, promises_1.realpath)(new URL(url, pu)));
    }
    if ((0, path_1.isAbsolute)(url)) {
        if (!(await (0, file_exists_js_1.fileExists)(url))) {
            throw (0, errors_js_1.moduleNotFound)(url, String(parentURL));
        }
        return (0, url_1.pathToFileURL)(await (0, promises_1.realpath)(url));
    }
    if (module_1.default.isBuiltin(String(url))) {
        return String(url);
    }
    // ok, we have to resolve it. some kind of bare dep import,
    // either a package name resolving to module or main, or a named export.
    const parentPath = (0, to_path_js_1.toPath)(parentURL || (0, path_1.resolve)(await (0, promises_1.realpath)(process.cwd()), 'x'));
    const opts = {
        ...options,
        originalParent: String(options.originalParent || parentPath),
    };
    if (url) {
        return (0, resolve_package_import_js_1.resolvePackageImport)(url, parentPath, opts);
    }
    else {
        return (0, resolve_dependency_export_js_1.resolveDependencyExports)(url, parentPath, opts);
    }
};
exports.resolveImport = resolveImport;
//# sourceMappingURL=resolve-import.js.map