"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveDependencyExports = void 0;
const path_1 = require("path");
const url_1 = require("url");
const errors_js_1 = require("./errors.js");
const file_exists_js_1 = require("./file-exists.js");
const find_dep_package_js_1 = require("./find-dep-package.js");
const read_pkg_js_1 = require("./read-pkg.js");
const resolve_export_js_1 = require("./resolve-export.js");
/**
 * Resolve a dependency like '@dep/name/sub/module' where
 * '@dep/name' is in node_modules somewhere and exports './sub/module'
 */
const resolveDependencyExports = async (url, parentPath, options) => {
    const { originalParent } = options;
    const parts = url?.match(/^(@[^\/]+\/[^\/]+|[^\/]+)(?:\/(.*))?$/);
    const [, pkgName, sub] = url === null ? [, null, ''] : parts || ['', '', ''];
    const ppath = await (0, find_dep_package_js_1.findDepPackage)(pkgName, parentPath);
    if (!ppath) {
        throw (0, errors_js_1.packageNotFound)(pkgName, originalParent);
    }
    const indexjs = (0, path_1.resolve)(ppath, 'index.js');
    const pj = (0, path_1.resolve)(ppath, 'package.json');
    const pkg = await (0, read_pkg_js_1.readPkg)(pj);
    const subpath = sub ? (0, path_1.resolve)(ppath, sub) : false;
    // if not a package, then the sub can still be a direct path
    // if no sub, then resolves to index.js if available.
    if (!pkg) {
        if (!subpath) {
            // try index.js, otherwise fail
            if (await (0, file_exists_js_1.fileExists)(indexjs))
                return (0, url_1.pathToFileURL)(indexjs);
            else
                throw (0, errors_js_1.packageNotFound)(ppath, originalParent);
        }
        else {
            if (await (0, file_exists_js_1.fileExists)(subpath)) {
                return (0, url_1.pathToFileURL)(subpath);
            }
            else
                throw (0, errors_js_1.moduleNotFound)(subpath, originalParent);
        }
    }
    // ok, have a package, look up the export if present.
    // otherwise, use main, otherwise index.js
    if (pkg.exports) {
        const subPath = (0, resolve_export_js_1.resolveExport)(sub, pkg.exports, pj, originalParent, options);
        const resolved = (0, path_1.resolve)(ppath, subPath);
        if (await (0, file_exists_js_1.fileExists)(resolved))
            return (0, url_1.pathToFileURL)(resolved);
        else
            throw (0, errors_js_1.moduleNotFound)(resolved, originalParent);
    }
    else if (subpath) {
        if (await (0, file_exists_js_1.fileExists)(subpath))
            return (0, url_1.pathToFileURL)(subpath);
        else
            throw (0, errors_js_1.moduleNotFound)(subpath, originalParent);
    }
    else if (pkg.main) {
        // fall back to index.js if main is missing
        const rmain = (0, path_1.resolve)(ppath, pkg.main);
        if (await (0, file_exists_js_1.fileExists)(rmain))
            return (0, url_1.pathToFileURL)(rmain);
        else if (await (0, file_exists_js_1.fileExists)(indexjs))
            return (0, url_1.pathToFileURL)(indexjs);
        else
            throw (0, errors_js_1.packageNotFound)(ppath, originalParent);
    }
    else if (await (0, file_exists_js_1.fileExists)(indexjs)) {
        return (0, url_1.pathToFileURL)(indexjs);
    }
    else {
        throw (0, errors_js_1.packageNotFound)(ppath, originalParent);
    }
};
exports.resolveDependencyExports = resolveDependencyExports;
//# sourceMappingURL=resolve-dependency-export.js.map